function rnd = lognormal_rnd (a, v, r, c)
  #
  # usage:  lognormal_rnd (a, v [, r, c])
  #
  # lognormal_rnd (a, v) returns a matrix of random samples from the
  # lognormal distribution with parameters a and v.  The size of the
  # matrix is the common size of a and v.
  #
  # lognormal_rnd (a, v, r, c) returns an r by c matrix of random
  # samples from the lognormal distribution with parameters a and v.
  # Both a and v must be scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("lognormal_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("lognormal_rnd:  c must be a positive integer");
    endif
    [retval, a, v] = common_size (a, v, zeros (r, c));
    if (retval > 0)
      error (["lognormal_rnd:  ", ...
	  "a and v must be scalar or of size ", ... 
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, a, v] = common_size (a, v);
    if (retval > 0)
      error (["lognormal_rnd:  ", ...
	  "a and v must be of common size or scalar"]); 
    endif
  else
    error ("usage:  lognormal_rnd (a, v [, r, c])");
  endif

  [r, c] = size (a);
  s = r * c;
  a = reshape (a, 1, s);
  v = reshape (v, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(a > 0) | !(a < Inf) | !(v > 0) | !(v < Inf));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((a > 0) & (a < Inf) & (v > 0) & (v < Inf));
  if any (k)
    current_distribution = rand ("dist");
    rand ("normal");
    rnd(k) = a(k) .* exp (sqrt (v(k)) .* rand (1, length (k)));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction