function inv = normal_inv (x, m, v)

  # usage:   normal_inv (x [, m, v])
  #
  # For each element of x, compute the quantile (the inverse of the
  # CDF) at x of the normal distribution with mean m and variance v. 
  # Default values are m = 0, v = 1.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if !((nargin == 1) || (nargin == 3))
    error ("usage:  normal_inv (x [, m, v])");
  endif

  if (nargin == 1)
    m = 0;
    v = 1;
  endif

  [retval, x, m, v] = common_size (x, m, v);
  if (retval > 0)
    error (["normal_inv:  ", ...
	"x, m and v must be of common size or scalars"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  m = reshape (m, 1, s);
  v = reshape (v, 1, s);
  inv = zeros (1, s);

  k = find (isinf (m) | isnan (m) | !(v >= 0) | !(v < Inf));
  if any (k)
    inv(k) = NaN * ones (1, length (k));
  endif
  
  k = find (!isinf (m) & !isnan (m) & (v > 0) & (v < Inf));
  if any (k)
    inv(k) = m(k) + sqrt (v(k)) .* stdnormal_inv (x(k));
  endif

  inv = reshape (inv, r, c);
  
endfunction
