function pdf = normal_pdf (x, m, v)

  # usage:   normal_pdf (x [, m, v])
  #
  # For each element of x, compute the probability density function
  # (PDF) at x of the normal distribution with mean m and variance v.
  # Default values are m = 0, v = 1.
  
  # Written by TT (Teresa.Twaroch@ci.tuwien.ac.at) on Jun 3, 1993
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if !((nargin == 1) || (nargin == 3))
    error ("usage:  normal_pdf (x [, m, v])");
  endif

  if (nargin == 1)
    m = 0;
    v = 1;
  endif

  [retval, x, m, v] = common_size (x, m, v);
  if (retval > 0)
    error (["normal_pdf:  ", ...
	"x, m and v must be of common size or scalars"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  m = reshape (m, 1, s);
  v = reshape (v, 1, s);
  pdf = zeros (1, s);

  k = find (isinf (m) | isnan (m) | !(v >= 0) | !(v < Inf));
  if any (k)
    pdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find (!isinf (m) & !isnan (m) & (v > 0) & (v < Inf));
  if any (k)
    pdf(k) = stdnormal_pdf ((x(k) - m(k)) ./ sqrt (v(k))) ...
	./ sqrt (v(k)); 
  endif

  pdf = reshape (pdf, r, c);
  
endfunction
