function rnd = normal_rnd (m, v, r, c)
  #
  # usage:  normal_rnd (m, v [, r, c])
  #
  # normal_rnd (m, v) returns a matrix of random samples from the
  # normal distribution with parameters m and v.  The size of the
  # matrix is the common size of m and v.
  #
  # normal_rnd (m, v, r, c) returns an r by c matrix of random samples
  # from the normal distribution with parameters m and v.  Both m and
  # v must be scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("normal_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("normal_rnd:  c must be a positive integer");
    endif
    [retval, m, v] = common_size (m, v, zeros (r, c));
    if (retval > 0)
      error (["normal_rnd:  m and v must be scalar or of size ", ...
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, m, v] = common_size (m, v);
    if (retval > 0)
      error ("normal_rnd:  m and v must be of common size or scalar");
    endif
  else
    error ("usage:  normal_rnd (m, v [, r, c])");
  endif

  [r, c] = size (m);
  s = r * c;
  m = reshape (m, 1, s);
  v = reshape (v, 1, s);
  rnd = zeros (1, s);
  
  k = find (isnan (m) | isinf (m) | !(v > 0) | !(v < Inf));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((m > -Inf) & (m < Inf) & (v > 0) & (v < Inf));
  if any (k)
    current_distribution = rand ("dist");
    rand ("normal");
    rnd(k) = m(k) + sqrt (v(k)) .* rand (1, length (k));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction