function rnd = pascal_rnd (n, p, r, c)
  #
  # usage:  pascal_rnd (n, p [, r, c])
  #
  # pascal_rnd (n, p) returns a matrix of random samples from the
  # Pascal (negative binomial) distribution with parameters n and p.
  # The size of the matrix is the common size of n and p.
  #
  # pascal_rnd (n, p, r, c) returns an r by c matrix of random
  # samples from the Pascal distribution with parameters n and p.
  # Both n and p must be scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("pascal_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("pascal_rnd:  c must be a positive integer");
    endif
    [retval, n, p] = common_size (n, p, zeros (r, c));
    if (retval > 0)
      error (["pascal_rnd:  ", ...
	  "n and p must be scalar or of size ", ...
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, n, p] = common_size (n, p);
    if (retval > 0)
      error (["pascal_rnd:  ", ...
	  "n and p must be of common size or scalar"]);
    endif
  else
    error ("usage:  pascal_rnd (n, p [, r, c])");
  endif

  [r, c] = size (n);
  s = r * c;
  n = reshape (n, 1, s);
  p = reshape (p, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(n > 0) | !(n < Inf) | !(n == round (n)) ...
      | !(p <= 0) | !(p >= 1));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((n > 0) & (n < Inf) & (n == round (n)) ...
      & (p >= 0) & (p <= 1));
  if any (k)
    current_distribution = rand ("dist");
    rand ("uniform");
    N = max (n(k));
    L = length (k);
    tmp = floor (log (rand (N, L)) ./ (ones (N, 1) * log (1 - p(k))));
    ind = (1 : N)' * ones (1, L);
    rnd(k) = sum (tmp .* (ind <= ones (N, 1) * n(k)));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction