function inv = poisson_inv (x, l)
  
  # usage:  poisson_inv (x, lambda)
  
  # For each component of x, compute the quantile (the inverse of the
  # CDF) at x of the Poisson distribution with parameter lambda.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/26
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin != 2)
    error ("usage:  poisson_inv (x, lambda)");
  endif

  [retval, x, l] = common_size (x, l);
  if (retval > 0)
    error (["poisson_inv:  ", ...
	"x and lambda must be of common size or scalar"]);
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  l = reshape (l, 1, s);
  inv = zeros (1, s);

  k = find ((x < 0) | (x > 1) | isnan (x) | !(l > 0));
  if any (k)
    inv(k) = NaN * ones (1, length (k));
  endif

  k = find ((x == 1) & (l > 0));
  if any (k)
    inv(k) = Inf * ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < 1) & (l > 0));
  if any (k)
    cdf = exp (-l(k));
    while (1)
      m = find (cdf < x(k));
      if any (m)
	inv(k(m)) = inv(k(m)) + 1;
	cdf(m) = cdf(m) + poisson_pdf (inv(k(m)), l(k(m)));
      else
	break;
      endif
    endwhile
  endif
    
  inv = reshape (inv, r, c);
  
endfunction