function rnd = poisson_rnd (l, r, c)
  #
  # usage:  poisson_rnd (lambda [, r, c])
  #
  # poisson_rnd (lambda) returns a matrix of random samples from the
  # Poisson distribution with parameter lambda.  The size of the
  # matrix is the size of lambda.
  #
  # poisson_rnd (lambda, r, c) returns an r by c matrix of random
  # samples from the Poisson distribution with parameter lambda,
  # which must be a scalar or of size r by c.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 3)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("poisson_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("poisson_rnd:  c must be a positive integer");
    endif
    [retval, l] = common_size (l, zeros (r, c));
    if (retval > 0)
      error (["poisson_rnd:  ", ...
	  "lambda must be scalar or of size ", ... 
	  sprintf ("%d by %d", r, c)]);
    endif
  else
    if !(nargin == 1)
      error ("usage:  poisson_rnd (lambda [, r, c])");
    endif
  endif
  
  [r, c] = size (l);
  s = r * c;
  l = reshape (l, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(l > 0) | !(l < Inf));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((l > 0) & (l < Inf));
  if any (k)
    l = l(k);
    len = length (k);
    num = zeros (1, len);
    sum = - log (1 - rand (1, len)) ./ l;
    current_distribution = rand ("dist");
    rand ("uniform");
    while (1)
      ind = find (sum < 1);
      if any (ind)
	sum(ind) = sum(ind) ...
	    - log (1 - rand (1, length (ind))) ./ l(ind);
	num(ind) = num(ind) + 1;
      else
	break;
      endif
    endwhile
    rnd(k) = num;
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction
