function [q, s] = qqplot (x, dist, ...)
  
  # usage:  [q, s] = qqplot (x [, dist [, params]])
  #
  # Performs a QQ-plot (quantile plot).
  # If F is the CDF of the distribution `dist' with parameters `params'
  # and G its inverse, and x a sample vector of length n, the QQ-plot  
  # graphs ordinate s(i) = i-th largest element of x versus abscissa
  # q(i) = G((i - 0.5)/n).
  # If the sample comes from F except for a transformation of location
  # and scale, the pairs will approximately follow a straight line.
  #
  # The default for `dist' is the standard normal distribution.  The
  # optional argument `params' contains a list of parameters of `dist'.
  # E.g., for a quantile plot of the uniform distribution on [2,4] and
  # x, use `qqplot (x, "uniform", 2, 4)'.
  # 
  # If no output arguments are given, the data are plotted directly.

  # Written by KH (Kurt.Hornik@tuwien.ac.at) on Aug 31, 1994
  # Updated by KH on Mar 11, 1995
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin < 1)
    usage ("qqplot (x [,dist [,params]])");
  endif
  
  if !(is_vector(x))
    error ("qqplot:  x must be a vector.");
  endif

  s = sort (x);
  n = length (x);
  t = ((1 : n)' - .5) / n;
  if (nargin == 1)
    f = "stdnormal_inv";
  else
    f = sprintf ("%s_inv", dist);
  endif;
  if (nargin <= 2)
    q = feval (f, t);
    q_label = f;
  else
    param_string = sprintf ("%g", va_arg ());
    for k = 2 : (nargin - 2);
      param_string = sprintf ("%s, %g", param_string, va_arg ())
    endfor
    q = eval (sprintf ("%s (t, %s);", f, param_string));
    q_label = sprintf ("%s with parameter(s) %s", f, param_string);
  endif
  
  if (nargout == 0)
    xlabel (q_label);
    ylabel ("sample points");
    set nokey;
    plot (q, s);
  endif

endfunction
