function rho = spearman (x, y)
  
# usage:  spearman (x [, y])
#
# Computes Spearman's rank correlation coefficient rho for each of the
# variables specified by the input arguments. 
# For matrices, each row is an observation and each column a variable;
# vectors are always observations and may be row or column vectors.
# spearman (x) is equivalent to spearman (x, x).
#
# For two data vectors x and y, Spearman's rho is the correlation of
# the ranks of x and y.  
# If x and y are drawn from independent distributions, rho has zero
# mean and variance 1 / (n - 1), and is asymptotically normally
# distributed. 

# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/25
# Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/30
# Copyright Dept of Probability Theory and Statistics TU Wien
  
  if ((nargin < 1) || (nargin > 2))
    usage ("spearman (x [, y])");
  endif

  if (rows (x) == 1)
    x = x';
  endif
  n = rows (x);
  
  if (nargin == 1)
    rho = corrcoef (ranks (x));
  else
    if (rows (y) == 1)
      y = y';
    endif
    if (rows (y) <> n)
      error (["spearman:  ", ...
	      "x and y must have the same number of observations"]);
    endif
    rho = corrcoef (ranks (x), ranks (y));
  endif
  
endfunction
