function cdf = t_cdf (x, n)
  
  # usage:   t_cdf (x, n)
  #
  # For each element of x, compute the CDF at x of the Student (t)
  # distribution with n degrees of freedom, i.e., PROB( t(n) <= x ). 

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Jun 10, 1993
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 13, 1994
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 2)
    error ("usage:  t_cdf (x, n)");
  endif

  [retval, x, n] = common_size (x, n);
  if (retval > 0)
    error ("t_inv:  x and n must be of common size or scalar");
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  n = reshape (n, 1, s);
  cdf = zeros (1, s);
  
  k = find (isnan (x) | !(n > 0));
  if any (k)
    cdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x == Inf) & (n > 0));
  if any (k)
    cdf(k) = ones (1, length (k));
  endif

  k = find ((x > -Inf) & (x < Inf) & (n > 0));
  if any (k)
    cdf(k) = betai (n(k) / 2, 1 / 2, 1 ./ (1 + x(k) .^ 2 ./ n(k))) / 2;
    ind = find (x(k) > 0);
    if any (ind)
      cdf(k(ind)) = 1 - cdf(k(ind));
    endif
  endif

  # should we really only allow for positive integer n?
  k = find (n != round (n));
  if any (k)
    fprintf (stderr, ...
	"WARNING:  n should be positive integer\n");
    cdf(k) = NaN * ones (1, length (k));
  endif

  cdf = reshape (cdf, r, c);
  
endfunction
