function pdf = t_pdf (x, n)
  
  # usage:   t_pdf (x, n)
  #
  # For each element of x, compute the probability density function
  # (PDF) at x of the t (Student) distribution with n degrees of
  # freedom. 

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 13, 1994
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 2)
    error ("usage:  t_pdf (x, n)");
  endif

  [retval, x, n] = common_size (x, n);
  if (retval > 0)
    error ("t_pdf:  x and n must be of common size or scalar");
  endif
  
  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  n = reshape (n, 1, s);
  pdf = zeros (1, s);

  k = find (isnan (x) | !(n > 0) | !(n < Inf));
  if any (k)
    pdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find (!isinf (x) & !isnan (x) & (n > 0) & (n < Inf));
  if any (k)
    pdf(k) = exp (- (n(k) + 1) .* log (1 + x(k) .^ 2 ./ n(k)) / 2) ...
      ./ (sqrt (n(k)) .* beta (n(k) / 2, 1 / 2));
  endif

  # should we really only allow for positive integer n?
  k = find (n != round (n));
  if any (k)
    fprintf (stderr, ...
	"WARNING:  n should be positive integer\n");
    pdf(k) = NaN * ones (1, length (k));
  endif

  pdf = reshape (pdf, r, c);

endfunction
