function [pval, t, df] = t_test (x, m, alt)
  
  # usage:  [pval, t, df] = t_test (x, m [, alt])
  #
  # For a sample x from a normal distribution with unknown mean and
  # variance, perform a t-test of the null hypothesis mean(x) == m.
  # Under the null, the test statistic t follows a Student distribution
  # with df = length (x) - 1 degrees of freedom.
  #
  # With the optional argument string alt, the alternative of interest
  # can be selected.  
  # If alt is "<>" or "!=", the null is tested against the two-sided
  # alternative mean(x) != m.
  # If alt is ">", the one-sided alternative mean(x) > m is considered,
  # similarly for "<".  
  # The default is the two-sided case.
  #
  # pval is the p-value of the test.
  #  
  # If no output argument is given, the p-value of the test is
  # displayed.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/08/02
  # Last updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/09/13
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if ((nargin < 2) || (nargin > 3))
    error ("usage:  [pval, t, df] = t_test (x, m [, alt])");
  endif
    
  if !(is_vector (x))
    error("t_test:  x must be a vector.");
  endif
  if !(is_scalar (m))
    error("t_test:  m must be a scalar.");
  endif
  
  n   = length (x);
  df  = n - 1;
  t   = sqrt (n) * (sum (x) / n - m) / std (x);
  cdf = t_cdf (t, df);
  
  if (nargin == 2)
    alt  = "!=";
  endif

  if !isstr (alt)
    error ("t_test:  alt must be a string");
  endif
  if (strcmp (alt, "<>") || strcmp (alt, "!="))
    pval = 2 * min (cdf, 1 - cdf);
  elseif strcmp (alt, ">")
    pval = 1 - cdf;
  elseif strcmp (alt, "<")
    pval = cdf;
  else
    error (sprintf ("t_test:  option %s not recognized", alt));
  endif

  if (nargout == 0)
    printf ("  pval:  %g\n", pval);
  endif
  
endfunction
