function [pval, t, df] = t_test_2 (x, y, alt)
  
  # usage:  [pval, t, df] = t_test_2 (x, y [, alt])
  #
  # For two samples x and y from normal distributions with unknown means
  # and unknown equal variances, perform a two-sample t-test of the null
  # hypothesis of equal means.
  # Under the null, the test statistic t follows a Student distribution
  # with df degrees of freedom.
  #
  # With the optional argument string alt, the alternative of interest
  # can be selected.  
  # If alt is "<>" or "!=", the null is tested against the two-sided
  # alternative mean(x) != mean(y).
  # If alt is ">", the one-sided alternative mean(x) > mean(y) is used,
  # similarly for "<".  
  # The default is the two-sided case.
  #
  # pval is the p-value of the test.
  #  
  # If no output argument is given, the p-value of the test is
  # displayed.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/08/02
  # Last updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/09/13
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if ((nargin < 2) || (nargin > 3))
    error ("usage:  [pval, t, df] = t_test_2 (x, y [, alt])");
  endif
    
  if !(is_vector (x) && is_vector (y))
    error ("t_test_2:  both x and y must be vectors");
  endif

  n_x  = length (x);
  n_y  = length (y);
  df   = n_x + n_y - 2;
  mu_x = sum (x) / n_x;
  mu_y = sum (y) / n_y;
  v    = sumsq (x - mu_x) + sumsq (y - mu_y);
  t    = (mu_x - mu_y) ...
           * sqrt ( (n_x * n_y * df) / (v * (n_x + n_y)) );
  cdf  = t_cdf (t, df);

  if (nargin == 2)
    alt  = "!=";
  endif

  if !isstr (alt)
    error ("t_test_2:  alt must be a string");
  endif
  if (strcmp (alt, "<>") || strcmp (alt, "!="))
    pval = 2 * min (cdf, 1 - cdf);
  elseif strcmp (alt, ">")
    pval = 1 - cdf;
  elseif strcmp (alt, "<")
    pval = cdf;
  else
    error (sprintf ("t_test_2:  option %s not recognized", alt));
  endif

  if (nargout == 0)
    printf ("  pval:  %g\n", pval);
  endif

endfunction
