function [pval, t, df, r] = t_test_corr (X, Y, alt)
  
  # usage:  [pval, t, df, r] = t_test_corr (X [, Y [, alt]])
  #         [pval, t, df, r] = t_test_corr (r, n [, alt]) 
  #
  # Based on two normal samples x and y, test the hypothesis that the
  # underlying correlation coefficient r is zero.
  #
  # If the second argument is a scalar, it is assumed to be the sample
  # size and the first argument is assumed to be the correlation (if
  # this is a matrix, the test is applied pointwise).
  #
  # With the optional argument string alt, the alternative of interest
  # can be selected.  
  # If alt is "<>" or "!=", the null is tested against the two-sided
  # alternative r != 0.
  # If alt is ">", the one-sided alternative r > 0 is considered,
  # similarly for "<".  
  # The default is the two-sided case.
  #
  # pval is the p-value of the test.
  #
  # If no output argument is given, the pval is displayed.

  # Under the null, the test statistic t follows a Student distribution
  # with df = n - 2 degrees of freedom.

  # Written by FL (Friedrich.Leisch@ci.tuwien.ac.at) on Oct 2, 1994
  # Copyright Department of Probability Theory and Statistics TU Wien    

  if ((nargin == 0) || (nargin > 3))
    error ("t_test_corr:  see help text for usage");
    endif
  
  if (nargin == 1)

    df = rows (X) - 2;
    r = corrcoef (X);
    alt = "!=";
  
  elseif (nargin == 2)
  
    if (isstr (Y))
      df = rows (X) - 2;
      r = corrcoef (X);
      alt = Y;
    elseif (is_matrix (Y) || is_vector (Y))
      df = rows(X) - 2;
      r = corrcoef (X, Y);      
      alt = "!=";
    elseif (is_scalar (Y))
      df = Y - 2;
      r = X;
      alt = "!=";
    else
      error("t_test_corr: see help text for usage");
    endif
      
  elseif (nargin == 3)
    
    if (is_matrix (Y) || is_vector (Y))
      df = rows(X) - 2;
      r = corrcoef (X, Y);      
    elseif (is_scalar (Y))
      df = Y - 2;
      r = X;
    endif
    
  endif
     
  t = r .* sqrt (df) ./ sqrt(1 - r.^2);
  cdf  = t_cdf (t, df);

  if !isstr (alt)
    error("t_test_corr:  alt must be a string");
  endif
  if (strcmp (alt, "<>") || strcmp (alt, "!="))
    pval = 2 * min (cdf, 1 - cdf);
  elseif strcmp (alt, ">")
    pval = 1 - cdf;
  elseif strcmp (alt, "<")
    pval = cdf;
  else
    error (sprintf ("t_test_corr:  option %s not recognized", alt));
  endif

  if (nargout == 0)
    printf ("pval:  %g\n", pval);
  endif
  
endfunction
