function cdf = uniform_cdf (x, a, b)
  
  # usage:  uniform_cdf (x [, a, b])
  #
  # Returns the CDF at x of the uniform distribution on [a, b], i.e., 
  # PROB( uniform(a,b) <= x ).  
  # Default values are a = 0, b = 1.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on Aug 2, 1994 
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on Oct 26, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if !(nargin == 1 || nargin == 3)
    error ("usage:  uniform_cdf (x [, a, b])");
  endif

  if (nargin == 1)
    a = 0;
    b = 1;
  endif

  [retval, x, a, b] = common_size (x, a, b);
  if (retval > 0)
    error (["uniform_cdf:  ", ...
	"x, a and b must be of common size or scalar"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  a = reshape (a, 1, s);
  b = reshape (b, 1, s);
  cdf = zeros (1, s);
  
  k = find (isnan (x) | !(a < b));
  if any (k)
    cdf(k) = NaN * ones (1, length (k));
  endif

  k = find ((x >= b) & (a < b));
  if any (k)
    cdf(k) = ones (1, length (k));
  endif
  
  k = find ((x > a) & (x < b));
  if any (k)
    cdf(k) = (x(k) < b(k)) .* (x(k) - a(k)) ./ (b(k) - a(k));
  endif
  
  cdf = reshape (cdf, r, c);
  
endfunction
