function cdf = weibull_cdf (x, l, a)
  
  # usage:  weibull_cdf (x, lambda, alpha)
  #
  # For each element of x, compute the cumulative distribution function
  # (CDF) at x of the Weibull distribution with parameters lambda and
  # alpha which is 1 - exp(- lambda * x^alpha), x >= 0.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/08/13
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/15
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/26
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin != 3)
    error ("usage:  weibull_cdf (x, lambda, alpha)");
  endif

  [retval, x, l, a] = common_size (x, l, a);
  if (retval > 0)
    error (["weibull_cdf:  ", ...
	"x, lambda and alpha must be of common size or scalar"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  l = reshape (l, 1, s);
  a = reshape (a, 1, s);
  cdf = zeros (1, s);
  
  k = find (!(a > 0) | !(l > 0) | isnan (x));
  if any (k)
    cdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x == Inf) & (a > 0) & (l > 0));
  if any (k)
    cdf(k) = ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (a > 0) & (l > 0));
  if any (k)
    cdf(k) = 1 - exp (- l(k) .* (x(k) .^ a(k)));
  endif

  cdf = reshape (cdf, r, c);
  
endfunction
