function pdf = weibull_pdf (x, l, a)
  
  # usage:  weibull_pdf (x, lambda, alpha)
  #
  # For each element of x, compute the probability density function (PDF)
  # at x of the weibull distribution with parameters lambda and alpha,
  # which is given by 
  #    alpha * lambda * x^(alpha-1) * exp(- lambda * x^alpha), x > 0.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/08/13
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/15
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/26
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin != 3)
    error ("usage:  weibull_pdf (x, lambda, alpha)");
  endif

  [retval, x, l, a] = common_size (x, l, a);
  if (retval > 0)
    error (["weibull_pdf:  ", ...
	"x, lambda and alpha must be of common size or scalar"]);
  endif

  [r, c] = size (x);
  s = r * c;
  x = reshape (x, 1, s);
  l = reshape (l, 1, s);
  a = reshape (a, 1, s);
  pdf = zeros (1, s);
  
  k = find (!(a > 0) | !(l > 0) | isnan (x));
  if any (k)
    pdf(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((x > 0) & (x < Inf) & (a > 0) & (l > 0));
  if any (k)
    pdf(k) = a(k) .* l(k) .* (x(k) .^ (a(k) - 1)) ...
	.* exp(- l(k) .* (x(k) .^ a(k)));
  endif

  pdf = reshape (pdf, r, c);
  
endfunction
