function rnd = weibull_rnd (l, a, r, c)
  #
  # usage:  weibull_rnd (l, a [, r, c])
  #
  # weibull_rnd (lambda, alpha) returns a matrix of random samples
  # from the Weibull distribution with parameters lambda and alpha.
  # The size of the matrix is the common size of lambda and alpha.
  #
  # weibull_rnd (lambda, alpha, r, c) returns an r by c matrix of
  # random samples from the Weibull distribution with parameters
  # lambda and alpha.  Both lambda and alpha must be scalar or of 
  # size r by c. 
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/11/01
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin == 4)
    if !(is_scalar (r) && (r > 0) && (r == round (r)))
      error ("weibull_rnd:  r must be a positive integer");
    endif
    if !(is_scalar (c) && (c > 0) && (c == round (c)))
      error ("weibull_rnd:  c must be a positive integer");
    endif
    [retval, l, a] = common_size (l, a, zeros (r, c));
    if (retval > 0)
      error (["weibull_rnd:  ", ...
	  "lambda and alpha must be scalar or of size ", ...
	  sprintf ("%d by %d", r, c)]);
    endif
  elseif (nargin == 2)
    [retval, l, a] = common_size (l, a);
    if (retval > 0)
      error (["weibull_rnd:  ", ...
	  "lambda and alpha must be of common size or scalar"]);
    endif
  else
    error ("usage:  weibull_rnd (lambda, alpha [, r, c])");
  endif

  [r, c] = size (l);
  s = r * c;
  l = reshape (l, 1, s);
  a = reshape (a, 1, s);
  rnd = zeros (1, s);
  
  k = find (!(l > 0) | !(l < Inf) | !(a > 0) | !(a < Inf));
  if any (k)
    rnd(k) = NaN * ones (1, length (k));
  endif
  
  k = find ((l > 0) & (l < Inf) & (a > 0) & (a < Inf));
  if any (k)
    current_distribution = rand ("dist");
    rand ("uniform");
    rnd(k) = (- log (1 - rand (1, length (k))) ./ l(k)) .^ (1 ./ a(k));
    rand (current_distribution);
  endif
  
  rnd = reshape (rnd, r, c);
  
endfunction