function [pval, z] = wilcoxon_test (x, y, alt)
  
# usage:  [pval, z] = wilcoxon_test (x, y [, alt])
#
# For two matched-pair sample vectors x and y, perform a Wilcoxon
# signed-rank test of the null hypothesis PROB(x > y) == 1/2.
# Under the null, the test statistic z approximately follows a
# standard normal distribution. 
#
# With the optional argument string alt, the alternative of interest
# can be selected.  
# If alt is "<>" or "!=", the null is tested against the two-sided
# alternative PROB(x > y) != 1/2.
# If alt is ">", the one-sided alternative PROB(x > y) > 1/2 is
# considered, similarly for "<".
# The default is the two-sided case.
#
# pval is the p-value of the test.
#  
# If no output argument is given, the p-value of the test is
# displayed.
   
# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/21
# Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/01/30
# Copyright Dept of Probability Theory and Statistics TU Wien
  
  if ((nargin < 2) || (nargin > 3))
    usage ("[pval, z] = wilcoxon_test (x, y [, alt])");
  endif
    
  if !( is_vector (x) && is_vector (y) && (length (x) == length (y)))
    error (["wilcoxon_test:  ", ...
	"x and y must be vectors of the same length"]);
  endif

  n   = length (x);
  x   = reshape (x, 1, n);
  y   = reshape (y, 1, n);
  d   = x - y;
  d   = d (find (d <> 0));
  n   = length (d);
  if (n > 0)
    r   = ranks (abs (d));
    z   = sum (r (find (d > 0)));
    z   = (z - n * (n + 1) / 4) ...
	  / sqrt (n * (n + 1) * (2 * n + 1) / 24);
  else
    z   = 0;
  endif

  cdf  = stdnormal_cdf (z);
  
  if (nargin == 2)
    alt  = "!=";
  endif

  if !isstr (alt)
    error("wilcoxon_test:  alt must be a string");
  endif
  if (strcmp (alt, "<>") || strcmp (alt, "!="))
    pval = 2 * min (cdf, 1 - cdf);
  elseif strcmp (alt, ">")
    pval = 1 - cdf;
  elseif strcmp (alt, "<")
    pval = cdf;
  else
    error (sprintf (["wilcoxon_test:  ", ...
	"option %s not recognized"], alt));
  endif

  if (nargout == 0)
    printf ("  pval:  %g\n", pval);
  endif
  
endfunction
