function y = conv2(x,h,area)
#
# usage: y = conv2(x,h)
# 
# Returns the two dimensional convolution of image x with mask h.
# The size of the resultant matrix y = [mx+mh-1,nx+nh-1],
# where [mx,nx] = size(x) and [mh,nh] = size(h).
#
# usage: y = conv2(h,x,area)
#
# Returns a subsection of the 2-D convolution of x with h specified
# by 'area':
#    "full"  - returns the full 2-D convolution (default), 
#    "same"  - returns a matrix of the same size as the image 
#              matrix x from the central part of the convolution. 
#    "valid" - returns the part of the convolution that
#              corresponds to a 'true' convolution, i.e. the
#              part that is computed without the zero-padded
#              edges: size(y) = [mx-mh+1, nx-nh+1] 
#              for size(x) > size(h).
#
#	See also conv, xcorr2, xcorr, deconv, filter2.

#   Copyright by LA de Waal (ladewaal@umberto.csir.co.za) 22 October 1995.


  error(nargchk(2,3,nargin));
  if (nargin < 3) 
    area = "full"; 
  endif


    if (!(strcmp(area, "full") || strcmp(area, "valid") || strcmp(area, "same"))) 
       error("Unknown area parameter, use \"full\", \"same\" or \"valid\"."); 
    endif

    save_do_fortran_indexing = do_fortran_indexing;
    do_fortran_indexing = "true";

    #Swap the arguments if the mask is greater than the image
    # to improve speed
    swapped = prod(size(x)) < prod(size(h));
    if(swapped)   
       t = x; x = h; h = t;
    endif
	
    [mx,nx] = size(x);
    [mh,nh] = size(h);
    y = zeros(mx+mh-1, nx+nh-1);
    rows = 0:mx-1;
    cols = 0:nx-1;
    for j = 1:nh
      for i = 1:mh
        y(i+rows,j+cols) = y(i+rows,j+cols) + h(i,j)*x;
      endfor
    endfor

  # Calculate return area  
    if (strcmp(area, "same"))
      # Return the central part that is the same size as the image matrix. 
      if(swapped)
        rows = floor(mx/2) + (1:mh);
        cols = floor(nx/2) + (1:nh);
      else    
        rows = floor(mh/2) + (1:mx);
        cols = floor(nh/2) + (1:nx);
      endif
      y = y(rows,cols);
    elseif (strcmp(area, "valid")
      # Return the central part that is computed without the padded edges.
      rows = (mh:mx);
      cols = (nh:nx);
      y = y(rows,cols);
    endif

    do_fortran_indexing = save_do_fortran_indexing;

endfunction












