function z = xcorr(x, y, option)
#
# usage : z = xcorr(x, y)
# 
#   Returns the cross-correlation estimate of vector x and vector y.
#
# usage : z = xcorr(x)
#
#   Returns the auto-correlation estimate of vector x.
#   If x is a matrix of size m-by-n the function returns a 2*m-1 by 
#   n^2 matrix whose rows contain the the cross-correlation sequences
#   for all combinations of the columns of x. Added for Matlab 
#   compatibility.
#
# usage : z = xcorr(x,y,option)
#    or : z = xcorr(x,option) 
#
#   Returns the cross-correlation of vector x and vector y, both of 
#   length L,  using the normalisation technique specified by option:
#     "none"     - No normalisation (default).
#     "biased"   - Biased estimate: scales the estimate by 1/L.
#     "unbiased" - Scales the estimate by 1/(L-abs(idx)), where idx
#                  is the index into the result, with zero lag at 
#                  idx = 0;
#     "coeff"    - Scales the estimate to ensure that the correlation
#                  at zero lag equals 1.  For complex values the absolute
#                  value of the correlation at zero lag equals 1.

#   Copyright by LA de Waal (ladewaal@umberto.csir.co.za) 28 October 1995.

#  error(nargchk(1,3,nargin));

  if (nargin == 1)
    option = "none";
    y = x;
  else
    if (nargin == 2)
      if (isstr(y))
        option = y;
        y = x;
      else
        option = "none";
      endif
    endif
    if (is_matrix(x) || is_matrix(y))
      error("Only vectors are supported, use xcorr2 for matrix correlation");
    end
    option = tolower(option);
    if (!strcmp(option, "none") && (length(x) != length(y)))
      error ("Only option \"none\" supports different vector lengths");
    endif
  endif

  if(!(strcmp(option, "none") || strcmp(option, "biased") || \ 
       strcmp(option, "unbiased") || strcmp(option, "coeff")))
    error("Unknown option, use \"normal\", \"biased\", \"unbiased\" or \"coeff\"");
  endif

  save_do_fortran_indexing = do_fortran_indexing;
  do_fortran_indexing = "true";

  [mx,nx] = size(x);
  if (strcmp(option, "none") && (is_matrix(x)))
    z = zeros(2*mx - 1, nx^2);
    [mz, nz] = size(z);

    # This piece of code is to vectorise the data, thus avoiding 
    # a double loop - for performance reasons.
    t = zeros(mz, nx);
    t(1:mx,:) = x;
    t = reshape(t, 1, mz*nx);
    t = t(1:mz*nx - (mz-mx));
 
   for i=1:nx
      z(:,(i-1)*nx+1:i*nx) = reshape(conj(conv(conj(t), ... 
                             rot90(x(:,i), 2 ))), mz, nx); 
    endfor
  else      
    z = conj(conv(conj(y),rot90(x,2)));
    l = length(x);
    if (strcmp(option, "biased"))
      z = z ./ l;
    elseif (strcmp(option, "unbiased"))
      z = z ./ (l - abs( [-(l-1):1:(l-1)] ));
    elseif (strcmp(option, "coeff"))
      z = z ./ z(ceil(length(z)/2));
    endif
  endif

  do_fortran_indexing = save_do_fortran_indexing;

endfunction









