function subplot (rows, columns, index)

% usage: subplot (rows,columns,index)
%        or 
%        subplot(rcn)
%
% NOTE: this will work only with gnuplot installed with
%       multiplot patch (or version 3.6 beta)
%
%    Sets gnuplot in multiplot mode and plots in location
%    given by index (there are columns X rows subwindows)
%
%    Input:
%
%    rows   : number of rows in subplot grid
%    columns: number of columns in subplot grid
%    index  : index of subplot where to make the next plot
%
%    If only one arg, then it (crn) has to be three digit 
%    value specifying the location in digit 1 (rows) and 
%    2 (columns) and digit 3 is the plot index
%
%    The plot index runs row-wise,i.e., first all the
%    columns in a row are filled and then the next
%    row is filled
%
%    For example, plot with 4 X 2 grid, will have
%    plot indices running as follows:
%
%      -----------------------------------
%      |        |       |       |        |
%      |    1   |    2  |    3  |    4   |
%      |        |       |       |        |
%      -----------------------------------
%      |        |       |       |        |
%      |    5   |    6  |    7  |    8   |
%      |        |       |       |        |
%      -----------------------------------
%

%
%    Autor: Vinayak Dutt, Dutt.Vinayak@mayo.EDU
%

%
% global variables to keep track of multiplot options
%
  global multiplot_mode 
  global multi_xn multi_yn
  global multi_xsize multi_ysize 

% check calling argument count

  if (nargin != 3 && nargin != 1)
	error "usage: subplot (rows,columns,index) or subplot(rcn)"
  endif

% check for scalar inputs

  if( nargin == 1 )
  	if (!is_scalar(rows))
		error ("subplot: input rcn has to be a scalar");
  	endif
	xnp = rows;
	rows = fix(xnp/100);
	columns = fix((xnp - 100*rows)/10);
	index = xnp - 10*columns - 100*rows;
  else
  	if (!(is_scalar(columns) && is_scalar(rows) && is_scalar(index)))
		error ("subplot: columns, rows, and index  have to be scalars");
  	endif
  endif
  columns = fix(columns);
  rows = fix(rows);
  index = fix(index);

  if (index > columns*rows)
  	error ("subplot: index must be less than columns*rows");
  endif;

  if( columns < 1 || rows < 1 || index < 1)
  	error ("subplot: columns,rows,index must be be positive");
  endif;

  if((columns*rows) == 1)
%
% switching to single plot ?
%
	set nomultiplot;
	set size 1,1
	set origin 0,0
	multi_xn = 1;
	multi_yn = 1;
	multiplot_mode = 0;
  	return;
  endif;
%
% doing multiplot plots
%
   doagain = 0;
   if ( exist ("multiplot_mode") != 1 )
   	doagain = 1;
   else 
   	if ( (multiplot_mode != 1) || (multi_xn != columns) || (multi_yn != rows) )
   		doagain = 1;
	endif;
   endif;
   if ( doagain )
	multiplot_mode = 1;
	multi_xn = columns;
	multi_yn = rows;
   	multi_xsize = 1.0 ./ columns;
   	multi_ysize = 1.0 ./ rows;
   	set multiplot;
   	command = sprintf ("set size %g,%g", multi_xsize,multi_ysize);
   	eval (command);
   endif;
%
% get the sub plot location
%
  yp = fix((index-1)/columns);
  xp = index - yp*columns - 1;
  xp = xp + 1;
  yp = yp + 1;
%
% set the origin
%
  xo = (xp - 1.0)*multi_xsize;
  yo = (rows - yp)*multi_ysize;
  command = sprintf ("set origin %g,%g", xo,yo);
  eval (command);
%
endfunction


