function [a,b,c,d] = tf2ss(num,den)

% Usage: [A,B,C,D] = tf2ss(num,den)
%
% Converts from transfer function to state space form.
% The inputs num and den are the polynomial coefficients
% for a SIMO transfer function matrix.  The number of
% rows in num equals the number of system outputs.
% The transfer function
%
%                -1         num(s)
% G(s) = C (sI-A)  B + D =  ------
%                           den(s)
%
% is converted to state-space form:
%   .
%   x = A x + B u
%   y = C x + D u
%
% The state-space matrices A, B, C, D are returned in 
% controler canonical form.
%
% Note: tf2ss() works for continuous and discrete systems.

% Written by Michael Fikes (mfikes@mnsinc.com) September 1995.

% check args
if nargin != 2
   usage("tf2ss(num,den)");
end

[numr, numc] = size(num);
[denc] = size(den,2);

% Pad num with zeros if smaller than den
if numc < denc
   num=[zeros(numr,denc-numc) num];
end

if numc > denc
   error("tf2ss: numerator of greater order than denominator")
end

ns = denc - 1;
if ns > 0
   num=num/den(1);
   den=den/den(1);
   a = [-den(2:ns+1); eye(ns-1) zeros(ns-1,1)];
   b = [1; zeros(ns-1,1)];
   c = num(:,2:ns+1) -  num(:,1)*den(2:ns+1);
   d = num(:,1);
else
   a = -0.5;
   b = 1;
   c = zeros(size(num,1),1);
   d = num/den;
end
