function y = arch_rnd (a, b, T)
  
  # usage:  y = arch_rnd (a, b, T)  
  #
  # Simulates an ARCH sequence y of length T with AR coefficients b
  # and CH coefficients a.
  # I.e., y follows the model
  #     y(t) = b(1) + b(2) * y(t-1) + ... + b(lb) * y(t-lb+1) + e(t),
  # where e(t), given y up to time t-1, is N(0, h(t)), with
  #     h(t) = a(1) + a(2) * e(t-1)^2 + ... + a(la) * e(t-la+1)^2.

  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/08/11
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 94/10/06
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin <> 3)
    error ("usage:  arch_rnd (a, b, T)");
  endif
  
  if !( (min (size (a)) == 1) && (min (size (b)) == 1) )
    error ("arch_rnd:  a and b must both be scalars or vectors");
  endif
  if !( is_scalar (T) && (T > 0) && (rem (T, 1) == 0) )
    error ("arch_rnd:  T must be a positive integer");
  endif
  
  if !(a(1) > 0)
    error ("arch_rnd:  a(1) must be positive");
  endif
  # perhaps add a test for the roots of a(z) here ...
  
  la = length (a);
  a  = reshape (a, 1, la);
  if (la == 1)
    a  = [a, 0];
    la = la + 1;
  endif
  lb = length (b);
  b  = reshape (b, 1, lb);
  if (lb == 1)
    b  = [b, 0];
    lb = lb + 1;
  endif
  M  = max([la lb]);
  
  e  = zeros (T, 1);
  h  = zeros (T, 1);
  y  = zeros (T, 1);
  
  orig_dist = rand ("dist");
  rand ("normal");
  
  h(1) = a(1);
  e(1) = sqrt (h(1)) * rand;
  y(1) = b(1) + e(1);
  
  for t= 2 : M;
    ta   = min ([t la]);
    h(t) = a(1) + a(2:ta) * e(t-1:t-ta+1).^2;
    e(t) = sqrt (h(t)) * rand;
    tb   = min ([t lb]);
    y(t) = b(1) + b(2:tb) * y(t-1:t-tb+1) + e(t);
  endfor
  if (T > M)
    for t = M+1 : T;
      h(t) = a(1) + a(2:la) * e(t-1:t-la+1).^2;
      e(t) = sqrt (h(t)) * rand;
      y(t) = b(1) + b(2:lb) * y(t-1:t-tb+1) + e(t);
    endfor
  endif
  
  y = y(1:T);
  
  rand (orig_dist);
  
endfunction
    