function [pval, lm] = arch_test (y, X, p)

  # usage:  [pval, lm] = arch_test (y, X, p)
  #         [pval, lm] = arch_test (y, k, p)
  #
  # arch_test (y, X, p) performs a Lagrange Multiplier (LM) test of
  # the null hypothesis of no conditional heteroscedascity in the
  # linear regression model y = X * b + e against the alternative of
  # CH(p).   
  # I.e., the model is
  #     y(t) = b(1) * x(t,1) + ... + b(k) * x(t,k) + e(t),
  # where given y up to t-1 and x up to t, e(t) is N(0, h(t)) with
  #     h(t) = v + a(1) * e(t-1)^2 + ... + a(p) * e(t-p)^2,
  # and the null is a(1) == ... == a(p) == 0.
  #
  # arch_test (y, k, p) does the same in a linear autoregression model
  # of order k, i.e., with [1, y(t-1), ..., y(t-k)] as the t-th row of
  # X. 
  #
  # Under the null, lm approximately has a chisquare distribution with
  # p degrees of freedom.  pval is the p-value (1 minus the CDF of
  # this distribution at lm) of the test.
  #
  # If no output argument is given, the p-value is displayed.
  
  # Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/08/11
  # Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/09/13
  # Copyright Dept of Probability Theory and Statistics TU Wien
  
  if (nargin <> 3)
    error ("usage:  arch_test needs 3 input arguments");
  endif

  if !(is_vector (y))
    error ("arch_test:  y must be a vector");
  endif
  T   = length (y);
  y   = reshape (y, T, 1);
  [rx, cx] = size (X);
  if ((rx == 1) && (cx == 1))
    X = autoreg_matrix (y, X);
  elseif !(rx == T)
    error (["arch_test:  ", ...
	"either rows(X) == length(y), or X is a scalar"]);
  endif
  if !(is_scalar(p) && (rem(p, 1) == 0) && (p > 0))
    error("arch_test:  p must be a positive integer.");
  endif
  
  [b, v_b, e] = ols (y, X);
  Z    = autoreg_matrix (e.^2, p);
  f    = e.^2 / v_b - ones (T, 1);
  f    = Z' * f;
  lm   = f' * inv (Z'*Z) * f / 2;
  pval = 1 - chisquare_cdf (lm, p);
  
endfunction