function [H, r, x, y] = hurst (A)

# Computes the Hurst parameter H of the vector A using the variance-time
# plot method.
# 
# Use the variance computed over intervals of different lenghts.  Plotted on
# a log-log diagram, var (x) versus the interval length (y) should be
# least-squares interpolated by a line of slope beta=2H.  r is the
# correlation coefficient and gives a "reliability factor" of the H
# estimate: values higher than 0.9 should be expected.
#
# I don't know about any data on the precision of this method.  All I know
# is that it cannot provide a confidence interval measure.
# 
# Leland, Taqqu, Willinger, Wilson, "On the Self-Similar Nature of Ethernet
# Traffic (Extended Version)", IEEE/ACM Trans. on Networking Vol.2, Num.1,
# 1994.

# Put in the public domain by Francesco Potorti` <pot@cnuce.cnr.it>
# Mon Aug 28 17:07:26 MET 1995

  if (!is_vector(A))
    error ("A should be a vector");
  endif

  # m is the minimum interval length over which the var is computed.  M is
  # the minimum number of intervals over which that same quantity is
  # computed.  k is the number of interval lengths per decade used for the
  # computation.  minp is the minimum number of points that must be used.
  # maxp is the maximum number of points that must be used.
  m = 10;
  M = 10;
  k = 5;
  minp = 5;
  maxp = 20;

  r = length (A);
  minr = ceil (m*M*10^((minp-1)/k));
  if (r < minr)
    error (sprintf ("A should have at least %d elements\n", minr));
  endif

  # n is the number of points we use to interpolate the line on a log-log
  # plot whose slope is 2H.  x is a vector containing their abscissae, y
  # contains their ordinates, that is, the variance of the increments of the
  # vector A at distances x.
  
  n = min (maxp, floor (k * log10(r/m/M)));
  x = floor (logspace (log10(m), log10(r/M), n))';
  cumA = cumsum (A);
  for i = 1:n
    xi =x(i);
    Y = cumA (xi:xi:r) - [0; cumA (xi:xi:r-xi)];
    sets = length (Y);
    y(i) = (sumsq(Y) - cumA(sets*xi)^2/sets) / (sets-1);
  endfor
  x = log (x);
  y = log (y);
  alfabeta = [ones (size (x)), x] \ y;
  H = alfabeta(2) / 2;
  if (nargout > 1) r = corrcoef (x, y); endif
endfunction



