function X = synthesis (Y, c)
  
  # usage: X = synthesis (Y, c)
  #
  # recovers a signal X from its short-time Fourier transform Y.
  # c = [win_size, increment, window_type].
  #
  # Y and c can be derived by [Y, c] = stft (X [, ...]).
  
  # Written by AW (Andreas.Weingessel@ci.tuwien.ac.at) on Nov 17, 1994
  # Copyright Dept of Probability Theory and Statistics TU Wien

  if (nargin != 2)
    error ("usage: X = synthesis (Y, c)");
  endif
  
  [nr nc] = size (c);
  if (nr * nc != 3)
    error ("synthesis:  c must contain exactly 3 elements");
  endif
  
  # not necessary, enables better reading
  win = c(1);      
  inc = c(2);
  w_type = c(3);
  
  if (w_type == 1)
    H = hanning (win);
  elseif (w_type == 2)
    H = hamming (win);
  elseif (w_type == 3)
    H = ones (win, 1);
  else
    error ("synthesis:  window_type must be 1, 2, or 3");
  endif
    
  Z = real (ifft (Y));
  st = fix ((win-inc) / 2);
  Z = Z(st:st+inc-1, :);
  H = H(st:st+inc-1);

  nc = columns(Z);
  for i = 1:nc
    Z(:, i) = Z(:, i) ./ H;
  endfor
  
  X = reshape(Z, inc * nc, 1);
  
endfunction
