function y = funm (x, fun)
  
# usage:  funm (x, fun)
#
# Computes the matrix function of the square matrix x of the scalar
# function specified by the string fun.  
# For example, funm (x, "exp") computes the matrix exponential of x
# (but expm (x) is better ...).
  
# Written by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1994/10/12
# Updated by KH (Kurt.Hornik@ci.tuwien.ac.at) on 1995/02/07
# Copyright Department of Statistics and Probability Theory TU Wien
  
# Computation is based on Parlett's method, cf. Golub & Van Loan
# (1989), Matrix Computations, page 543 
  
  if (nargin <> 2)
    usage ("funm (x, fun)");
  endif

  [m, n] = size (x);
  if (m != n)
    error ("funm:  x must be square");
  endif
  
  if !(isstr (fun) && (exist (fun) == 2))
    error ("funm:  fun must be a string naming an existing function");
  endif
  
  [q, t] = schur (x + zeros (size (x)) * i);
  f      = diag (eval (sprintf ("%s (diag (t));", fun)));
  tol    = eps * norm (t, 1);
  
  for p = 1 : n - 1
    for i = 1 : n - p
      j = i + p;
      s = t(i, j) * (f(j, j) - f(i, i));
      for k = i + 1 : j - 1
	s = s + t(i, k) * f(k, j) - f(i, k) * t(k, j);
      endfor
      d = (t(j, j) - t(i, i));
      if (abs (d) <= tol)
	fprintf ("stderr", ...
	    "WARNING:  result from funm probably incorrect\n");
	d = tol;
      endif
      f(i, j) = s / d;
    endfor
  endfor
  
  y = q * f * q';
  
endfunction
  