/*
OS/2 - emx version of file system operations.  Based on the Unix (BSD)
version, but takes into account drives in pathnames.
Sentot Kromodimoeljo, October 1993.
*/

/*
(c) Copyright Taiichi Yuasa and Masami Hagiya, 1984.  All rights reserved.
Copying of this file is authorized to users who have executed the true and
proper "License Agreement for Kyoto Common LISP" with SIGLISP.
*/

#include "include.h"
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>


object Kwild;


coerce_to_filename(pathname, p)
object pathname;
char *p;
{
	int i;
	object namestring;

	namestring = coerce_to_namestring(pathname);
	if (namestring->st.st_fillp >= MAXPATHLEN - 16) {
		vs_push(namestring);
		FEerror("Too long filename: ~S.", 1, namestring);
	}
	for (i = 0;  i < namestring->st.st_fillp;  i++)
		p[i] = namestring->st.st_self[i];
	p[i] = '\0';
}

object
truename(pathname)
object pathname;
{
	register char *p, *q;
	char filename[MAXPATHLEN];
	char truefilename[MAXPATHLEN];
	char current_directory[MAXPATHLEN];
	char directory[MAXPATHLEN];
	char *filenamestart;
	char current_drive, drive;

	coerce_to_filename(pathname, filename);
	drive = _fngetdrive(filename);
	current_drive = _getdrive();
	getwd(current_directory);

	if (drive == 0) {
	  drive = current_drive;
	  filenamestart = filename;
	} else {
	  _chdrive (drive);
	  filenamestart = filename + 2;
	}
	  
	for (p = filenamestart, q = 0;  *p != '\0';  p++)
		if (*p == '/')
			q = p;
	if (q == filenamestart) {
		q++;
		p = "/";
	} else if (q == 0) {
		q = filenamestart;
		p = current_directory;
	} else {
		*q++ = '\0';
		if (chdir(filenamestart) < 0)
		    FEerror("Cannot get the truename of ~S.", 1, pathname);
		p = getwd(directory);
	}

	truefilename[0] = drive;
	truefilename[1] = ':';

	if (p[0] == '/' && p[1] == '\0') {
		if (strcmp(q, "..") == 0)
			strcpy(truefilename + 2, "/.");
		else
			sprintf(truefilename + 2, "/%s", q);
	} else if (strcmp(q, ".") == 0)
		strcpy(truefilename + 2, p);
	else if (strcmp(q, "..") == 0) {
		for (q = p + strlen(p);  *--q != '/';) ;
		if (p == q)
			strcpy(truefilename + 2, "/.");
		else {
			*q = '\0';
			strcpy(truefilename + 2, p);
			*q = '/';
		}
	} else
		sprintf(truefilename + 2, "%s/%s", p, q);
	chdir(current_directory);
	vs_push(make_simple_string(truefilename));
	pathname = coerce_to_pathname(vs_head);
	vs_pop;
	return(pathname);
}

bool
file_exists(file)
object file;
{
	char filename[MAXPATHLEN];
	struct stat filestatus;

	coerce_to_filename(file, filename);
	if (stat(filename, &filestatus) >= 0)
		return(TRUE);
	else
		return(FALSE);
}

FILE *
backup_fopen(filename, option)
{
	char backupfilename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];

	strcat(strcpy(backupfilename, filename), ".BAK");
	sprintf(command, "mv %s %s", filename, backupfilename);
	system(command);
	return(fopen(filename, option));
}

int
file_len(fp)
FILE *fp;
{
	struct stat filestatus;

	fstat(fileno(fp), &filestatus);
	return(filestatus.st_size);
}

Ltruename()
{
	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	vs_base[0] = truename(vs_base[0]);
}

Lrename_file()
{
	char filename[MAXPATHLEN];
	char newfilename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];

	check_arg(2);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	check_type_or_Pathname_string_symbol(&vs_base[1]);
	coerce_to_filename(vs_base[0], filename);
	vs_base[0] = coerce_to_pathname(vs_base[0]);
	vs_base[1] = coerce_to_pathname(vs_base[1]);
	vs_base[1] = merge_pathnames(vs_base[1], vs_base[0], Cnil);
	coerce_to_filename(vs_base[1], newfilename);
	sprintf(command, "mv %s %s", filename, newfilename);
	system(command);
	vs_push(vs_base[1]);
	vs_push(truename(vs_base[0]));
	vs_push(truename(vs_base[1]));
	vs_base += 2;
}

Ldelete_file()
{
	char filename[MAXPATHLEN];

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	coerce_to_filename(vs_base[0], filename);
	if (unlink(filename) < 0)
		FEerror("Cannot delete the file ~S.", 1, vs_base[0]);
	vs_base[0] = Ct;
}

Lprobe_file()
{
	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	if (file_exists(vs_base[0]))
		vs_base[0] = truename(vs_base[0]);
	else
		vs_base[0] = Cnil;
}

Lfile_write_date()
{
	char filename[MAXPATHLEN];
	struct stat filestatus;

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	coerce_to_filename(vs_base[0], filename);
	if (stat(filename, &filestatus) < 0)
		FEerror("Cannot get the file status of ~S.", 1, vs_base[0]);
	vs_base[0] = unix_time_to_universal_time(filestatus.st_mtime);
}

Lfile_author()
{
	char filename[MAXPATHLEN];
	struct stat filestatus;
	struct passwd *pwent;
	extern struct passwd *getpwuid();

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	coerce_to_filename(vs_base[0], filename);
	if (stat(filename, &filestatus) < 0)
		FEerror("Cannot get the file status of ~S.", 1, vs_base[0]);
	vs_base[0] = make_simple_string("OS/2");
}

Luser_homedir_pathname()
{
  char *getenv(), *cp;

  if (vs_top - vs_base > 1)
    too_many_arguments();

  if ((cp = getenv ("HOME")) == NULL)
    cp = "/";

  vs_base[0] = make_simple_string(cp);
  vs_top = vs_base+1;
  vs_base[0] = coerce_to_pathname(vs_base[0]);
}


Ldirectory()
{
	char filename[MAXPATHLEN];
	char command[MAXPATHLEN * 2];
	FILE *fp;
	register i, c;
	object *top = vs_top;
	char iobuffer[BUFSIZ];
	extern FILE *popen();

	check_arg(1);

	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	vs_base[0] = coerce_to_pathname(vs_base[0]);
	if (vs_base[0]->pn.pn_name==Cnil && vs_base[0]->pn.pn_type==Cnil) {
		coerce_to_filename(vs_base[0], filename);
		strcat(filename, "*");
	} else if (vs_base[0]->pn.pn_name==Cnil) {
		vs_base[0]->pn.pn_name = Kwild;
		coerce_to_filename(vs_base[0], filename);
		vs_base[0]->pn.pn_name = Cnil;
	} else if (vs_base[0]->pn.pn_type==Cnil) {
		coerce_to_filename(vs_base[0], filename);
		strcat(filename, "*");
	} else
		coerce_to_filename(vs_base[0], filename);
	sprintf(command, "ls -d %s 2> /dev/nul", filename);
	fp = popen(command, "r");
	fp->buffer = NULL;
	setbuf(fp, iobuffer);
	for (;;) {
		for (i = 0;  c = getc(fp);  i++)
			if (c <= 0)
				goto L;
			else if (c == '\n')
				break;
			else
				filename[i] = c;
		filename[i] = '\0';
		vs_push(make_simple_string(filename));
		vs_head = truename(vs_head);
	}
L:
	fp->buffer = NULL;
	pclose(fp);
	vs_push(Cnil);
	while (vs_top > top + 1)
		stack_cons();
	vs_base = top;
}


siLchdir()
{
	char filename[MAXPATHLEN];

	check_arg(1);
	check_type_or_pathname_string_symbol_stream(&vs_base[0]);
	coerce_to_filename(vs_base[0], filename);
	if (chdir(filename) < 0)
		FEerror("Can't change the current directory to ~S.",
			1, vs_base[0]);
}

init_unixfsys()
{
	make_function("TRUENAME", Ltruename);
	make_function("RENAME-FILE", Lrename_file);
	make_function("DELETE-FILE", Ldelete_file);
	make_function("PROBE-FILE", Lprobe_file);
	make_function("FILE-WRITE-DATE", Lfile_write_date);
	make_function("FILE-AUTHOR", Lfile_author);
	make_function("USER-HOMEDIR-PATHNAME", Luser_homedir_pathname);
	make_function("DIRECTORY", Ldirectory);

	make_si_function("CHDIR", siLchdir);
}
