/*************************************************************************
*
* filename		: mle.cpp	(implementation of class MLE)
*
* description	: manage a Multi-Line Entryfield
*
* methods		: MLE::Open
*				  MLE::SaveFile
*				  MLE::Close
*				  MLE::SaveFileAs
*				  MLE::SetSizeAndPosition
*				  MLE::Touched
*				  MLE::Cut, Copy, Paste, Clear
*				  MLE::SetColor
*				  MLE::GetText
*				  MLE::Show
*
* functions		: CreateMLE			- create frame- and client-window
*				  ProcMLE			- MLE message processing
*				  FatalError		- display message and terminate
*
* APIs			: WinMessageBox				WinDestroyWindow
* 				  WinSetPointer				WinQuerySysPointer
*		  		  WinInitialize				WinCreateMsgQueue
*		  		  WinRegisterClass			WinDestroyMsgQueue
*		  		  WinTerminate				WinCreateStdWindow
*		  		  WinQueryWindowPos			WinSetWindowPos
*		  		  WinShowWindow             DosPostEventSem
*		  		  WinGetMsg					WinDispatchMsg
*		  		  DosEnterCritSec			DosExitCritSec
* 				  _beginthread				WinFileDlg
* 				  WinSendMsg				WinDefWindowProc
*	  	  		  WinQueryWindowRect		WinSetFocus
*		  		  WinSetWindowULong			WinQueryWindowULong
*		  		  WinQueryWindowPtr			WinBeginPaint
*		  		  WinFillRect				WinEndPaint
*		  		  WinPostMsg
*
* Used Classes	: [none]
*
* copyright (C) 1993 Jrg Caumanns (caumanns@cs.tu-berlin.de)
*
*************************************************************************/#define INCL_WINWINDOWMGR
#define INCL_WINMESSAGEMGR
#define INCL_WINFRAMEMGR
#define INCL_WINSYS
#define INCL_WINMESSAGES
#define INCL_WINPOINTERS
#define INCL_WINMLE
#define INCL_WININPUT
#define INCL_WINSTDFILE
#define INCL_WINACCELERATORS
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <process.h>
#include <stdarg.h>
#include <mem.h>

#include "mle.h"

/*
* offsets of window-instance data
*/
#define QWL_FILE		0
#define QWL_HWNDMLE		4
#define QWL_THIS		8

/*
* user-defined window messages
*/
#define SM_SETFOCUS		WM_USER +1
#define SM_SETCOLOR 	WM_USER +2
#define SM_LOADFILE 	WM_USER +3
#define SM_SAVEFILE 	WM_USER +4
#define SM_SAVEFILEAS 	WM_USER +5
#define SM_TOUCHED		WM_USER +6
#define SM_MESSAGE		WM_USER +7
#define SM_COPY			WM_USER +8

/*
* global variables
*/
static BOOL fRegistered = FALSE;	// register the windowclass only once
static LONG cDiagonal	= 0;		// simulate FCF_SHELLPOSITION

/*
* parameters needed for the window-creation
*/
struct THREADPARAM	{
	HWND	*phwnd;
	HWND	*phwndClient;
	CHAR	*pszFile;
	ULONG	ulFlags;
	HEV		hev;
	MLE		*pMLE;
	BOOL	*pfCreated;
	};

/*
* forward declarations
*/
MRESULT	EXPENTRY procMLE(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
VOID FatalError(HWND hwnd, CHAR *msg);
extern BOOL LoadFile(HWND hwnd, HWND hwndMLE, CHAR *pszFile);
extern BOOL SaveFile(HWND hwnd, HWND hwndMLE, CHAR *pszFile);
extern BOOL ValidateFilename(HWND hwnd, CHAR *pszFile);
extern BOOL CopyTextToBuffer(HWND hwnd, HWND hwndMLE, CHAR **ppszBuffer);

/*
* pointer to the active MLE
*/
MLE *MLE::pmleActive = (MLE*)0;

/*************************************************************************
*
* Name	: FatalError
*
* Descr.: Display an error-message and terminate the window
*
* APIs	: WinMessageBox				WinDestroyWindow
*
* Param.: HWND	hwnd - Window to be destroyed
*		  CHAR	*msg - message
*
* Return: -
*
*************************************************************************/
VOID FatalError(HWND hwnd, CHAR *msg)	{

	WinMessageBox(HWND_DESKTOP, hwnd, msg, "Fatal Error", 0, MB_CANCEL|MB_ERROR);
	WinDestroyWindow(hwnd);
	}

/*************************************************************************
*
* Name	: CreateMLE
*
* Descr.: create a window where the MLE will be mapped on
*
* APIs	: WinSetPointer				WinQuerySysPointer
*		  WinInitialize				WinCreateMsgQueue
*		  WinRegisterClass			WinDestroyMsgQueue
*		  WinTerminate				WinCreateStdWindow
*		  WinQueryWindowPos			WinSetWindowPos
*		  WinShowWindow             DosPostEventSem
*		  WinGetMsg					WinDispatchMsg
*		  DosEnterCritSec			DosExitCritSec
*
* Param.: HWND *ptp->phwnd			new created window
*		  HWND *ptp->phwndClient	new window's client handle
*		  CHAR *ptp->pszFile		file to be loaded
*		  ULONG	ptp->ulFlags		flags
*     	  HEV	ptp->hev			semaphore to restart main thread
*		  MLE  *ptp->pMLE			'this'
*		  BOOL *ptp->pfCreated		TRUE if window was created
*
* Return: -
*
*************************************************************************/
VOID CreateMLE(THREADPARAM *ptp)	{
	HAB   hab;
	HMQ	  hmq;
	CHAR  szClassName[] = "EDITWIN";
	ULONG fctlFrame = FCF_SIZEBORDER | FCF_MINMAX | FCF_TITLEBAR |
					  FCF_TASKLIST | FCF_SYSMENU;
	SWP   swp;

	/*
	* change icon to clock
	*/
	WinSetPointer(HWND_DESKTOP,
					WinQuerySysPointer(HWND_DESKTOP, SPTR_WAIT, FALSE));
	*ptp->pfCreated = FALSE;

	/*
	* create message queue
	*/
	if((hab = WinInitialize(0)) == NULLHANDLE)	{
		DosPostEventSem(ptp->hev);
		return;
		}

	if((hmq = WinCreateMsgQueue(hab, 0)) == NULLHANDLE)	{
		DosPostEventSem(ptp->hev);
		return;
		}

	/*
	* register window class (only if not registered)
	*/
	DosEnterCritSec();
	if(!fRegistered)	{
		fRegistered = TRUE;
		DosExitCritSec();
		if(!WinRegisterClass(hab,
							 szClassName,
							 (PFNWP)procMLE,
							 CS_SIZEREDRAW,
							 32L))	{
			WinDestroyMsgQueue(hmq);
			WinTerminate(hab);
			DosPostEventSem(ptp->hev);
			return;
			}
		}
	else
		DosExitCritSec();

	/*
	* create frame window
	*/
	CHAR *pszTitle = (ptp->pszFile)? ptp->pszFile : "Untitled";

	HWND hwndFrame;
	FRAMECDATA fcdata;
	fcdata.cb            = sizeof(FRAMECDATA);
	fcdata.flCreateFlags = fctlFrame;
	fcdata.hmodResources = (HMODULE)NULL;
	fcdata.idResources   = 0;
	if((*ptp->phwnd = (HWND)WinCreateWindow(HWND_DESKTOP,
										  WC_FRAME,
										  pszTitle,
										  0,
										  0, 0, 0, 0,
										  NULLHANDLE,
										  HWND_TOP,
										  1,
										  &fcdata,
										  NULL)) == NULLHANDLE)	{

		WinDestroyMsgQueue(hmq);
		WinTerminate(hab);
		DosPostEventSem(ptp->hev);
		return;
		}

	/*
	* create client window
	*/
	if((*ptp->phwndClient = (HWND)WinCreateWindow(*ptp->phwnd,
										  szClassName,
										  "",
										  0,
										  0, 0, 0, 0,
										  *ptp->phwnd,
										  HWND_BOTTOM,
										  FID_CLIENT,
										  (VOID*)ptp->pMLE,
										  NULL)) == NULLHANDLE)	{

		WinDestroyMsgQueue(hmq);
		WinTerminate(hab);
		DosPostEventSem(ptp->hev);
		return;
		}

	/*
	* post semaphore to free main-thread
	*/
	*ptp->pfCreated = TRUE;
	DosPostEventSem(ptp->hev);

	/*
	* show windows
	*/
	ULONG fl = SWP_SIZE | SWP_MOVE;
	if(ptp->ulFlags & MLE_MAXIMIZE)
		fl |= SWP_MAXIMIZE;
	if(ptp->ulFlags & MLE_MINIMIZE)
		fl |= SWP_MINIMIZE;
	WinQueryWindowPos(HWND_DESKTOP, &swp);
	if(((cDiagonal * MLE_DISTANCE) > (swp.x + swp.cx * MLE_XSIZE / 100)) ||
	   ((cDiagonal * MLE_DISTANCE) > (swp.y + swp.cy * MLE_YSIZE / 100)))
		 cDiagonal = 0;
	WinSetWindowPos(*ptp->phwnd,
					HWND_TOP,
					cDiagonal * MLE_DISTANCE,
					swp.cy - cDiagonal * MLE_DISTANCE - swp.cy *MLE_YSIZE / 100,
					swp.cx * MLE_XSIZE / 100,
					swp.cy * MLE_YSIZE / 100,
					fl);
	cDiagonal++;
	WinQueryWindowPos(*ptp->phwnd, &swp);
	WinSetWindowPos(*ptp->phwndClient, HWND_TOP,
					WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER),
					WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER) -1,
					swp.cx - WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER)*2,
					swp.cy - WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER)-21,
					SWP_MOVE | SWP_SIZE);
	if(!(ptp->ulFlags & MLE_HIDE))	{
		WinShowWindow(*ptp->phwnd, TRUE);
		WinShowWindow(*ptp->phwndClient, TRUE);
		}

	/*
	* Message-Loop
	*/
	QMSG  qmsg;
	while(WinGetMsg(hmq, (PQMSG)&qmsg, NULLHANDLE, 0, 0))
		WinDispatchMsg(hmq, (PQMSG)&qmsg);

	/*
	* Termination
	*/
	delete ptp;
	WinDestroyMsgQueue(hmq);
	WinTerminate(hab);
	return;
	}


/*************************************************************************
*
* Name	: MLE::Open
*
* Descr.: create an MLE-window and load a file to it.
*
* APIs	: _beginthread			WinFileDlg
*
* Param.: CHAR  *pszFile		file to be loaded;
*								if NULL: flags & MLE_NEWFILE -> empty MLE
*										 otherwise -> "open"-dialogbox
*		  ULONG	ulFlags			flags:
*								  MLE_HIDE	   : create MLE invisible
*                                 MLE_MAXIMIZE : create with max. size
*								  MLE_MINIMIZE : create with min. size
*								  MLE_NEWFILE  : see above
*
* Return: TRUE if window has been created
*
*************************************************************************/
BOOL MLE::OpenFile(CHAR *pszFile, ULONG ulf)	{
	THREADPARAM *ptp = new THREADPARAM;
	BOOL fCreated;

	/*
	* set instance variables
	*/
	if(pszFile != NULL)	{
		pszLoadedFile = new(CHAR[strlen(pszFile)+1]);
		strcpy(pszLoadedFile, pszFile);
		}
	ulFlags = ulf;

	/*
	* "open"-dialogbox, if no filename given and MLE_NEWFILE not set
	*/
	FILEDLG	fdFiledlg;
	if(!pszFile && !(ulf & MLE_NEWFILE)) {
		memset(&fdFiledlg, 0, sizeof(FILEDLG));
		fdFiledlg.cbSize	= sizeof(FILEDLG);
		fdFiledlg.fl		= FDS_HELPBUTTON | FDS_OPEN_DIALOG;
		fdFiledlg.pszTitle	= "Open File";
		strcpy(fdFiledlg.szFullFile, "*.*");
		WinFileDlg(HWND_DESKTOP, HWND_DESKTOP, &fdFiledlg);
		if(fdFiledlg.lReturn == DID_OK)	{
			pszLoadedFile = new(CHAR[strlen(fdFiledlg.szFullFile)+1]);
			strcpy(pszLoadedFile, fdFiledlg.szFullFile);
			}
		else
			return FALSE;
		}

	/*
	* Create event semaphore that will be posted by the create-thread
	*/
	HEV hev;
	if(DosCreateEventSem(NULL, &hev, 0, 0))
		return FALSE;

	/*
	* set thread parameters and start create-thread
	*/
	ptp->phwnd		= &hwnd;
	ptp->phwndClient= &hwndClient;
	ptp->pszFile	= pszLoadedFile;
	ptp->ulFlags	= ulf;
	ptp->hev		= hev;
	ptp->pMLE		= this;
	ptp->pfCreated	= &fCreated;

	_beginthread((void(*)(void*))CreateMLE, 8192, (void *)ptp);

	/*
	* wait till end of MLE-creation
	*/
	DosWaitEventSem(hev, 120000);

	return fCreated;
	}


/*************************************************************************
*
* Name	: MLE::Close
*
* Descr.: close the MLE. If the contents have changed ask the user if
*		  these changes should be copied back to the file.
*
* APIs	: WinPostMsg
*
* Param.: [none]
*
* Return: fSuccess
*
*************************************************************************/
BOOL MLE::CloseFile(VOID)	{
	if(fAlive)
		WinSendMsg(hwndClient, WM_CLOSE, MPVOID, MPVOID);
	return TRUE;
	}


/*************************************************************************
*
* Name	: MLE::SaveFile
*
* Descr.: save the contents of the MLE to the file that was loaded to
*		  the MLE.
*
* APIs	: WinPostMsg
*
* Param.: [none]
*
* Return: fSuccess
*
*************************************************************************/
BOOL MLE::SaveFile(VOID)	{
	if(!fAlive)
		return TRUE;

	return (BOOL)WinSendMsg(hwndClient, SM_SAVEFILE, MPVOID, MPVOID);
	}


/*************************************************************************
*
* Name	: MLE::SaveFileAs
*
* Descr.: save the contents of the MLE to another file than it was
*		  loaded from.
*
* APIs	: WinPostMsg
*
* Param.: [none]
*
* Return: fSuccess
*
*************************************************************************/
BOOL MLE::SaveFileAs(VOID)	{
	if(!fAlive)
		return TRUE;

	return (BOOL)WinSendMsg(hwndClient, SM_SAVEFILEAS, MPVOID, MPVOID);
	}

/*************************************************************************
*
* Name	: MLE::Resize
*
* Descr.: change size and/or position of the MLE-window
*
* APIs	: WinSetWindowPos
*
* Param.: INT	x, y	- lower left corner
*         INT	cx, cy	- size
*		  ULONG fl		- change flag
*
* Return: fSuccess
*
*************************************************************************/
VOID MLE::SetSizeAndPosition(INT x, INT y, INT cx, INT cy, ULONG fl)	{
	ULONG flag = SWP_SHOW;

	if(fAlive)	{
		if(fl & MLE_SIZEPOS_SIZE)
			flag |= SWP_SIZE;
		if(fl & MLE_SIZEPOS_POS)
			flag |= SWP_MOVE;
		if(fl & MLE_SIZEPOS_MAX)
			flag |= SWP_MAXIMIZE;
		if(fl & MLE_SIZEPOS_MIN)
			flag |= SWP_MINIMIZE;

		WinSetWindowPos(hwnd, HWND_TOP, x, y, cx, cy, flag);
		}
	}

/*************************************************************************
*
* Name	: MLE::SetColor
*
* Descr.: change forground and background color of an MLE-window
*
* APIs	: WinPostMsg
*
* Param.: LONG  clrForground
*		  LONG	clrBackground
*
* Return: fSuccess
*
*************************************************************************/
VOID MLE::SetColor(LONG clrF, LONG clrB)	{

	if(fAlive)	{
		clrForeground = clrF;
		clrBackground = clrB;
		WinPostMsg(hwndClient, SM_SETCOLOR, MPFROMLONG(clrF), MPFROMLONG(clrB));
		}
	}

/*************************************************************************
*
* Name	: MLE::Touched
*
* Descr.: Check if window was touched
*
* APIs	: WinSendMsg
*
* Param.: [none]
*
* Return: fTouched
*
*************************************************************************/
BOOL MLE::Touched(VOID)	{
	if(!fAlive)
		return FALSE;

	if(fTouched)
		return TRUE;

	return (BOOL)WinSendMsg(hwndClient, SM_TOUCHED, MPVOID, MPVOID);
	}


/*************************************************************************
*
* Names	: MLE::Cut, MLE::Copy, MLE::Paste, MLE::Clear
*
* Descr.: Move text between MLE and clipboard
*
* APIs	: WinSendMsg
*
* Param.: [none]
*
* Return: -
*
*************************************************************************/
VOID MLE::Cut(VOID)	{
	WinSendMsg(hwndClient, SM_MESSAGE, MPFROMLONG(MLM_CUT),   MPVOID);
	}

VOID MLE::Copy(VOID)	{
	WinSendMsg(hwndClient, SM_MESSAGE, MPFROMLONG(MLM_COPY),  MPVOID);
	}

VOID MLE::Paste(VOID)	{
	WinSendMsg(hwndClient, SM_MESSAGE, MPFROMLONG(MLM_PASTE), MPVOID);
	}

VOID MLE::Clear(VOID)	{
	WinSendMsg(hwndClient, SM_MESSAGE, MPFROMLONG(MLM_CLEAR), MPVOID);
	}


/*************************************************************************
*
* Name	: MLE::GetText
*
* Descr.: Copy the contents of the MLE to a buffer
*
* APIs	: WinSendMsg
*
* Param.: CHAR **ppszBuffer	- pointer to buffer (buffer will be allocated
*							  by this method)
*
* Return: fSuccess
*
*************************************************************************/
BOOL MLE::GetText(CHAR **ppszBuffer)	{
	if(!fAlive)
		return TRUE;

	return (BOOL)WinSendMsg(hwndClient,
							SM_COPY,
							MPFROMP((VOID*)ppszBuffer),
							MPVOID);
	}



/*************************************************************************
*
* Name	: MLE::Print
*
* Descr.: Insert text at the current cursor position
*
* APIs	: WinSendMsg
*
* Param.: see 'printf()'
*
* Return: -
*
*************************************************************************/
VOID MLE::Print(CHAR *pszMask, ...)	{
	CHAR szText[2048];

	/*
	* evaluate argument list
	*/
	va_list ap;
	va_start(ap, pszMask);
	vsprintf(szText, pszMask, ap);
	va_end(ap);

	/*
	* print text
	*/
	WinSendMsg(hwndClient, SM_MESSAGE, MPFROMLONG(MLM_INSERT),
													MPFROMP((VOID*)szText));
	}


/*************************************************************************
*
* Name	: MLE::Show
*
* Descr.: Show/Hide the MLE
*
* APIs	: WinShowWindow
*
* Param.: BOOL f	- TRUE  -> show MLE
*					  FALSE -> hide MLE
*
* Return: -
*
*************************************************************************/
VOID MLE::Show(BOOL f)	{

	WinShowWindow(hwnd, f);
	WinShowWindow(hwndClient, f);
	}


/*************************************************************************
*
* Name	: MRESULT EXPENTRY procMLE()
*
* Descr.: MLE message processing
*
* APIs	: WinSendMsg				WinDefWindowProc
*	  	  WinCreateWindow			WinQueryWindowRect
*		  WinSetWindowULong			WinQueryWindowULong
*		  WinQueryWindowPtr			WinBeginPaint
*		  WinFillRect				WinEndPaint
*		  WinSetWindowPos			WinSetFocus
*		  WinPostMsg			    WinFileDlg
*		  WinSetWindowPtr			WinMessageBox
*
* Param.: HWND	 hwnd		- window handle
*		  ULONG  msg		- message identifier
*		  MPARAM mp1, mp2   - message parameters
*
* Return: fQuit
*
*************************************************************************/
MRESULT	EXPENTRY procMLE(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)	{
	HPS   hps;
	RECTL rcl;
	HWND  hwndMLE;
	CHAR  *pszFile;
	MLE   *pmle;
	SHORT sResponce;
	CHAR  szQuery[256];
	LONG  clrF, clrB;

	switch(msg)	{

		/*
		* put an MLE on top of the client area
		*/
		case WM_CREATE:	/* Fenster-Initialisierung */
			WinQueryWindowRect(hwnd, (PRECTL)&rcl);
			hwndMLE = WinCreateWindow(hwnd, WC_MLE, (PSZ)NULL,
							  MLS_HSCROLL | MLS_VSCROLL | WS_VISIBLE,
							  (SHORT)rcl.xLeft, (SHORT)rcl.yBottom,
							  (SHORT)rcl.xRight, (SHORT)rcl.yTop,
							  hwnd, HWND_TOP, 0, NULL, NULL);
			if(!hwndMLE)
				return (MRESULT)TRUE;	/* Programm-Abbruch */

			WinSetWindowULong(hwnd, QWL_HWNDMLE, (ULONG)hwndMLE);

			/*
			* initialize MLE
			*/
			pmle = (MLE*)mp1;
			WinSetWindowPtr(hwnd, QWL_THIS, (VOID *)pmle);
			pszFile = pmle->LoadedFile();
			WinSetWindowPtr(hwnd, QWL_FILE, (VOID *)pszFile);
			pmle->GetColor(&clrF, &clrB);
			WinSendMsg(hwndMLE, MLM_SETTEXTCOLOR,
											MPFROMLONG(clrF), NULL);
			WinSendMsg(hwndMLE, MLM_SETBACKCOLOR,
											MPFROMLONG(clrB), NULL);
			WinSendMsg(hwndMLE, MLM_RESETUNDO, NULL, NULL);

			WinPostMsg(hwnd, SM_SETFOCUS, MPVOID, MPVOID);
			WinPostMsg(hwnd, SM_LOADFILE, MPVOID, MPVOID);
			break;

		/*
		* repaint the client area
		*/
		case WM_PAINT:
			hps = WinBeginPaint(hwnd, (HPS)NULL, &rcl);
			WinFillRect(hps, &rcl, CLR_BACKGROUND);
			WinEndPaint(hps);
			break;

		/*
		* resize the MLE
		*/
		case WM_SIZE:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			WinSetWindowPos(hwndMLE, HWND_TOP, 0, 0, SHORT1FROMMP(mp2),
											SHORT2FROMMP(mp2), SWP_SIZE);
			break;

		/*
		* MLE got the focus
		*/
		case WM_ACTIVATE:
			if(SHORT1FROMMP(mp1))	{
				pmle = (MLE*)WinQueryWindowPtr(hwnd, QWL_THIS);
				pmle->Activate();
				}
			break;

		case WM_SETFOCUS:
			if(SHORT1FROMMP(mp1))	{
				hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
				WinPostMsg(hwnd, SM_SETFOCUS, NULL, NULL);
				}
			break;

		case SM_SETFOCUS:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			WinSetFocus(HWND_DESKTOP, hwndMLE);
			break;

		/*
		* Send a message to the MLE
		*   mp1		: message-ID
		*   mp2		: first message parameter
		*	return  : message result
		*/
		case SM_MESSAGE:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			return WinSendMsg(hwndMLE, (ULONG)LONGFROMMP(mp1), mp2, MPVOID);

		/*
		* change foreground and background color
		*   mp1		: foreground color
		*	mp2		: background color
		*	return	: -
		*/
		case SM_SETCOLOR:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			WinSendMsg(hwndMLE, MLM_SETTEXTCOLOR, mp1, MPVOID);
			WinSendMsg(hwndMLE, MLM_SETBACKCOLOR, mp2, MPVOID);
			break;

		/*
		* check if window contents changed
		*	return	: TRUE if contents changed, else FALSE
		*/
		case SM_TOUCHED:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			return WinSendMsg(hwndMLE, MLM_QUERYCHANGED, MPVOID, MPVOID);

		/*
		* copy MLE-contents to a buffer
		*	mp1		: buffer address
		*	mp2		: -
		*	return	: TRUE if text was copied
		*/
		case SM_COPY:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			return (MRESULT)CopyTextToBuffer(hwnd, hwndMLE, (CHAR**)(mp1));

		/*
		* load a file to the MLE
		*/
		case SM_LOADFILE:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			pszFile = (CHAR*)WinQueryWindowPtr(hwnd, QWL_FILE);
			if(pszFile)
				LoadFile(WinQueryWindow(hwnd, QW_PARENT), hwndMLE, pszFile);
			break;

		/*
		* save MLE contents (if it changed)
		* if no filename was given, SM_SAVEFILEAS is called implicit.
		*/
		case SM_SAVEFILE:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);
			pszFile = (CHAR*)WinQueryWindowPtr(hwnd, QWL_FILE);
			if(pszFile)	{
				if(SaveFile(WinQueryWindow(hwnd, QW_PARENT),
							hwndMLE,
							pszFile))	{
					pmle = (MLE*)WinQueryWindowPtr(hwnd, QWL_THIS);
					pmle->TouchFile(FALSE);
					return (MRESULT)TRUE;
					}
				return (MRESULT)FALSE;
				}

		/*
		* ask for a filename and copy the cntents of the MLE to this file
		*/
		case SM_SAVEFILEAS:
			hwndMLE = WinQueryWindowULong(hwnd, QWL_HWNDMLE);

			/*
			* display "save as"-dialogbox
			*/
			FILEDLG	fdFiledlg;
			memset(&fdFiledlg, 0, sizeof(FILEDLG));
			fdFiledlg.cbSize	= sizeof(FILEDLG);
			fdFiledlg.fl		= FDS_HELPBUTTON | FDS_CENTER | FDS_SAVEAS_DIALOG;
			fdFiledlg.pszTitle	= "Save File As...";
			strcpy(fdFiledlg.szFullFile, "*.*");
			WinFileDlg(HWND_DESKTOP, hwnd, &fdFiledlg);
			if(fdFiledlg.lReturn == DID_OK)	{
				/*
				* check if file already exists (ask if overwrite)
				*/
				if(ValidateFilename(WinQueryWindow(hwnd, QW_PARENT),
														fdFiledlg.szFullFile))
					if(SaveFile(WinQueryWindow(hwnd, QW_PARENT),
								hwndMLE,
								fdFiledlg.szFullFile))	{
						pmle = (MLE*)WinQueryWindowPtr(hwnd, QWL_THIS);
						pmle->TouchFile(FALSE);
						return (MRESULT)TRUE;
						}
				}
			return (MRESULT)FALSE;

		/*
		* close the MLE (ask the user if contents should be saved)
		*/
		case WM_CLOSE:
			pmle = (MLE*)WinQueryWindowPtr(hwnd, QWL_THIS);
			pszFile = (CHAR*)WinQueryWindowPtr(hwnd, QWL_FILE);
			if(pmle->Touched())		{
				if(!pszFile)
					pszFile = "[Untitled]";
				sprintf(szQuery, "%s has been changed! Save it?", pszFile);
				sResponce = WinMessageBox(HWND_DESKTOP, hwnd, szQuery,
									"Close File", 0, MB_QUERY | MB_YESNO);

				if(sResponce == MBID_YES)
					pmle->SaveFile();
				}

			pmle->SetClosed();
			return(WinDefWindowProc(hwnd, msg, mp1, mp2));

		default:
			return(WinDefWindowProc(hwnd, msg, mp1, mp2));
		}
	return((MRESULT)NULL);
	}


