

/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */


#include <iostream.h>
#include <fstream.h>
#include <limits.h>

#include "ids.h"
#include "appwin.h"
#include "draghndl.h"
#include "vobjset.h"
#include "vobjdesc.h"
#include "util.h"

#include "base/strgseq.h"
#include "base/integer.h"

#include "ui/composit.h"
#include "ui/stddlg.h"
#include "ui/label.h"
#include "ui/menu.h"
#include "ui/dsplsurf.h"
#include "ui/shadorec.h"
#include "ui/dialog.h"
#include "ui/font.h"
#include "ui/applic.h"
#include "ui/cursor.h"



UI_MenuItemDescriptor FileMenu [] = {
  {"&New...",       ID_NEW,      NULL}
, {"&Open...",      ID_OPEN,     NULL}
, {"&Save As...",   ID_SAVEAS,   NULL}
, {UIMenu_Separator,NULL,        NULL}
, {"&Font...",      ID_FONT,     NULL}
, {"&Test...",      ID_TESTDLG,  NULL}
, {UIMenu_Separator,NULL,        NULL}
, {"E&xit",         ID_QUIT,     NULL}
, {NULL,            0,           NULL}
};


UI_MenuItemDescriptor EditMenu [] = {
  {"&Cut",            ID_CUT,      NULL}
, {"C&opy",           ID_COPY,     NULL}
, {"&Paste",          ID_PASTE,    NULL}
, {"&Delete",         ID_DELETE,   NULL}
, {UIMenu_Separator,  NULL,        NULL}
, {"Rep&licate...",   ID_REPL,     NULL}
, {"&Align...",       ID_ALIGN,    NULL}
, {"&Size...",        ID_SIZE,     NULL}
, {"P&roperties...",  ID_PROP,     NULL}
, {NULL,              0,           NULL}
};


UI_MenuItemDescriptor ObjectsMenu [] = {
  {"&Label",              ID_LABEL,    NULL}
, {UIMenu_Separator,      NULL    ,    NULL}
, {"&StringView",         ID_STRVW,    NULL}
, {"&ComboBox",           ID_COMBO,    NULL}
, {UIMenu_Separator,      NULL    ,    NULL}
, {"St&ringEditor",       ID_STRED,    NULL}
, {"TextE&ditor",         ID_TEXT,     NULL}
, {UIMenu_Separator,      NULL    ,    NULL}
, {"&PushButton",         ID_PUSHBN,   NULL}
, {"&ToggleButton",       ID_TGLBN,    NULL}
, {"&ExOr ToggleButton",  ID_XRTGL,    NULL}
, {UIMenu_Separator,      NULL    ,    NULL}
, {"&Vert. Scroll Bar",   ID_VSCRL,    NULL}
, {"&Horz. Scroll Bar",   ID_HSCRL,    NULL}
, {UIMenu_Separator,      NULL    ,    NULL}
, {"&Or Button Group",    ID_ORGRP,    NULL}
, {"E&xOr Button Group",  ID_XRGRP,    NULL}
, {NULL,                  0,           NULL}
};



UI_MenuItemDescriptor HelpMenu [] = {
  {"&About...",     ID_ABOUT,    NULL}
, {NULL,            0,           NULL}
};


UI_MenuItemDescriptor MainMenuDesc [] = {
    {"&File",       ID_FILE,     FileMenu}
  , {"&Edit",       ID_EDIT,     EditMenu}
  , {"&Create",     ID_CREATE,   ObjectsMenu}
  , {"&Help",       ID_HELP,     HelpMenu}
  , {NULL,          0,           NULL}
};


typedef CL_Binding0<AppWindow> AppBind;

#if defined(__GNUC__)
template class CL_Binding0<AppWindow>;
#endif


#if defined(__X_MOTIF__)
static short FONT_SIZE = 12; // X fonts seem much smaller than Windows
                             // and OS/2 versions
#elif defined(__OS2__)
static short FONT_SIZE = 9;  // And OS/2 fonts are big!
#else
static short FONT_SIZE = 10;
#endif



AppWindow::AppWindow()
: UI_CompositeVObject (NULL, UI_Rectangle (50, 100, 550, 400)), _editor (this)
{
    Title() = "YACL Dialog Editor";
    UI_MenuBar* bar = new UI_MenuBar (this, MainMenuDesc);
    _driver.Setup (bar);
}



AppWindow::~AppWindow()
{
    DestroyDisplaySurface();
    delete _moveHandler;
    delete _sizeHandler;
    delete _rectPainter;
}

void AppWindow::Initialize ()
{
    Invalidate ();
    SetMode (EditMode);
    _activeHandler = NULL;
    AppBind rectBind (this, &AppWindow::_RectPaintFinished);
    _rectPainter   = new RectanglePainter (this, rectBind);
    AppBind sizeBind (this, &AppWindow::_SizerFinished);
    _sizeHandler = new SizeHandler (this, sizeBind);
    AppBind moveBind (this, &AppWindow::_MoverFinished);
    _moveHandler   = new MoveHandler (this, moveBind);
}


void AppWindow::ShowVObj (VObjDesc& desc)
{
    if (!_displaySurface)
        return;
    //    _displaySurface->SaveState ();
    _displaySurface->Pen().Color   (UIColor_Black);
    _displaySurface->Pen().Pattern (UIPen_Solid);
    _displaySurface->DrawRectangle (desc._shape);
    UI_Rectangle clipRect = UI_Rectangle
        (desc._shape.Origin(), desc._shape.Width(),
         minl (desc._shape.Height(), 20));
    if (desc._title.Size() > 0) {
        _displaySurface->WriteString (desc._title.AsPtr(), clipRect,
                                      UIText_Center);
    }
    else {
        CL_String s = StringForm (desc._type) + " " + CL_String (desc._id);
        _displaySurface->WriteString (s.AsPtr(), clipRect, UIText_Center);
    }
    //    _displaySurface->RestoreState ();
}


bool AppWindow::MouseMove (const UI_Point& pt)
{
    UI_CursorType c =  (_currentMode == CreateMode ||
                        _currentMode == PasteMode)
        ? UICursor_CrossHairs 
        :_sizeHandler->CursorType (pt);
    UI_Cursor& current = Cursor();
    if (current != c) // This test is needed so that we don't set cursors
                      // on every mouse move. Cursor setting on some
                      // platforms (e.g. X) is expensive.
        current = c;
    return FALSE;
}


bool AppWindow::Paint (const UI_Rectangle& rect)
{
    _Redraw ();
    return FALSE;
}



bool AppWindow::ButtonDown (const UI_Point& pt, UI_MouseButton btn,
                            bool shiftKey, bool /* ctrlKey */)
{
    _vObjAtBtnDown = NULL;
    if (btn != UIM_Left)
        return FALSE;
    if (_currentMode == CreateMode) {
        CreateDisplaySurface();
        _displaySurface->Font() = UI_FontDesc (UIFont_Helvetica, FONT_SIZE,
                                               UIFont_BoldFace);
        _EraseSelectionHandles();
        _editor.Selection().MakeEmpty();
        VObjDesc* vObj = _editor.Add (_driver.CurrentObjectType(), pt);
        ShowVObj (*vObj);
        _editor.Selection().Add (vObj);
        _DrawSelectionHandles ();
        _currentMode = EditMode;
        DestroyDisplaySurface ();
    }
    else if (_currentMode == PasteMode) {
        _editor.DoPaste (pt);
        _currentMode = EditMode;
    }
    else {
        CreateDisplaySurface(); // Will be destroyed by the drag handler
                                // notification method
        // Now let's decide if we want moving, resizing or selection
        VObjDesc* vObj = _editor.Store().PointHits (pt);
        if (vObj) {
            if (_editor.Selection().Includes (vObj)) {
                // Only selected objects can be moved or resized
                UI_CursorType c = Cursor().Type();
                if (c != UICursor_Arrow) {
                    _activeHandler = _sizeHandler;
                    _vObjAtBtnDown = vObj;
                    _sizeHandler->SetSizingParams (vObj, pt);
                }
                else
                    _activeHandler = _moveHandler;
            }
            else {
                _vObjAtBtnDown = vObj;
                _activeHandler = _rectPainter;
            }
                
        }
        else
            _activeHandler = _rectPainter;
        _activeHandler->SetActiveState (TRUE);
        _shiftKeyDown = shiftKey;
    }
    return FALSE;
}



bool AppWindow::DoubleClick (const UI_Point& , UI_MouseButton btn)
{
    if (btn == UIM_Left)
        _editor.DoProperties ();
    return FALSE;
}



bool AppWindow::_RectPaintFinished ()
{
    _DoSelect (_rectPainter->Rectangle());
    DestroyDisplaySurface ();
    _activeHandler->SetActiveState (FALSE);
    _vObjAtBtnDown = NULL;
    return TRUE;
}


bool AppWindow::_MoverFinished ()
{
    DestroyDisplaySurface ();
    _activeHandler->SetActiveState (FALSE);
    const UI_Vector& vec = _moveHandler->Translation();
    VObjSetIterator itr (_editor.Selection());
    while (itr.More()) {
        VObjDesc* v = itr.Next();
        v->_shape += vec;
    }
    Invalidate(); // So that everything is redrawn
    return TRUE;
}


bool AppWindow::_SizerFinished ()
{
    DestroyDisplaySurface ();
    _vObjAtBtnDown->_shape = _sizeHandler->Rectangle();
    _vObjAtBtnDown = NULL;
    _activeHandler->SetActiveState (FALSE);
    Invalidate();
    return TRUE;
}


void AppWindow::_DoSelect (const UI_Rectangle& rect)
{
    // Called  in Edit mode after a drag has finished
    _EraseSelectionHandles ();
    VObjSet& selection = _editor.Selection();
    if (!_shiftKeyDown)
        selection.MakeEmpty ();
    _shiftKeyDown = FALSE; 
    if (rect.Width() <= 2 && rect.Height() <= 2) {
        // The user probably just clicked once
        if (_vObjAtBtnDown)
            selection.Add (_vObjAtBtnDown);
    }
    else {
        // Looks like the user drew a big rectangle; find all the objects
        // that intersect it, and select all of them
        VObjSetIterator itr (_editor.Store());
        while (itr.More()) {
            VObjDesc* vObj = itr.Next();
            if (vObj->_shape.IsContainedIn (rect))
                _editor.Selection().Add (vObj);
        }
    }
    if (selection.Size())
        _DrawSelectionHandles (); // Draw the new selections
}


void AppWindow::_DrawSelectionHandles ()
{
    _displaySurface->Mode (UI_DisplaySurface::GMode_Xor);
    _displaySurface->Pen().Pattern (UIPen_Solid);
    _displaySurface->Pen().Color   (UIColor_MediumGray);
    VObjSetIterator itr (_editor.Selection());
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        _DrawHandles (vObj->_shape);
    }
    _displaySurface->Mode (UI_DisplaySurface::GMode_Copy);
}

void AppWindow::_EraseSelectionHandles ()
{
    _DrawSelectionHandles();
}


void AppWindow::_DrawHandles (const UI_Rectangle& r)
{
    _displaySurface->DrawRectangle
        (UI_Rectangle (r.TopLeft()     + UI_Vector (-3,-3), 3, 3));
    _displaySurface->DrawRectangle
        (UI_Rectangle (r.TopRight()    + UI_Vector ( 0,-3), 3, 3));
    _displaySurface->DrawRectangle
        (UI_Rectangle (r.BottomLeft()  + UI_Vector (-3, 0), 3, 3));
    _displaySurface->DrawRectangle
        (UI_Rectangle (r.BottomRight(), 3, 3));
}



void AppWindow::SetMode (AppWindow::Mode newMode)
{
    _currentMode = newMode;
    Cursor() = (newMode == CreateMode || newMode == PasteMode)
        ? UICursor_CrossHairs : UICursor_Arrow;
}


void AppWindow::_Redraw ()
{
    UI_DisplaySurface& sfc = CreateDisplaySurface ();
    _displaySurface->Font() = UI_FontDesc (UIFont_Helvetica, FONT_SIZE,
                                           UIFont_BoldFace);
    UI_Rectangle r = Shape();
    r.Origin (UI_Point (0, 0));
    sfc.ColorRectangle (r, UIColor_White);
    VObjSetIterator itr (_editor.Store());
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        ShowVObj (*vObj);
    }
    _DrawSelectionHandles ();
    DestroyDisplaySurface ();
}
