

/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */




#include <iostream.h>
#include <fstream.h>
#include <limits.h>

#include "ids.h"
#include "appwin.h"
#include "vobjset.h"
#include "vobjdesc.h"
#include "util.h"

#include "base/strgseq.h"
#include "base/integer.h"

#include "io/pathname.h"

#include "ui/composit.h"
#include "ui/cntroler.h"
#include "ui/stddlg.h"
#include "ui/label.h"
#include "ui/dialog.h"
#include "ui/font.h"
#include "ui/applic.h"

#include "dlgedit.h"


static const char* EDITOR_VERSION = "0.1" ;

static const int MINIMUM_ID = 50; // Smallest view id assigned

static const char* MagicBeginMarker = "YACLDlgEditorBegin";
static const char* MagicEndMarker   = "YACLDlgEditorEnd";

#if defined(__X_MOTIF__)
static short DialogFontSize = 12; // X fonts seem much smaller than Windows
                                  // and OS/2 versions
#elif defined(__OS2__)
static short DialogFontSize = 9;  // And OS/2 fonts are big!
#else
static short DialogFontSize = 10;
#endif


DialogEditor::DialogEditor (AppWindow* comp)
    : _client (comp), _currentWorkingDir (".")
{
    _client = comp;
    extern UI_ViewDescriptor PropDlg_dlgDesc[];
    extern UI_RectangleStruct PropDlg_shape;
    extern UI_DialogEventDescriptor terminators[];
    _propDlg = new UI_Dialog
        (_client, PropDlg_dlgDesc, PropDlg_shape, terminators, ID_OK);
    _propDlg->Title() = "Properties";
    _propDlg->Font()  = UI_FontDesc (UIFont_Helvetica, DialogFontSize,
                                     UIFont_BoldFace);
    _propDlg->MakeInvisible ();
}


DialogEditor::~DialogEditor ()
{
    _store.DestroyContents();
    _clipBoard.DestroyContents ();
}



struct SizeStruct {
    SizeStruct (short wid, short ht) : w (wid), h(ht) {};
    short w;
    short h;
};

static SizeStruct _DefaultSize (UI_ViewType type)
{
    switch (type) {
    case View_PushButton:
    case View_ToggleButton:
    case View_ExOrToggleButton:
        return SizeStruct (75, 30);

    case View_ComboBox:
    case View_TextView:
        return SizeStruct (75, 60);
        
    case View_HScrollBar:
        return SizeStruct (230, 30);
        
    case View_VScrollBar:
        return SizeStruct (30, 230);
        
    case View_StringViewSingle:
    case View_StringViewMulti:
    case View_OrButtonGroup:
    case View_ExOrButtonGroup:
        return SizeStruct (125, 90);
        
    default:
        return SizeStruct (125, 30);
    }
}

VObjDesc* DialogEditor::Add (UI_ViewType typ, const UI_Point& pt)
{
    UI_ViewID id = AllocateId ();
    SizeStruct sz = _DefaultSize (typ);
    UI_Rectangle rect (pt, sz.w, sz.h);
    VObjDesc* vObj = new VObjDesc (typ, id, rect);
    _store.Add (vObj);
    _usedIdSet.Add (vObj->_id);
    return vObj;
}


void DialogEditor::DoNew ()
{
    extern UI_ViewDescriptor NewDialog_dlgDesc[];
    extern UI_RectangleStruct NewDialog_shape;
    UI_Dialog* newDlg = new UI_Dialog
        (_client, NewDialog_dlgDesc, NewDialog_shape, NULL, ID_OK);
    newDlg->Title() = "Create a New Dialog";
    newDlg->ExecuteModal();
    _dlgName = (*newDlg)[ID_DLGNAME]->Model();
    YACLApp()->Destroy (newDlg);
    _client->Title() = "YACL Dialog Editor: " + _dlgName;
    _store.DestroyContents ();
    _selection.MakeEmpty ();
}


UI_ViewID DialogEditor::AllocateId ()
{
    UI_ViewID id = _usedIdSet.SmallestNonMemberAbove (MINIMUM_ID);
    _usedIdSet.Add (id);
    return id;
}


void DialogEditor::_Replicate (UI_Dialog* dlg)
{
    short nRows    = ((CL_String&) ((*dlg)[ID_NROWS]->Model())).AsLong();
    short nCols    = ((CL_String&) ((*dlg)[ID_NCOLS]->Model())).AsLong();
    short rowPxls  = ((CL_String&) ((*dlg)[ID_ROWPXLS]->Model())).AsLong();
    short colPxls  = ((CL_String&) ((*dlg)[ID_COLPXLS]->Model())).AsLong();
    VObjDesc* vObj = (VObjDesc*) _selection.ItemWithRank (0);
    short i, j;
    long y = vObj->_shape.Top();
    for (i = 0; i < nRows; i++) {
        long x = vObj->_shape.Left();
        for (j = 0; j < nCols; j++) {
            if (i != 0 || j != 0) { // Don't generate the first one
                UI_ViewID id = _usedIdSet.SmallestNonMemberAbove (MINIMUM_ID);
                UI_Rectangle shape (x, y, vObj->_shape.Width(),
                                    vObj->_shape.Height());
                VObjDesc* v = new VObjDesc (vObj->_type, id, shape);
                Store().Add (v);
                _usedIdSet.Add (id);
            }
            x += colPxls + vObj->_shape.Width();
        }
        y += rowPxls + vObj->_shape.Height();
    }
}


void DialogEditor::DoProperties ()
{
    if (_selection.Size() != 1) {
        UI_StandardDialog ("Please select one object (and no more)",
                           "Dialog editor", _client);
        return;
    }
    VObjDesc* v = (VObjDesc*) _selection.ItemWithRank (0);
    (*_propDlg)[ID_TYPES]->Model()   = StringForm (v->_type);
    (*_propDlg)[ID_VIEWID]->Model()  = CL_String (v->_id);
    (*_propDlg)[ID_SYMNAME]->Model() = v->_symbolicName;
    (*_propDlg)[ID_POSX]->Model()    = CL_String (v->_shape.Left());
    (*_propDlg)[ID_POSY]->Model()    = CL_String (v->_shape.Top());
    (*_propDlg)[ID_WIDTH]->Model()   = CL_String (v->_shape.Width());
    (*_propDlg)[ID_HEIGHT]->Model()  = CL_String (v->_shape.Height());
    (*_propDlg)[ID_TITLE]->Model()   = v->_title;
    (*_propDlg)[ID_TABSTOP]->Model() = CL_Integer (v->_isTabStop);
    _propDlg->MakeVisible();
    YACLApp()->Controller().GiveFocusTo (_propDlg);
    if (_propDlg->ExecuteModal().id == ID_OK) {
        v->_title = CL_CAST_REF(CL_String&, (*_propDlg)[ID_TITLE]->Model());
        v->_symbolicName = CL_CAST_REF
            (CL_String&, (*_propDlg)[ID_SYMNAME]->Model());
        v->_isTabStop = CL_CAST_REF (CL_Integer&,
                                     (*_propDlg)[ID_TABSTOP]->Model());
        long x = CL_CAST_REF (CL_String&,
                              (*_propDlg)[ID_POSX]->Model()).AsLong();
        long y = CL_CAST_REF (CL_String&,
                              (*_propDlg)[ID_POSY]->Model()).AsLong();
        long w = CL_CAST_REF (CL_String&,
                              (*_propDlg)[ID_WIDTH]->Model()).AsLong();
        long h = CL_CAST_REF (CL_String&,
                              (*_propDlg)[ID_HEIGHT]->Model()).AsLong();
        if (x >= 0 && y >= 0 && w >= 4 && h >= 4)
            v->_shape = UI_Rectangle (x, y, w, h);
        _client->Invalidate();
    }
    _propDlg->MakeInvisible();
}


void DialogEditor::DoReplicate ()
{
    if (_selection.Size() != 1) {
        UI_StandardDialog ("Please select one object (and no more)",
                           "Dialog editor", _client);
        return;
    }
    extern UI_ViewDescriptor ReplicateDlgDesc[];
    extern UI_RectangleStruct ReplicateDlgShape;
    extern UI_DialogEventDescriptor terminators[];
    UI_Dialog* replDlg = new UI_Dialog
        (_client, ReplicateDlgDesc, ReplicateDlgShape, terminators, ID_OK);
    replDlg->Font()  = UI_FontDesc (UIFont_Helvetica, DialogFontSize,
                                    UIFont_BoldFace);
    replDlg->Title() = "Replicate";
    (*replDlg)[ID_NROWS]->Model() = CL_String ("2");
    (*replDlg)[ID_NCOLS]->Model() = CL_String ("2");
    (*replDlg)[ID_ROWPXLS]->Model() = CL_String ("5");
    (*replDlg)[ID_COLPXLS]->Model() = CL_String ("5");
    if (replDlg->ExecuteModal().id == ID_OK) {
        _Replicate (replDlg);
        _client->Invalidate();
    }
    YACLApp()->Destroy (replDlg);
}




static UI_ViewDescriptor _DescriptorOf (VObjDesc* vObj)
{
    UI_ViewDescriptor dsc;
    if (vObj) {
        dsc.type     = vObj->_type;
        dsc.id       = vObj->_id;
        dsc.shape.x  = vObj->_shape.Left();
        dsc.shape.y  = vObj->_shape.Top();
        dsc.shape.w  = vObj->_shape.Width();
        dsc.shape.h  = vObj->_shape.Height();
        dsc.title    = (char*) vObj->_title.AsPtr();
        dsc.tab_stop = FALSE;
        dsc.enclosed = NULL;
    }
    return dsc;
}


static UI_ViewDescriptor* _DescArrayOf (const VObjSet& objSet, const
                                        UI_Point& p)
{
    short n = objSet.Size();
    if (!n)
        return NULL;
    UI_ViewDescriptor* desc = new UI_ViewDescriptor [n + 1];
    VObjSetIterator itr (objSet);
    short i = 0;
    while (itr.More()) {
        desc[i] = _DescriptorOf (itr.Next());
        desc[i].shape.x -= p.XCoord();
        desc[i].shape.y -= p.YCoord();
        i++;
    }
    desc[i].type = View_None;
    desc[i].id   = 0;
    return desc;
}

void DialogEditor::DoTest ()
{
    UI_ViewDescriptor* desc = new UI_ViewDescriptor [_store.Size() + 1];
    short n = 0;
    VObjSet storeCopy = _store;
    VObjSet groups;

    // First, we separate out the  button groups
    VObjSetIterator itr (storeCopy);
    while (itr.More()) {
        VObjDesc* vObj   = itr.Next();
        if (vObj->_type == View_OrButtonGroup ||
            vObj->_type == View_ExOrButtonGroup)
            groups.Add (vObj);
    }

    // Now we build descriptors for the groups
    VObjSetIterator grpIter (groups);
    VObjSet buttons;
    while (grpIter.More()) {
        VObjDesc* grp = grpIter.Next();
        VObjSetIterator itr (storeCopy);
        VObjSet children;
        while (itr.More()) {
            VObjDesc* vObj   = itr.Next();
            if (((vObj->_type == View_ToggleButton && grp->_type ==
                 View_OrButtonGroup) ||
                (vObj->_type == View_ExOrToggleButton && grp->_type ==
                 View_ExOrButtonGroup)) &&
                 vObj->_shape.IsContainedIn (grp->_shape)) {
                children.Add (vObj);
                buttons.Add (vObj);
            }
        }
        desc[n]           = _DescriptorOf (grp);
        desc[n].enclosed  = _DescArrayOf  (children, grp->_shape.Origin());
        n++;
    }

    // Finally, we build the descriptors for the remaining objects
    storeCopy -= buttons;
    storeCopy -= groups;
    
    itr.Reset ();
    while (itr.More()) {
        VObjDesc* vObj   = itr.Next();
        desc[n++] = _DescriptorOf (vObj);
    }    

    // And we're ready to roll.
    UI_Rectangle shape = _client->Shape();
    shape.Origin (UI_Point (40, 40));
    UI_Dialog* testDlg = new UI_Dialog (_client, desc, shape);
    testDlg->Title() = "Test Dialog";
    if (_chosenFont.TypeFace().Size())
        testDlg->Font()  = _chosenFont;
    testDlg->ExecuteModal();
    YACLApp()->Destroy (testDlg);

    for (short nn = 0; nn < n; nn++)
        if (desc[nn].enclosed)
            delete [] desc[nn].enclosed;
    delete [] desc;
}


void DialogEditor::DoCut ()
{
    if (!_selection.Size())
        return;
    _clipBoard.DestroyContents();
    _clipBoard = _selection;
    _store    -= _selection;
    VObjSetIterator itr (_selection);
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        _usedIdSet.Remove (vObj->_id);
    }
    _selection.MakeEmpty ();
    _client->Invalidate ();
}


void DialogEditor::DoCopy ()
{
    if (!_selection.Size())
        return;
    _clipBoard.DestroyContents ();
    VObjSetIterator itr (_selection);
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        _clipBoard.Add (new VObjDesc (*vObj));
    }
}


void DialogEditor::DoPaste (const UI_Point& pt)
{
    if (_clipBoard.Size() == 0)
        return;
    // First, let's compute how much to translate by:
    long xmin = LONG_MAX, ymin = LONG_MAX;
    VObjSetIterator itr (_clipBoard);
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        xmin = minl (xmin, vObj->_shape.Left());
        ymin = minl (ymin, vObj->_shape.Top());
    }
    UI_Vector trans = UI_Vector (pt) - UI_Vector (xmin, ymin);
    if (trans == UI_Vector (0, 0))
        return; // Don't overwrite

    // Now let's translate all the pasted objects and add them to the store.
    itr.Reset ();
    while (itr.More()) {
        VObjDesc* vObj = new VObjDesc (*(VObjDesc*) itr.Next());
        vObj->_id = AllocateId();
        vObj->_symbolicName = "";
        vObj->_shape += trans;
        _store.Add (vObj);
    }
    _client->Invalidate();
}


static CL_String _IdFileName (CL_String& itemFileName)
{
    short n = itemFileName.Size() - 1;
    for (; n >= 0; n--) {
        if (itemFileName[n] == '.')
            break;
    }
    if (n > 0)
        return itemFileName (0, n) + ".ids";
    else
        return itemFileName + ".ids";
}



CL_String DialogEditor::_GroupName (VObjDesc& dsc)
{
    return  "Group_" + CL_String (dsc._id);
}


CL_String DialogEditor::_BuildGroup (VObjDesc& d1, VObjSet& saveButtons,
                                     CL_String& erMsg)
{
    CL_String dsc;
    CL_String groupName = _GroupName (d1);
    const UI_Point& origin = d1._shape.Origin();
    dsc.AssignWithFormat
        ("UI_ViewDescriptor %s [] = {\n//---- %s %s group %s %ld %ld ------\n",
         (_dlgName + groupName).AsPtr(), MagicBeginMarker,
         _dlgName.AsPtr(), groupName.AsPtr(),
         origin.XCoord(), origin.YCoord());
    VObjSetIterator itr (_store);
    UI_ViewType childType = d1._type == View_OrButtonGroup ?
        View_ToggleButton : View_ExOrToggleButton;
    while (itr.More()) {
        VObjDesc* d2 = itr.Next();
        if ( d1._id != d2->_id && d2->_shape.IsContainedIn (d1._shape)) {
            if (d2->_type != childType) {
                erMsg.AssignWithFormat
                    ("Invalid group: id %ld contains a %s", d1._id,
                     StringForm (d2->_type).AsPtr());
                return "";
            }
            VObjDesc newBtn (*d2);
            newBtn._shape -= UI_Vector (d1._shape.Origin());
            dsc += "{" + newBtn.AsString() + "},\n";
            saveButtons.Add (d2);
        }
    }
    dsc += "{View_None, 0, 0}\n//---- " + CL_String (MagicEndMarker)
        + " " + groupName + " ----\n};\n\n";
    return dsc;
}


CL_String DialogEditor::_BuildButtonGroups (CL_String& desc,
                                            VObjSet& savedButtons)
{
    // Build button-group descriptor string and return it in desc.
    VObjSetIterator itr1 (_store);
    while (itr1.More()) {
        VObjDesc* d1 = itr1.Next();
        if (d1->_type == View_OrButtonGroup || d1->_type ==
            View_ExOrButtonGroup) {
            CL_String errMsg;
            CL_String dsc = _BuildGroup (*d1, savedButtons, errMsg);
            if (errMsg.Size())
                return errMsg;
            desc += dsc;
        }
    }
    return "";                    
}

void DialogEditor::DoSave ()
{
    CL_String groupDescs;
    VObjSet savedButtons; // Used to save buttons that are children of
                          // groups, so that they are not output multiple times
    CL_String errMsg = _BuildButtonGroups (groupDescs, savedButtons);
    if (errMsg.Size()) {
        UI_StandardDialog (errMsg, "Invalid button group", _client);
        return;
    }

    CL_String fileName = UI_FileSelectDialog (_client, ".", "Save As", "");
    if (fileName.Size() <= 0)
        return;
    ofstream dlgStream (fileName.AsPtr());
    if (dlgStream.bad()) {
        CL_String s = "Cannot open file " + fileName;
        UI_StandardDialog (s.AsPtr(), "Open failure", _client, UIS_Ok,
                           UIS_Error);
        return;
    }

    dlgStream << groupDescs;
    
    _store -= savedButtons; // Note that I cannot remove buttons from the
                           // store inside the above loop, because the store
                           // has an iterator active on it.
    UI_Rectangle shape = _client->Shape();
    dlgStream << "\n\nUI_RectangleStruct " << _dlgName << "_shape = {\n";
    dlgStream << "//---- " << MagicBeginMarker << ' ' << _dlgName
              << " shape ------\n";
    dlgStream << "    " << shape.Left() << ", " << shape.Top() << ", "
              << shape.Width() << ", " << shape.Height() << '\n';
    dlgStream << "//---- " << MagicEndMarker << ' ' << _dlgName
              << " shape ------\n};\n\n";
    dlgStream << "\n\nUI_ViewDescriptor " << _dlgName
              << "_dlgDesc [] = {\n";
    dlgStream << "//---- " << MagicBeginMarker << ' ' << _dlgName
              << " items ------\n";
    VObjSetIterator itr (_store);
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        dlgStream << '{' << vObj->AsString().AsPtr();
        if (vObj->_type == View_OrButtonGroup ||
            vObj->_type == View_ExOrButtonGroup)
            dlgStream << ", " << _GroupName (*vObj);
        dlgStream << "},\n";
    }
    dlgStream << "{View_None, 0, 0}\n";
    dlgStream << "//---- " << MagicEndMarker << ' ' << _dlgName
               << " items ------\n";
    dlgStream << "};\n\n";

    if (savedButtons.Size())
        _store += savedButtons;

    
    // Now output the symbolic names into the file filename.ids
    
    itr.Reset();
    while (itr.More()) {
        VObjDesc* vObj = (VObjDesc*) itr.Next();
        if (vObj->_symbolicName.Size())
            break;
    }
    if (!itr.More())
        return; // No symbolic names
    fileName = _IdFileName (fileName);
    ofstream idStream (fileName);
    if (idStream.bad()) {
        CL_String s = "Cannot open file " + fileName;
        UI_StandardDialog (s.AsPtr(), "Open failure", _client);
        return;
    }
    idStream << "enum " << _dlgName << "_dlg_idEnum {\n";
    idStream << "//---- " << MagicBeginMarker << ' ' << _dlgName
             << " ids ------\n";
    itr.Reset(); short count = 0;
    while (itr.More()) {
        VObjDesc* vObj = (VObjDesc*) itr.Next();
        if (vObj->_symbolicName.Size()) {
            if (count > 0)
                idStream << ", ";
            else
                idStream << "  ";
            idStream << vObj->_symbolicName << " = " << vObj->_id << '\n';
            count++;
        }
    }
    idStream << "//---- " << MagicEndMarker << ' ' << _dlgName
             << " ids ------\n";
    idStream << "};\n";


}

void DialogEditor::DoDelete ()
{
    _store -= _selection;
    VObjSetIterator itr (_selection);
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        _usedIdSet.Remove (vObj->_id);
    }
    _selection.DestroyContents ();
    _client->Invalidate ();
}   


void DialogEditor::_SetSizes (UI_Dialog* sizeDlg)
{
    CL_Integer hId = ((CL_Integer&) ((*sizeDlg) [ID_HGROUP])->Model());
    CL_Integer vId = ((CL_Integer&) ((*sizeDlg) [ID_VGROUP])->Model());
    if (hId == ID_NOCHG_H && vId == ID_NOCHG_V)
        return;
    VObjSetIterator itr (_selection);
    long maxh = -1, minh = LONG_MAX, maxw = -1, minw = LONG_MAX;
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        minh = minl (minh, vObj->_shape.Height());
        maxh = maxl (maxh, vObj->_shape.Height());
        minw = minl (minw, vObj->_shape.Width());
        maxw = maxl (maxw, vObj->_shape.Width());
    }
    long newW = -1;
    switch (hId) {
    case ID_SHRINK_H:
        newW = minw;
        break;

    case ID_GROW_H:
        newW = maxw;
        break;
    }
    long newH = -1;
    switch (vId) {
    case ID_SHRINK_V:
        newH = minh;
        break;

    case ID_GROW_V:
        newH = maxh;
        break;
    }
    for (itr.Reset(); itr.More();) {
        VObjDesc* vObj = (VObjDesc*) itr.Next();
        if (newH != -1)
            vObj->_shape.SetHeight (newH);
        if (newW != -1)
            vObj->_shape.SetWidth (newW);
    }
}


void DialogEditor::DoSize ()
{
    extern UI_ViewDescriptor Size_dlgDesc[];
    extern UI_RectangleStruct Size_shape;
    extern UI_DialogEventDescriptor terminators[];
    UI_Dialog* sizeDlg = new UI_Dialog (_client, Size_dlgDesc, Size_shape,
                                        terminators, ID_OK);
    sizeDlg->Font()  = UI_FontDesc (UIFont_Helvetica, DialogFontSize,
                                    UIFont_BoldFace);
    (*sizeDlg)[ID_HGROUP]->Model() = CL_Integer (ID_NOCHG_H);
    (*sizeDlg)[ID_VGROUP]->Model() = CL_Integer (ID_NOCHG_V);
    sizeDlg->Title() = "Set Sizes";
    if (sizeDlg->ExecuteModal().id == ID_OK) {
        _SetSizes (sizeDlg);
        _client->Invalidate();
    }
    YACLApp()->Destroy (sizeDlg);
}



void DialogEditor::_SetAlignments (UI_Dialog* alignDlg)
{
    CL_Integer hId = (CL_Integer&) (*alignDlg)[ID_HGROUP]->Model();
    CL_Integer vId = (CL_Integer&) (*alignDlg)[ID_VGROUP]->Model();
    if (hId == ID_NOCHG_H && vId == ID_NOCHG_V)
        return;
    VObjSetIterator itr (_selection);
    long maxRight = -1, minLeft = LONG_MAX, hCenter;
    long maxBot   = -1, minTop  = LONG_MAX, vCenter;
    while (itr.More()) {
        VObjDesc* vObj = itr.Next();
        maxRight = maxl (maxRight, vObj->_shape.Right());
        minLeft  = minl (minLeft,  vObj->_shape.Left ());
        maxBot   = maxl (maxBot,   vObj->_shape.Bottom());
        minTop   = minl (minTop,   vObj->_shape.Top  ());
    }
    hCenter = (minLeft + maxRight)/2;
    vCenter = (minTop  + maxBot)/2;
    long newTop = -1, newLeft = -1, newRight = -1, newBot = -1,
        newVCenter = -1, newHCenter = -1;
    switch (hId) {
    case ID_LEFTALIGN:
        newLeft = minLeft;
        break;

    case ID_RIGHTALIGN:
        newRight = maxRight;
        break;

    case ID_HCENTERALIGN:
        newHCenter = hCenter;
        break;
    }

    switch (vId) {
    case ID_TOPALIGN:
        newTop  = minTop;
        break;

    case ID_BOTTOMALIGN:
        newBot   = maxBot;
        break;

    case ID_VCENTERALIGN:
        newVCenter = vCenter;
        break;
    }
    long myWidth = _client->Shape().Width(),
        myHeight = _client->Shape().Height();
    for (itr.Reset(); itr.More();) {
        VObjDesc* vObj = itr.Next();
        const UI_Point& p = vObj->_shape.Origin();
        if (newLeft != -1)
            vObj->_shape.Origin
                (UI_Point (newLeft, p.YCoord()));
        if (newRight != -1)
            vObj->_shape.Origin
                (UI_Point (newRight - vObj->_shape.Width() + 1,
                           p.YCoord()));
        if (newHCenter != -1)
            vObj->_shape.Origin
                (UI_Point (newHCenter - vObj->_shape.Width()/2 + 1,
                           p.YCoord()));
        if (newTop != -1)
            vObj->_shape.Origin (UI_Point (p.XCoord(), newTop));
        if (newBot != -1)
            vObj->_shape.Origin
                (UI_Point (p.XCoord(),
                           newBot - vObj->_shape.Height() + 1));
        if (newVCenter != -1)
            vObj->_shape.Origin
                (UI_Point (p.XCoord(),
                           newVCenter - vObj->_shape.Height()/2 + 1));
        if (hId == ID_HCENTERDLG)
            vObj->_shape += UI_Vector (myWidth/2 - hCenter, 0);
        if (vId == ID_VCENTERDLG)
            vObj->_shape += UI_Vector (0, myHeight/2 - vCenter);
    }
}

void DialogEditor::DoAlign ()
{
    extern UI_ViewDescriptor AlignDlgDesc[];
    extern UI_RectangleStruct AlignDlgShape;
    UI_Dialog* alignDlg = new UI_Dialog (_client, AlignDlgDesc, AlignDlgShape);
    alignDlg->Title() = "Align Objects";
    alignDlg->Font()  = UI_FontDesc (UIFont_Helvetica, DialogFontSize,
                                     UIFont_BoldFace);
    (*alignDlg)[ID_HGROUP]->Model() = CL_Integer (ID_NOCHG_H);
    (*alignDlg)[ID_VGROUP]->Model() = CL_Integer (ID_NOCHG_V);
    if (alignDlg->ExecuteModal().id == ID_OK) {
        _SetAlignments (alignDlg);
        _client->Invalidate ();
    }
    YACLApp()->Destroy (alignDlg);
}









CL_String DialogEditor::_BuildVObjDesc (const char* descriptorLine,
                                        VObjDesc& dsc)
{
    CL_String line (descriptorLine);
    line.Replace ("{", "");
    CL_String fields[8];
    long nFields = line.Split (fields, 8, " ,");
    if (fields[0] == "View_None")
        return "";
    if (nFields != 8)
        return "Invalid number of fields " + CL_String (nFields);
    fields[0].Replace ("View_", "");
    dsc._type = ViewTypeOf (fields[0]);
    long id = fields[1].AsLong();
    if (id) {
        dsc._id = id;
        dsc._symbolicName = "";
    }
    else  {
        dsc._id = _nameIdMap[fields[1]];
        dsc._symbolicName = fields[1];
        if (!dsc._id)
            return "Invalid symbolic name specified";
    }
    long x = fields[2].AsLong();
    long y = fields[3].AsLong();
    long w = fields[4].AsLong();
    long h = fields[5].AsLong();
    dsc._shape = UI_Rectangle (x, y, w, h);
    dsc._isTabStop = fields[6].InUpperCase() == "TRUE" ? TRUE : FALSE;
    short i;
    short n = fields[7].Size() - 1; 
    for (i = n; i >= 0; i--) {
        if (fields[7][i] == '"')
            break;
    }
    if (i < 0)
        return "No title specified";
    short j;
    for (j = 0; j <= n; j++) {
        if (fields[7][j] == '"')
            break;
    }
    if (j < 0)
        return "No title specified";
    dsc._title = fields[7] (j+1, i-j-1);
    return "";
}


void DialogEditor::_BuildIdMap (istream& stream)
{
    CL_String line;
    bool beginFound = FALSE;
    long lineNo = 0;
    while (line.ReadLine (stream)) {
        lineNo++;
        if (line.Index (MagicBeginMarker) >= 0) {
            beginFound = TRUE;
            break;
        }
    }
    if (!beginFound)
        return;
    while (line.ReadLine (stream)) {
        lineNo++;
        if (line.Index (MagicEndMarker) >= 0)
            return;
        CL_StringSequence sq = line.Split (" =,");
        if (sq.Size() == 2)
            _nameIdMap.Add (sq[0], sq[1].AsLong());
        else 
            UI_StandardDialog ("Invalid input in .ids file");
    }
}

void DialogEditor::_BuildVObjects (istream& stream)
{
    CL_String line;
    long lineNo = 0;
    do {
        bool beginFound = FALSE;
        while (line.ReadLine (stream)) {
            lineNo++;
            if (line.Index (MagicBeginMarker) >= 0) {
                beginFound = TRUE;
                break;
            }
        }
    if (!beginFound) break;
#if defined(_MSC_VER)
        // Here's another MSVC idiosyncracy: it doesn't do default
        // arguments properly. That's why we have this ifdef. Of course, I
        // could cow down to MSVC and use explicit arguments for all
        // compilers, but I don't want to do that!
        CL_String typeField = line.Field(4, " ");
#else
        CL_String typeField = line.Field(4);
#endif
        if (typeField == "shape") {
            line.ReadLine (stream);
            long x = line.Field (1, " ,").AsLong();
            long y = line.Field (2, " ,").AsLong();
            long w = line.Field (3, " ,").AsLong();
            long h = line.Field (4, " ,").AsLong();
            _client->Shape() = UI_Rectangle (x, y, w, h);
            continue; // skip the processing loop below
        }
        UI_Vector offsetVec (0, 0);
        if (typeField == "group") {
            // We have a button group
            long offsetX = line.Field (6, " ").AsLong();
            long offsetY = line.Field (7, " ").AsLong();
            offsetVec = UI_Vector (offsetX, offsetY);
        }
        while (line.ReadLine (stream)) {
            lineNo++;
            if (line.Index (MagicEndMarker) >= 0) {
#if defined(_MSC_VER)
                if (line.Field(4, " ") == "items") {
#else
                if (line.Field(4) == "items") {
#endif
                    _dlgName = line.Field (3);
                    _client->Title() = "YACL Dialog Editor: " + _dlgName;
                    return;
                }
                break;
            }
            VObjDesc* dsc = new VObjDesc;
            CL_String msg = _BuildVObjDesc (line, *dsc);
            if (msg.Size()) {
                msg = "Line " + CL_String (lineNo) + ": " + msg;
                UI_StandardDialog (msg, "Invalid input", _client);
                delete dsc;
                return;
            }
            if (dsc->_type != View_None) {
                if (_usedIdSet.Includes (dsc->_id)) {
                    CL_String msg = "Line " + CL_String (lineNo) + ": "
                        "Duplicate id " + CL_String (dsc->_id);
                    UI_StandardDialog (msg, "Warning", _client);
                }
                _usedIdSet.Add (dsc->_id);
                dsc->_shape += offsetVec; // Adjust button groups
                _store.Add (dsc);
            }
        }
    } while (1);
}


void DialogEditor::DoOpen ()
{
    CL_String fileName = UI_FileSelectDialog (_client, _currentWorkingDir);
    if (fileName.Size() <= 0)
        return;
    ifstream itemStream (fileName);
    if (itemStream.bad()) {
        CL_String msg = "Cannot open file " + fileName;
        UI_StandardDialog (msg, "Open failure", _client);
        return;
    }
    _currentWorkingDir = CL_PathName (fileName).DirectoryName();
    ifstream idStream (_IdFileName (fileName).AsPtr());
    if (idStream.good())
        _BuildIdMap (idStream);
    _store.DestroyContents();
    _selection.MakeEmpty();
    _usedIdSet.MakeEmpty();
    _BuildVObjects (itemStream);
    _client->Invalidate ();
}



void DialogEditor::DoAbout ()
{
    extern UI_ViewDescriptor About_dlgDesc[];
    extern UI_RectangleStruct About_shape;
    extern UI_DialogEventDescriptor terminators[];
    UI_Dialog* about = new UI_Dialog
        (YACLApp()->MainWindow(), About_dlgDesc, About_shape, terminators,
         ID_OK);
    about->Title() = "About Dlged";
    ((UI_Label*) (*about)[ID_EDITORTITLE])->SetTextStyle (UIText_Center);
    UI_Label* yaclVer = (UI_Label*) (*about)[ID_YACLVER];
    yaclVer->SetTextStyle (UIText_Center);
    yaclVer->Model() = "YACL version " + CL_String (YACLVersion());
    UI_Label* editVer = ((UI_Label*) (*about)[ID_EDITVER]);
    editVer->SetTextStyle (UIText_Center);
    editVer->Model() = "Version " + CL_String (EDITOR_VERSION);
    about->Font() = UI_FontDesc (UIFont_Helvetica, DialogFontSize,
                                 UIFont_BoldFace);
    about->ExecuteModal();
    YACLApp()->Destroy (about);
}



void DialogEditor::DoFontChoice ()
{
    if (UI_FontDialog (_client, _chosenFont)) {
        _client->Font() = _chosenFont;
        _client->Invalidate ();
    }
}

