// File: MenuDriver.c++

#include "MenuDriver.h"

#include <ui/menu.h>
#include <ui/ApplicationWithAccel.h>
#include <ui/ControllerWithAccel.h>
#include <ui/stddlg.h>

typedef UI_EventBinding2 <MenuDriver> MenuBinding;
#if defined(__GNUC__)
template class UI_EventBinding2 <MenuDriver>; // Instantiate the binding
#endif

//////////////////////////////////////////////////////////////////////////
// BEGIN Menus

UI_MenuItemDescriptor FileMenu [] = {      // FILE pull-down menu
    { "&Open\tCtrl+O"     , IDM_FILE_OPEN   }
  , { "&Save\tCtrl+S"     , IDM_FILE_SAVE   }
  , { "Save &as..."        , IDM_FILE_SAVEAS }
  , { "&Print...\tCtrl+P" , IDM_FILE_PRINT  }
  , { UIMenu_Separator     , NULL            }
  , { "E&xit\tF3"         , IDM_FILE_EXIT   }
  , { NULL                 , 0               }
};

UI_MenuItemDescriptor EditMenu [] = {     // EDIT pull-down menu
    { "Cu&t\tShift+Del"   , IDM_EDIT_CUT }
  , { "&Copy\tCtrl+Ins"   , IDM_EDIT_COPY }
  , { "&Paste\tShift+Ins" , IDM_EDIT_PASTE }
  , {NULL             , 0       }
};

UI_MenuItemDescriptor MainMenuDesc [] = {   // Main menu bar
    { "&File", IDM_FILE,   FileMenu }
  , { "&Edit", IDM_EDIT,   EditMenu}
  , { "&Help", IDM_HELP,   NULL}
  , { NULL      ,         0,   NULL     }
};

struct DispatcherStruct {
   UI_ViewID              menu_id;
   MenuBinding::MethodPtr method;
   UI_EventType           event;
} FAR dispatch [] = {
    { IDM_FILE_OPEN,    MenuDriver::OnFileOpen,    Event_Select }
  , { IDM_FILE_SAVE,    MenuDriver::OnFileSave,    Event_Select }
  , { IDM_FILE_SAVEAS,   MenuDriver::OnFileSaveAs,   Event_Select }
  , { IDM_FILE_PRINT,   MenuDriver::OnFilePrint,   Event_Select }
  , { IDM_FILE_EXIT,    MenuDriver::OnFileExit,    Event_Select }
  , { IDM_EDIT_CUT,      MenuDriver::OnEditCut,      Event_Select }
  , { IDM_EDIT_COPY,    MenuDriver::OnEditCopy,    Event_Select }
  , { IDM_EDIT_PASTE,   MenuDriver::OnEditPaste,   Event_Select }
  , { IDM_HELP,         MenuDriver::OnHelp,         Event_Select }
};


// END Menus
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// BEGIN Accelerators

UI_AcceleratorTable accelDesc[] =
{
   { 'o' | Key_Ctrl,                        IDM_FILE_OPEN,        Event_Select }
 , { 's' | Key_Ctrl,                        IDM_FILE_SAVE,        Event_Select }
 , { 'p' | Key_Ctrl,                        IDM_FILE_PRINT,      Event_Select }
 , { VKey_F(3) | Key_Virtual,               IDM_FILE_EXIT,        Event_Select }
 , { VKey_Delete | Key_Virtual | Key_Shift, IDM_EDIT_CUT,        Event_Select }
 , { VKey_Insert | Key_Virtual | Key_Ctrl,  IDM_EDIT_COPY,        Event_Select }
 , { VKey_Insert | Key_Virtual | Key_Shift, IDM_EDIT_PASTE,      Event_Select }
 , { VKey_F(1) | Key_Virtual,               IDM_HELP,            Event_Select }
 , { 0, 0, Event_None }
};

// END Accelerators
//////////////////////////////////////////////////////////////////////////


MenuDriver::MenuDriver (UI_CompositeVObject* root)
{
   _menu =    new UI_MenuBar (root, MainMenuDesc);

   // Create the menus; see the <uidemo/menu0> sample program
   for (short i = 0; i < sizeof dispatch/sizeof (DispatcherStruct); i++)
   {
      UI_ViewID id = dispatch[i].menu_id;
      MenuBinding bind (this, dispatch[i].method, id);
      (*_menu)[id]->AddEventDependent (dispatch[i].event, bind);
   }

   // Add the accelerators; this is new
   UI_ControllerWithAccel *theController;
   theController = (UI_ControllerWithAccel *) &(YACLApp()->Controller());
   theController->AddAcceleratorTable(accelDesc, *_menu);

   // Start some menu items in disabled state
   (*_menu)[IDM_FILE_SAVE]->Disable();
   (*_menu)[IDM_FILE_SAVEAS]->Disable();
   (*_menu)[IDM_FILE_PRINT]->Disable();
   (*_menu)[IDM_EDIT_CUT]->Disable();
   (*_menu)[IDM_EDIT_COPY]->Disable();
   (*_menu)[IDM_EDIT_PASTE]->Disable();

   _fileSaved = TRUE;  // We can quit without confirmation
}


bool MenuDriver::OnFileOpen(UI_Event& e, UI_ViewID id)
// Call the File Open dialogue box, then enable some menu items
{
#ifdef TRACE
   CL_Error::Warning("OnFileOpen()");
#endif
   CL_String fileName = UI_FileSelectDialog(
      (UI_VisualObject *) YACLApp()->MainWindow());

   if(fileName.Length())
   {
      #ifdef TRACE
      CL_Error::Warning("You chose %s", (const char *) fileName);
      #endif

      (*_menu)[IDM_FILE_SAVE]->Enable();
      (*_menu)[IDM_FILE_SAVEAS]->Enable();
      (*_menu)[IDM_FILE_PRINT]->Enable();
      (*_menu)[IDM_EDIT_CUT]->Enable();
      (*_menu)[IDM_EDIT_COPY]->Enable();
      (*_menu)[IDM_EDIT_PASTE]->Enable();
   }
   return TRUE;
}


bool MenuDriver::OnFileSave(UI_Event& e, UI_ViewID id)
{
#ifdef TRACE
   CL_Error::Warning("OnFileSave()");
#endif
   _fileSaved = TRUE;
   return TRUE;
}

bool MenuDriver::OnFileSaveAs(UI_Event& e, UI_ViewID id)
// Call the File Save As dialogue box
{
#ifdef TRACE
   CL_Error::Warning("OnFileSaveAs()");
#endif

   CL_String fileName = UI_FileSelectDialog(
      (UI_VisualObject *) YACLApp()->MainWindow(), "",
      "Save file as...");
#ifdef TRACE
   if(fileName.Length())
      CL_Error::Warning("You chose %s", (const char *) fileName);
#endif
   _fileSaved = TRUE;
   return TRUE;
}

bool MenuDriver::OnFilePrint(UI_Event& e, UI_ViewID id)
{
#ifdef TRACE
   CL_Error::Warning("OnFilePrint()");
#endif
}


bool MenuDriver::OnFileExit(UI_Event& e, UI_ViewID id)
// Ask before quitting if _fileSaved is FALSE
{
#ifdef TRACE
   CL_Error::Warning("OnFileExit()");
#endif
   if(_fileSaved)
   {
      YACLApp()->End();
      return TRUE;   // No confirmation
   }

   UI_ViewID response = UI_SimpleDialog(
      "The file has not been saved; are you sure you want to quit ?",
      "Accelerator sample application",
      (UI_VisualObject *) YACLApp()->MainWindow(), UIS_YesNoCancel, UIS_Question);

   switch(response)
   {
      case UI_IDYES : OnFileSave(e, id);
      case UI_IDNO : YACLApp()->End(); break;
      case UI_IDCANCEL : break;    // Do nothing
   }
   return TRUE;
}


bool MenuDriver::OnEditCut(UI_Event& e, UI_ViewID id)
{
#ifdef TRACE
   CL_Error::Warning("OnEditCut()");
#endif
   _fileSaved = FALSE;
   return TRUE;
}

bool MenuDriver::OnEditCopy(UI_Event& e, UI_ViewID id)
{
#ifdef TRACE
   CL_Error::Warning("OnEditCopy()");
#endif
   return TRUE;
}

bool MenuDriver::OnEditPaste(UI_Event& e, UI_ViewID id)
{
#ifdef TRACE
   CL_Error::Warning("OnEditPaste()");
#endif
   _fileSaved = FALSE;
   return TRUE;
}

bool MenuDriver::OnHelp(UI_Event& e, UI_ViewID id)
{
   UI_SimpleDialog(
      "Accelerator sample application\n"
      "Copyright (c) 1996 Ludovic Brenta\n\n"
      "YACL Version 1.4.0 (c) 1994, 1995 M.A. Sridhar",

      "About the Accelerator sample...",

      (UI_VisualObject *) YACLApp()->MainWindow());

   return TRUE;
}

// EOF

