

/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */


// Define the following symbol if you want a 3dLabel to use a Bitmap drawn
// with an in-memory display surface.

#if !defined(__OS2__)
// I don't use the memory DS under OS/2 because the result looks ugly,
// since OS/2 seems to scale fonts and draw an extra black line at the
// top. I think these are gaps in my understanding of the memory DS, but
// until they're resolved.....
#define USE_MEMORY_DS
#endif



#include "ui/3dlabel.h"
#include "ui/dsplsurf.h"
#include "ui/shadorec.h"
#include "ui/vobjcoll.h"
#include "ui/dsinmem.h"
#include "ui/bitmap.h"


#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__) || defined(__OS2__)
static char ThreeDLabelClass[] = "YACL_3D_Label";
#endif


inline short UI_3DLabel::_EmbossWidth () const
{
    return 2 * _shadowThickness + 2;
}


typedef UI_EventBinding1<UI_3DLabel> Bind;

#if defined(__GNUC__)
template class UI_EventBinding1<UI_3DLabel>;
#endif

UI_3DLabel::UI_3DLabel (UI_VisualObject* parent, 
                        const UI_Rectangle& r, UI_ViewID id,
                        LabelStyle style, short thickness)
: UI_Label (parent, r, id), _style (style), _shadowThickness (thickness)
{
    Bind bind (this, &UI_3DLabel::_PaintEvent);
    parent->AddEventDependent (Event_Paint, bind);
    _bMap = 0;
}



UI_3DLabel::UI_3DLabel (UI_VisualObject* parent, CL_String* model,
                        const UI_Rectangle& r, UI_ViewID id,
                        LabelStyle style)
: UI_Label (parent, model, r, id), _style (style)
{
    Bind bind (this, &UI_3DLabel::_PaintEvent);
    parent->AddEventDependent (Event_Paint, bind);
    _bMap = 0;
}


UI_WindowClass UI_3DLabel::WindowClass () const
{
#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__) || defined(__OS2__)
    return ThreeDLabelClass;
#else
    // NotImplemented ("WindowClass");
    return (UI_WindowClass) 0;
#endif
}


bool UI_3DLabel::MakeVisualElement ()
{
    _created = TRUE;
    return FALSE; // No visual element
}


bool UI_3DLabel::DestroyVisualElement ()
{
    if (_bMap) {
        delete _bMap;
        _bMap = 0;
    }
    return TRUE;
}


void UI_3DLabel::_RebuildBitmap ()
{
    if (_bMap)
        delete _bMap;
    UI_DwgSurfaceInMemory ds (_shape.Width(), _shape.Height());
    _DoPaint (ds);
    _bMap = new UI_Bitmap;
    _bMap->CopyFrom (ds, UI_Rectangle (0, 0, _shape.Width(), _shape.Height()));
}


bool UI_3DLabel::_PaintEvent (UI_Event& e)
{
    if (!IsVisible() ||
        e.Origin() != Parent() || e.Destination() != Parent())
        // Only respond to paint events to the parent, not to any of its
        // children. Otherwise the parent will get flooded with Paint events
        // under X windows.
        return FALSE;
    const UI_Rectangle& invalid = e.Shape();
    if (!_shape.IntersectsBoundary (invalid) &&
        !_shape.IsContainedIn (invalid))
        return FALSE;
#if defined(USE_MEMORY_DS)
    if (!_bMap)
        _RebuildBitmap ();
#endif
    _Redraw ();
    return TRUE; // Golden rule: event dependent methods must ALWAYS return
                 // TRUE, otherwise other event dependents will not be
                 // called!
}

void UI_3DLabel::_Redraw ()
{
#if defined(USE_MEMORY_DS)
    if (!_bMap)
        return;
#endif
    UI_DisplaySurface* sfc = _parent->DisplaySurface();
    if (sfc) {
#if defined(USE_MEMORY_DS)
        _bMap->DrawOn (*sfc, _shape.Origin());
#else
       sfc->SaveState();
       _DoPaint (*sfc);
       sfc->RestoreState();
#endif
    }
    else {
        UI_DisplaySurface& sfc = _parent->CreateDisplaySurface();
#if defined(USE_MEMORY_DS)
        _bMap->DrawOn (sfc, _shape.Origin());
#else
        _DoPaint (sfc);
#endif
        _parent->DestroyDisplaySurface();
    }
}


bool UI_3DLabel::_TitleChanged ()
{
    if (!_created)
        return FALSE;
#if !defined(USE_MEMORY_DS)
    UI_DisplaySurface* sfc = _parent->DisplaySurface();
    if (sfc) {
        sfc->SaveState();
        sfc->Mode (UI_DisplaySurface::GMode_Copy);
        // Draw the new title
        _DrawText(*sfc);
        sfc->RestoreState();
    }
    else {
        UI_DisplaySurface& sfc = _parent->CreateDisplaySurface();
        _DrawText(sfc);
        _parent->DestroyDisplaySurface();
    }
#else // Use in-memory DS
    _RebuildBitmap ();
    _Redraw ();
#endif
    return FALSE;
}


bool UI_3DLabel::_ShapeRectChanged ()
{
#if defined(USE_MEMORY_DS)
    _RebuildBitmap ();
#endif
    _Redraw ();
    return FALSE;
}

void UI_3DLabel::_DoPaint (UI_DrawingSurface& sfc)
{
    // Redraw this label
    switch (_style) {
    case Recessed: {
#if defined(USE_MEMORY_DS)
        UI_ShadowRectangle r (0, 0, _shape.Width(), _shape.Height(),
                              UI_ShadowRectangle::Recessed,
                              _shadowThickness);
#else
        UI_ShadowRectangle r (_shape, UI_ShadowRectangle::Recessed);
#endif
        r.DrawOn (sfc);
        sfc.DrawRectangle (_shape, UID_Outline);
        break;
    }

    case Embossed: {
        short ew = _EmbossWidth ();
        long w   = _shape.Width();
        long h   = _shape.Height ();
#if defined(USE_MEMORY_DS)
        UI_ShadowRectangle r1 (0, 0, w, h, UI_ShadowRectangle::Raised,
                               _shadowThickness);
        UI_ShadowRectangle r2 (ew, ew, w - 2 * ew, h - 2 * ew,
                               UI_ShadowRectangle::Recessed,
                               _shadowThickness);
#else
        UI_ShadowRectangle r1 (_shape, UI_ShadowRectangle::Raised,
                               _shadowThickness);
        UI_ShadowRectangle r2 (_shape.Left() + ew,
                               _shape.Top () + ew,
                               w - 2 * ew, h - 2 * ew,
                               UI_ShadowRectangle::Recessed,
                               _shadowThickness);
#endif
        r1.DrawOn (sfc);
        r2.DrawOn (sfc);
        break;
    }

    case Raised: {
#if defined(USE_MEMORY_DS)
        UI_ShadowRectangle r (0, 0, _shape.Width(),
                              _shape.Height(),
                              UI_ShadowRectangle::Raised,
                              _shadowThickness);
#else
        UI_ShadowRectangle r (_shape, UI_ShadowRectangle::Raised);
#endif
        r.DrawOn (sfc);
        break;
    }

    case Plain:
    default: {
        sfc.Brush().Color (UIColor_MediumGray);
        UID_DrawOptions opt = (UID_DrawOptions)
            (BorderShown() ? (UID_Fill | UID_Outline) : UID_Fill);
#if defined(USE_MEMORY_DS)
        UI_Rectangle r (0, 0, _shape.Width(), _shape.Height());
        sfc.DrawRectangle (r, opt);
#else
        sfc.DrawRectangle (_shape, opt);
#endif
        break;
    }
    }
    _DrawText(sfc);
}


void UI_3DLabel::_DrawText (UI_DrawingSurface& sfc)
{
    short ew = _EmbossWidth ();
#if defined(USE_MEMORY_DS)
    UI_Rectangle textRect (2 * ew, 2 * ew + 2,
                           _shape.Width() - 4 * ew,
                           _shape.Height() - 4 * ew);
#else
    UI_Rectangle textRect (2 * ew + _shape.Left(),
                           2 * ew + _shape.Top(),
                           _shape.Width() - 4 * ew,
                           _shape.Height() - 4 * ew);
#endif
    // Erase the old title, if any
    sfc.Brush().Color (UIColor_MediumGray);
    sfc.DrawRectangle (textRect, UID_Fill);

    const char* data = _title.AsPtr();
    UI_Pen& pen = sfc.Pen();
    pen.Color (UIColor_Black);
    pen.Pattern (UIPen_Solid);
#if defined(USE_MEMORY_DS)
    UI_Font& fnt = Font();
    sfc.Font() = UI_FontDesc (fnt.TypeFace(), fnt.PointSize(), fnt.Style());
#endif
    sfc.WriteString (data, textRect, _textStyle);
}



void UI_3DLabel::MakeVisible()
{
    if (IsVisible())
        return;
    UI_VisualObject::MakeVisible();
    _parent->Invalidate (Shape());
}

void UI_3DLabel::MakeInvisible()
{
    if (!IsVisible())
        return;
    UI_VisualObject::MakeInvisible();
    _parent->Invalidate (Shape());
}



bool UI_3DLabel::_DoShowBorder (bool show)
{
    UI_Label::_DoShowBorder (show);
    if (_style != Plain)
        return FALSE;
    UI_ColorScheme clr = show ? UIColor_Black : UIColor_MediumGray;
    
    UI_DisplaySurface* sfc = _parent->DisplaySurface();
    if (sfc) {
        sfc->SaveState();
        sfc->Pen().Color (clr);
        sfc->DrawRectangle (Shape());
        sfc->RestoreState();
    }
    else {
        UI_DisplaySurface& sfc = _parent->CreateDisplaySurface();
        sfc.Pen().Color (clr);
        sfc.DrawRectangle (Shape());
        _parent->DestroyDisplaySurface();
    }
    return TRUE;
}


