
#ifndef _color_h_
#define _color_h_





/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



/* Authors:   M. A. Sridhar
              N. Bhowmik
   Modified:  R. Sampath
*/

// This class represents a single color value represented as a (red, green,
// blue) triple. Its instance is constructed by specifying red, green and
// blue values as {\tt double} values in the range~0 to~1. Thus {\tt
// UI_Color (1.0, 1.0, 1.0)} represents white. These values are scaled to
// the platform-specific maximum values (255 under MS Windows, 65535 under X
// windows) internally by this class.


#if defined(__GNUC__)
#pragma interface
#endif

#include "ui/uidefs.h"     // Include this first.

#include "base/object.h"
#include "ui/classid.h"

//[ Global
enum UI_ColorScheme {
    UIColor_Black, UIColor_Green,       UIColor_Blue,       UIColor_Red,
    UIColor_White, UIColor_LightGray,   UIColor_MediumGray, UIColor_DarkGray
};
//] Global

#if defined (__MS_WINDOWS__) || defined(__MS_WIN32__)
#   include <windows.h>
    enum {
        UI_MAXCOLORS = 255
    };

    typedef COLORREF UI_NativeColorRep;
    // Native platform's representation of color: a COLORREF under MS/Windows,
    // and an XColor under X Windows.
#elif defined (__OS2__)
    enum {
        UI_MAXCOLORS = 255
    };

    typedef long     UI_NativeColorRep;

#elif defined (__X_MOTIF__)  || defined(__X_YACL__)
#   include <X11/Xlib.h>
    enum {
        UI_MAXCOLORS = 65535
    };

    typedef XColor UI_NativeColorRep;
#endif


class YACL_UI UI_Color: public CL_Object {

public:
    UI_Color ();
    // Default constuctor: construct a black color.

    UI_Color (UI_ColorScheme c);
    // Construct the color defined by the given ColorScheme.

    UI_Color (double red, double green, double blue);
    // All parameters should be between 0 and 1. This constructor is
    // platform-independent. All  three zeros corresponds to black, and all
    // 1's to white.
    
    UI_Color (const UI_Color&);
    // Copy constructor.

    ~UI_Color () {};
    // Destructor: does nothing. Provided because the parent class'
    // destructor is pure virtual.
    
    void Red (double red);
    // Set the red intensity of this color. The least amount of red is 0,
    // and the highest is 1.

    double Red() const;
    // Return the red intensity of this color.

    void Green (double green) ;
    // Set the green intensity of this color. The least amount of green is 0,
    // and the highest is 1.

    double Green() const;
    // Return the green intensity of this color.

    void Blue (double blue) ;
    // Set the blue intensity of this color. The least amount of blue is 0,
    // and the highest is 1.

    double Blue() const;
    // Return the blue intensity of this color.

    UI_Color LightShadow () const;
    // Assuming  this color were the background color, return the light
    // shadow needed for a 3D effect.
    
    UI_Color DarkShadow () const;
    // Assuming  this color were the background color, return the dark
    // shadow needed for a 3D effect.
    
    UI_NativeColorRep NativeForm () const;
    // [For YACL internal use only.]
    
    bool operator==  (const UI_Color& x) const;
    // Return TRUE if this color has the same RGB values as {\tt x}.

    bool operator==  (const CL_Object& x) const
        {return x.IsA (*this) && *this == ( (const UI_Color &) x); };

    void operator=   (const CL_Object& x)
        {if (x.IsA (*this)) *this = ((const UI_Color&) x);};

    bool operator < (const CL_Object& x) const
        {return x.IsA (*this) && *this < ( (const UI_Color &) x); };

    bool operator < (const UI_Color& c) const;
    
    UI_Color& operator= (const UI_Color& color);

    const char* ClassName () const { return "UI_Color";};

    CL_ClassId ClassId () const {return _UI_Color_CLASSID;}
    
protected:

    UI_Color (long r, long g, long b);
    // For internal use only.
         
    static UI_NativeColorRep _ColorVal (long, long, long);

    long _red, _blue, _green;

    UI_NativeColorRep _colorVal;

#if defined(__X_MOTIF__)  || defined(__X_YACL__)
    ulong _xPixel;

public:
    ulong XPixel () const {return _xPixel;};
    // [X-windows-specific; YACL internal use only] return the pixel value
    // of this color in the default colormap.

#endif
 
};

inline double UI_Color::Red() const
{
    return _red / ((double) UI_MAXCOLORS);
}


inline double UI_Color::Green() const
{
    return _green / ((double) UI_MAXCOLORS);
}



inline double UI_Color::Blue() const
{
    return _blue / ((double) UI_MAXCOLORS);
}


#endif
