

#ifndef _dwgsurf_h_ /* Thu Sep 29 11:34:19 1994 */
#define _dwgsurf_h_




/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



// A DrawingSurface is an abstract class representing a surface that offers
// drawing capabilities. Classes derived from DrawingSurface specialize it
// for various devices.
//
// DrawingSurface includes three instance variables {\tt _pen}, {\tt
// _brush} and {\tt _font} that it initializes to NULL but dereferences in
// the {\tt Pen}, {\tt Brush} and {\tt Font} methods respectively. Each
// derived class is responsible for setting these instance variables.
// Similarly, the derived class must set up the {\tt _horzPixels} and {\tt
// _vertPixels} instance variables.
//
// The implementations of the non-pure-virtual methods include the platform
// API calls for drawing operations. Most drawing methods, however, are
// virtual in order to accommodate devices not yet included in YACL.

// Author: M. A. Sridhar


#if defined(__GNUC__)
#pragma interface
#endif


#include "base/object.h"
#include "base/genseq.h"

#include "ui/rectangl.h"
#include "ui/font.h"
#include "ui/pen.h"
#include "ui/brush.h"
#include "ui/uidefs.h"

// #include "ui/colormap.h"

class YACL_UI UI_VisualObject;
class YACL_UI UI_Bitmap;
class YACL_UI UI_Arc;
class YACL_UI UI_Chord;
class YACL_UI UI_PieWedge;


enum UID_DrawOptions {UID_Outline = 1, UID_Fill = 2};

class YACL_UI UI_DrawingSurface: public CL_Object {

public:

    UI_DrawingSurface ();
    // Default constructor: initializes instance variables.

    ~UI_DrawingSurface () {};

    // 
    // --------------------------- Query ---------------------------------
    //     

    virtual double HorzPixelsPerMM () const = 0;
    // [Pure virtual]
    // Return the number of pixels per millimeter in the horizontal
    // direction.

    virtual double VertPixelsPerMM () const = 0;
    // [Pure virtual]
    // Return the number of pixels per millimeter in the vertical direction.

    virtual UI_Rectangle DrawingArea() const = 0;
    // [Pure virtual]
    // Returns the rectangle enclosing the drawing surface, in pixels. The
    // origin is (0, 0).

    virtual UI_Rectangle DrawingAreaInMM () const = 0;
    // [Pure virtual]
    // Returns the drawing area, in millimeters. The origin is (0, 0).

    virtual bool SupportsColor () const {return TRUE;};
    // Return TRUE if this DrawingSurface supports color, FALSE otherwise.
    // The default implementation returns TRUE unconditionally.
    
    // Pen, Font and Brush  are resources
    // maintained by this object. These may be accessed and modified 
    // via the following reference methods.

    virtual UI_Font& Font();
    // Return (a reference to) the font used by this surface. The return
    // value can be modified by the Font methods.

    virtual UI_Pen&  Pen();
    // Return (a reference to) the pen used by this surface. The return
    // value can be modified by the Pen methods.


    virtual UI_Brush& Brush();
    // Return (a reference to) the background brush used by this surface.
    // The return value can be modified by the Brush methods.

    
    // 
    // -------------------------- Draw  operations ------------------------
    // 


    virtual bool DrawLine (const UI_Point& p, const UI_Point& q);
    // Draw a line from {\tt p} to {\tt q}, using our pen.

    virtual void ColorDisplay (const UI_Color& c);
    // Paint client rectangle with specified color.

    virtual void ColorRectangle (const UI_Rectangle& r, const UI_Color& cs);
    // Convenience function to color the interior of the given rectangle.
    
    virtual bool InvertRectangle (const UI_Rectangle&);
    // Reverse the foreground and background in the given rectangle. Return
    // TRUE on success, FALSE on failure.

    virtual void DrawRectangle (const UI_Rectangle& rect,
                                ushort opt = UID_Outline); 
    // Draw a rectangle using the current pen. The second parameter
    // specifies the outline and fill options.
                
    virtual void DrawEllipse (const UI_Rectangle& rect,
                              ushort opt = UID_Outline); 
    // Draw an ellipse with given bounding rectangle, using the current pen.
    // The second parameter specifies the outline and fill options.
    // (If the "rectangle" happens to be a square, then a circle is drawn.)
                
    virtual void DrawArc (const UI_Arc &arc);
    // Draw an arc with the using the parameters given to UI_Arc.

    virtual void DrawChord (const UI_Chord& chord,
                            ushort opt = UID_Outline); 
    // Draw a chord using the current pen. The second parameter
    // specifies the outline and fill options.

    virtual void DrawPieWedge (const UI_PieWedge& pie,
                               ushort opt = UID_Outline); 
    // Draw a pie wedge using the current pen. The second parameter
    // specifies the outline and fill options.
        
    virtual void DrawPolygon (UI_Point pointArray[], short numPoints,
                              ushort opt = UID_Outline);
    // Draw a closed polygon by connecting the points in the array with
    // line segments, in the order of occurrence of the points.  The second
    // parameter specifies the outline and fill options.
    // The first point in the array is taken as the starting point.

    virtual void DrawPolyLine (UI_Point pointArray[], short numPoints);
    // Draw a figure by connecting the points in the array with
    // line segments, in the order of occurrence of the points. 
    // The first point in the array is taken as the starting point.

    virtual void DrawPoint (const UI_Point& p);
    // Draw a point at position {\tt p} with the brush's color.

    virtual bool DrawBitmap (const UI_Bitmap& b, const UI_Point& topLeft);
    // Draw the given bitmap on this surface.

    // virtual bool BeginMaskMode (short w, short h, uchar* mask);
    // Begin a masking mode. The mask is interpreted as a binary array of
    // pixels of width w and height h.  With this mask set up, the drawing
    // surface is thought of as tiled with copies of this mask. All
    // subsequent drawing operations (until a subsequent call to the
    // EndMaskMode method) are such that they will only affect those pixels
    // that have a 1 in the corresponding (tiled) mask. Pixels that
    // correspond to 0 in the mask are unaffected.
    //
    // Return TRUE if already in mask mode, FALSE otherwise.

    // virtual bool EndMaskMode ();
    // Terminate the masking mode begun by {\tt BeginMaskMode}. Return TRUE
    // if currently in mask mode, FALSE otherwise.
    
    virtual bool DrawMaskedBitmap (const UI_Bitmap& b, uchar* andMask,
                                   const UI_Point& topLeft);
    // Draw the given masked bitmap on this surface.
    
    // 
    // ------------------------ Text operations -------------------------
    // 

    virtual void WriteString (const char* str, const UI_Rectangle& box,
                              UI_TextStyle style = UIText_Left);
    // Write the given text string within the given box, in the given
    // style. This method is guaranteed to handle a single line, not
    // necessarily multiple lines. It draws the text within the rectangle,
    // clipping the text if necessary.

    
    virtual short TextWidth (const char* s) const;
    // Return the width (in pixels) of the text string s if it were drawn
    // on this DisplaySurface using its current font.

    // ------------------------ Saving and restoring ---------------------
    virtual bool SaveState();
    // Save the current state of the DrawingSurface (i.e., its drawing
    // attributes, pen, brush, font etc.) on an internal stack. A subsequent
    // call to {\tt RestoreState} restores the most recently saved state.
    //
    // The method returns TRUE on success, FALSE on failure.

    virtual bool RestoreState ();
    // Restore the state most recently saved via the {\tt SaveState} method.
    // The return value is TRUE on success, FALSE on failure (e.g., if the
    // state stack was already empty).

    // Basic methods:
    
    const char* ClassName () const {return "UI_DrawingSurface";};

    // -------------------- End public protocol -------------------------

    
protected:
    UI_Pen*            _pen;       // _handle, _pen, _brush and _font  must
    UI_Font*           _font;      // be set up by the derived class.
    UI_Brush*          _brush;
    UI_DwgSurfHandle   _handle;
    long               _horzPixels;
    long               _vertPixels;
    CL_GenericSequence _stateStack;

    virtual bool _FontWillChange ();
    // [YACL internal use only] called by our Font instance to let us know
    // that it will change.

    virtual bool _FontChanged    ();
    // [YACL internal use only] called by our Font instance to let us know
    // that it was changed.


#if defined(__MS_WINDOWS__) || defined(__MS_WIN32__) || defined(__OS2__)
    long _BitBltROP () const;
    // [Windows/OS2-specific, YACL internal use only]
    
#endif


#if defined (__X_MOTIF__)  || defined(__X_YACL__)
public:
    Pixmap _maskPixmap;
    
    virtual Drawable _Drawable() const = 0;
    // [X-Windows specific; internal use only] Return the drawable
    // associated with this surface. Most drawing methods use {\tt
    // _Drawable} before drawing.

protected:
    void _DrawFontLine (const UI_Point& p1, const UI_Point& p2);
    // [X-Windows specific; internal use only] Used for supporting
    // underlined and struck-out text.

#endif
public:

    UI_DwgSurfHandle Handle() const;
    // [For YACL internal use only] Return the platform-specific handle for
    // this DrawingSurface.

};



inline UI_DwgSurfHandle UI_DrawingSurface::Handle () const
{
    return _handle;
}


inline UI_Font& UI_DrawingSurface::Font ()
{
    return *_font;
}

inline UI_Pen& UI_DrawingSurface::Pen ()
{
    return *_pen;
}

inline UI_Brush& UI_DrawingSurface::Brush ()
{
    return *_brush;
}


 
#endif /* _dwgsurf_h_ */
