

#ifndef _evtbind_h_ /* Mon Apr 17 16:15:38 1995 */
#define _evtbind_h_



/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



// An event binding is a binding whose client method expects a reference to
// an event as parameter. Event bindings are used by the event dependent
// methods of VisualObject and its derived classes.
//
// AbstractEventBinding is an abstract class that
// serves as base for all kinds of event bindings. It is derived from
// AbstractBinding, and overrides the inherited {\tt Execute} method with
// one that casts its parameter down to an Event and calls the protected
// pure virtual method {\tt ExecuteEvtBind}. The event binding classes
// derived from AbstractEventBinding override {\tt ExecuteEvtBind} according
// to their specifications.


#include "base/binding.h"
#include "ui/event.h"
#include "ui/classid.h"

class YACL_UI UI_AbstractEventBinding: public CL_AbstractBinding {

public:
    bool Execute (CL_Object& o) const {return ExecuteEvtBind ((UI_Event&) o);};
    // Execute the binding, i.e., invoke the client method and return its
    // return value. This method casts the parameter {\tt o} down to a {\tt
    // UI_Event&} and calls the pure virtual {\tt ExecuteEvtBind}.
    
protected:
    virtual bool ExecuteEvtBind (UI_Event& e) const = 0;
    // [Pure virtual] Invoke the client method.

};


// EventBinding0 is a class template with parameter {\tt Client}. It is derived
// from AbstractEventBinding and represents  an event binding whose client
// is of type {\tt Client} and whose client method takes no parameters. It
// overrides the inherited protected method {\tt ExecuteEvtBind} with an
// implementation that invokes the client method.

template <class Client>
class YACL_UI UI_EventBinding0: public UI_AbstractEventBinding {

public:
    //[ Local
    typedef bool (Client::*MethodPtr) ();
    // The type of the client method for this binding.
    //] Local
    
    UI_EventBinding0 (Client* client, MethodPtr method)
        {_client = client;  _method = method;};
    // Construct an event binding for an object of type {\tt Client}, that
    // has an event-handling method {\tt method}.

    UI_EventBinding0 () {_client = 0; _method = 0;};
    // Construct a null binding, one whose object and method pointers are
    // both NULL, for an object of type Client.

    UI_EventBinding0 (const UI_EventBinding0<Client>& b)
        {*this = b;};
    // Copy constructor.

    ~UI_EventBinding0 () {};
    // Destructor.
    
    bool Valid () const { return _client != 0 && _method != 0; };
    // Return TRUE if both object and method pointers of this binding are
    // non-NULL.
    
    void operator= (const CL_Object& o)
        {*this = ((const UI_EventBinding0<Client>&) o);};
    // Cast {\tt o} down to {\tt UI_EventBinding0<Client>} and assign it to
    // this  object.
    
    UI_EventBinding0<Client>& operator= (const UI_EventBinding0<Client>& b)
        {_client = b._client; _method = b._method; return *this;};
    // Assign {\tt b}'s contents to this object.
    
    
    bool operator== (const CL_Object& o) const
        {return *this == ((const UI_EventBinding0<Client>&) o);};
    // Cast the given object {\tt o} down to a Binding, and return TRUE if the
    // object pointers in it and  this object are the same. 
    
    bool operator== (const UI_EventBinding0<Client>& o) const
        { return _client == o._client; };
    // Return TRUE if this Binding has the same client pointer as {\tt o} does.

    CL_Object* Clone () const
        { return new UI_EventBinding0<Client> (_client, _method);};
    // Override the method inherited from {\tt CL_Object}.

    const  char* ClassName () const {return "UI_EventBinding0";};

    CL_ClassId ClassId () const {return _UI_EventBinding0_CLASSID;};
    
protected:
    Client*   _client;
    MethodPtr _method;

    bool ExecuteEvtBind (UI_Event& e) const
        { return Valid() ? (_client->*_method) () : FALSE;};
    // Override the virtual method inherited from {\tt AbstractEventBinding}.

};


// EventBinding1 is a class template with parameter {\tt Client}. It is
// derived 
// from AbstractEventBinding and represents  an event binding whose client
// is of type {\tt Client} and whose client method takes a single parameter
// of type reference-to-Event. It
// overrides the inherited protected method {\tt ExecuteEvtBind} with an
// implementation that invokes the client method.

template <class Client>
class YACL_UI UI_EventBinding1: public UI_AbstractEventBinding {

public:
    //[ Local
    typedef bool (Client::*MethodPtr) (UI_Event&);
    // The type of the client method.
    //] Local
   
    UI_EventBinding1 (Client* client, MethodPtr method)
        {_client = client;  _method = method;};
    // Construct an event binding for an object of type {\tt Client}, that
    // has an event-handling method {\tt method}.

    UI_EventBinding1 () {_client = 0; _method = 0;};
    // Construct a null binding, one whose object and method pointers are
    // both NULL, for an object of type Client.

    UI_EventBinding1 (const UI_EventBinding1<Client>& b)
        {*this = b;};
    // Copy constructor.

    ~UI_EventBinding1 () {};
    // Destructor.
    
    bool Valid () const { return _client != 0 && _method != 0; };
    // Return TRUE if both object and method pointers of this binding are
    // non-NULL.
    
    void operator= (const CL_Object& o)
        {*this = ((const UI_EventBinding1<Client>&) o);};
    // Cast {\tt o} down to {\tt UI_EventBinding1<Client>} and assign it to
    // this object.
    
    UI_EventBinding1<Client>& operator= (const UI_EventBinding1<Client>& b)
        {_client = b._client; _method = b._method; return *this;};
    // Assign {\tt b} to this object.
    
    bool operator== (const CL_Object& o) const
        {return *this == ((const UI_EventBinding1<Client>&) o);};
    // Cast the given object {\tt o} down to a Binding, and return TRUE if
    // this object has the same client pointer as {\tt o}.
    
    bool operator== (const UI_EventBinding1<Client>& o) const
        { return _client == o._client; };
    // Return TRUE if
    // this object has the same client pointer as {\tt o}.

    CL_Object* Clone () const
        { return new UI_EventBinding1<Client> (_client, _method);};
    // Override the method inherited from {\tt CL_Object}.

    const  char* ClassName () const {return "UI_EventBinding1";};
    
    CL_ClassId ClassId () const {return _UI_EventBinding1_CLASSID;};
    
protected:
    Client*   _client;
    MethodPtr _method;

    bool ExecuteEvtBind (UI_Event& e) const
        { return Valid() ? (_client->*_method) (e) : FALSE; };
    // Override the virtual method inherited from {\tt AbstractEventBinding}.

};



// EventBinding2 is a class template with parameter {\tt Client}. It is derived
// from AbstractEventBinding and represents  an event binding whose client
// is of type {\tt Client} and whose client method takes two parameters, the
// first being a reference to an event and the second the notification code.
// An EventBinding2 instance is created with three parameters: the identity
// of the client, the method to be invoked on the client, and a notification
// code that is meaningful to the client.
// When the client method is called, its first parameter will be the event
// that caused the binding to be executed, and the second parameter will be
// the notification code.
//
// EventBinding2
// overrides the inherited protected method {\tt ExecuteEvtBind} with an
// implementation that invokes the client method.

template <class Client>
class YACL_UI UI_EventBinding2: public UI_AbstractEventBinding {

public:
    typedef bool (Client::*MethodPtr) (UI_Event&,  long);
   
    UI_EventBinding2 (Client* client, MethodPtr method, long code)
        {_client = client;  _method = method; _code = code;};
    // Construct an event binding for an object of type {\tt Client}, that
    // has an event-handling method {\tt method}.

    UI_EventBinding2 () {_client = 0; _method = 0;};
    // Construct a null binding, one whose object and method pointers are
    // both NULL, for an object of type Client.

    UI_EventBinding2 (const UI_EventBinding2<Client>& b)
        {*this = b;};
    // Copy constructor.

    ~UI_EventBinding2 () {};
    // Destructor.
    
    bool Valid () const { return _client != 0 && _method != 0; };
    // Return TRUE if both object and method pointers of this binding are
    // non-NULL.
    
    void operator= (const CL_Object& o)
        {*this = ((const UI_EventBinding2<Client>&) o);};
    
    UI_EventBinding2<Client>& operator= (const UI_EventBinding2<Client>& b)
        {_client = b._client; _method = b._method;
         _code = b._code; return *this;};
    // Assign {\tt b} to this object.
    
    bool operator== (const CL_Object& o) const
        {return *this == ((const UI_EventBinding2<Client>&) o);};
    // Cast the given object {\tt o} down to a Binding, and return TRUE if
    // this object has the same client pointer as does {\tt o}.
    
    bool operator== (const UI_EventBinding2<Client>& o) const
        { return _client == o._client; };
    // Return TRUE if
    // this object has the same client pointer as does {\tt o}.


    CL_Object* Clone () const
        { return new UI_EventBinding2<Client> (_client, _method, _code);};
    // Override the method inherited from {\tt CL_Object}.

    const  char* ClassName () const {return "UI_EventBinding2";};
    
    CL_ClassId ClassId () const {return _UI_EventBinding2_CLASSID;};
    
protected:
    Client*   _client;
    MethodPtr _method;
    long      _code;

protected:
    bool ExecuteEvtBind (UI_Event& e) const
        { return Valid() ? (_client->*_method) (e, _code) : FALSE; };
    // Override the virtual method inherited from {\tt AbstractEventBinding}.

};



#endif /* _evtbind_h_ */


