#ifndef _INOTDISP_
#define _INOTDISP_
/*******************************************************************************
* FILE NAME: inotdisp.hpp
*
* DESCRIPTION:
*   Declaration of the class(es):
*     INotificationDispatcher
*
* COPYRIGHT:
*   Licensed Materials - Property of IBM
*   (C) Copyright IBM Corporation 1996
*   All Rights Reserved
*   US Government Users Restricted Rights - Use, duplication, or disclosure
*   restricted by GSA ADP Schedule Contract with IBM Corp.
*
*******************************************************************************/
#ifndef _IVBDEFS__
#include <ivbdefs.h>
#endif

#ifndef _IOBJWIN_
  #include <iobjwin.hpp>
#endif

// Other dependency classes.
#ifndef _IHANDLE_
  #include <ihandle.hpp>
#endif

#ifndef _IRESLOCK_
  #include <ireslock.hpp>
#endif

class INotifier;
class IStandardNotifier2;
class IObserver;
class INotificationEvent;
class IAsyncNotificationEvent;
class IEventData;
class INotificationDispatcherHandler;
template <class Element, class Key> class IKeySet;

// Align classes on four byte boundary.
#pragma pack(4)

class INotificationDispatcher : public IObjectWindow {
/*******************************************************************************
* This class synchronously or asynchronously routes notifications between an
* INotifier's implementation threads and the thread on which it was created.
*******************************************************************************/

public:
/*------------------------------ Constructors ----------------------------------
| You can construct an object of this class as follows.                        |
|   - With the default constructor.  This object will handle all               |
|     INotifier parts created on the current thread.  The object cleans itself |
|     up when all notifiers created on this thread are deleted.                |
|-----------------------------------------------------------------------------*/
  INotificationDispatcher ( );

virtual
 ~INotificationDispatcher ( );

/*----------------------------- Locate Or Add ----------------------------------
| Notifiers are expected to use this function rather than the constructor.     |
|   locateOrAdd - Searches the collection of INotificationDispatcher           |
|                 if one is found for the current thread, it is returned.  If  |
|                 not, one is created, added to the collection, and returned.  |
|-----------------------------------------------------------------------------*/
static INotificationDispatcher
 *locateOrAdd ( );

/*-------------------------------- Thread Id -----------------------------------
| Use this function to query the thread Id for this window.                    |
|   threadId - Returns the thread Id.                                          |
|-----------------------------------------------------------------------------*/
const IThreadId
 &threadId ( ) const;

/*------------------------- Notification Processing ----------------------------
| Used by IMultiThreadedNotifier objects to manage notifications.              |
|   postNotification    - Posts the notification for asynchronous processing.  |
|                         The event is adopted and will be deleted.            |
|   sendNotification    - Sends the notification for synchronous processing.   |
|   deleteNotifications - Ensures that all notifications for the passed        |
|                         object are never dispatched.  This function does the |
|                         thread check and synchronous thread switch trick.    |
|                         This function assumes the notifier is disabled for   |
|                         notifications.                                       |
|-----------------------------------------------------------------------------*/
INotificationDispatcher
 &postNotification    ( IAsyncNotificationEvent* anEvent ),
 &sendNotification    ( const INotificationEvent& anEvent ),
 &deleteNotifications ( const INotifier& aNotifier );

/*------------------------- Delete Support -------------------------------------
| These are used to support safe deletion of notifiers.                        |
|   deleteAsync - IStandardNotifier2::deleteAsync() is implemented as a posted |
|                 WM_USER message the handler understands.                     |
|   safeCleanUp - Used by INotifier destructor to synchronously thread switch  |
|                 and safely clean up items that must be cleaned up on the     |
|                 dispatch thread.  This function assumes the notifier is      |
|                 disabled for notifications.                                  |
|-----------------------------------------------------------------------------*/
INotificationDispatcher
 &deleteAsync ( IStandardNotifier2& aStandardNotifier2 ),
 &safeCleanUp ( INotifier& aNotifier );

/*--------------------- Add/Remove Observers Support ---------------------------
| These are used to support safe observer addition and removal.                |
|   safeAddObserver        - Does a synchronous thread switch and calls        |
|                            addObserver on the notifier.                      |
|   safeRemoveObserver     - Does a synchronous thread switch and calls        |
|                            removeObserver on the notifier.                   |
|   safeRemoveAllObservers - Does a synchronous thread switch and calls        |
|                            removeAllObservers on the notifier.               |
|-----------------------------------------------------------------------------*/
INotificationDispatcher
 &safeAddObserver        ( INotifier&        aNotifier,
                           IObserver&        anObserver,
                           const IEventData& userData ),
 &safeRemoveObserver     ( INotifier& aNotifier,
                           IObserver& anObserver ),
 &safeRemoveAllObservers ( INotifier& aNotifier );


private:
// The private copy constructor and assignment operator are not implemented.
  INotificationDispatcher ( const INotificationDispatcher& rhs );
INotificationDispatcher
 &operator= ( const INotificationDispatcher& rhs );

/*--------------------------- Private State Data -----------------------------*/
// The reference count.
unsigned long fNotifierCount;

// Need to remember our thread id to make sure all the thread checks work.
IThreadId fThreadId;

// Used to make sure we are not simultaneously posting an async notification
// and deleting async notifications from the queue in deleteNotifications.
IPrivateResource fAsyncKey;

// The single handler.
static INotificationDispatcherHandler* fgHandler;

// The collection of INotificationDispatcher objects.
static IKeySet<INotificationDispatcher *, IThreadId>* fgDispatchers;
static IPrivateResource                               fgDispatchersKey;

}; // INotificationDispatcher

// Key function used by collections of pointers to objects of this class.
const IThreadId& key ( INotificationDispatcher* const& element );

// Resume compiler default packing.
#pragma pack()

#endif // _INOTDISP_
