/*
** lister.C - a very simple line lister which manages its own screen
**            and provides functions which can be executed on the line
**            the cursor is on -- the "current" line.
**
** lister.C 1.38   Delta\'d: 08:25:56 11/10/92   Mike Lijewski, CNSF
**
** Copyright \(c\) 1991, 1992 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: \(1\) source distributions retain this entire copyright
** notice and comment, and \(2\) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University\'\' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS\'\' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/


#include <ctype.h>

#ifndef _IBMR2
#include <libc.h>
#endif

#include <osfcn.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#if defined(ESIX) || defined(__EMX__)
typedef int pid_t;
#endif /*ESIX*/
#include <sys/wait.h>
#include <unistd.h>

#include "classes.h"
#include "display.h"
#include "help.h"
#include "keys.h"
#include "lister.h"
#include "problem.h"
#include "utilities.h"
#include "version.h"

// our screen
static DList *screen;

/*
** initialize_lister - initialize the lister.  This is so we can
**                     call redisplay with no arguments.
*/

void initialize_lister(DList *dl) { screen = dl; }

/*
** redisplay - this routine redisplays the DList which is our screen.
**             It assumes that the physical screen has become corrupted,
**             clearing each line before writing to it.
*/

static void redisplay()
{
    DLink *ln = screen->firstLine();

    clear_display_area();

    for (int i = 0; i < rows() - 2 && ln; i++, ln = ln->next())
        display_string(ln->line(), ln->length());

    update_modeline();
    clear_message_line();

    if (screen->currLine()->length() > columns())
        leftshift_current_line(screen);
    else
        move_cursor(screen->savedYPos(), screen->savedXPos());
    synch_display();
}

/*
** scroll_up_one_line - Scroll the listing up one line.
**                      We only call this routine when we KNOW that
**                      there is at least one line below the window
**                      which can be scrolled into it and the cursor
**                      is on the last line of the screen.
*/

static void scroll_up_one_line(DList *dl)
{
    dl->setFirst(dl->firstLine()->next());
    dl->setLast(dl->lastLine()->next());
    dl->setCurrLine(dl->lastLine());

    if (CS)
    {
        scroll_listing_up_one();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else if (DL || SF)
    {
        clear_modeline();
        scroll_screen_up_one();
        update_modeline();
        move_cursor(rows()-3, 0);
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else
        redisplay();

    dl->saveYXPos(rows()-3, goal_column(dl));
    move_cursor(rows()-3, dl->savedXPos());
}

/*
** scroll_down_one_line - Scroll the listing down one line.
**                        We only call this routine when we KNOW
**                        that the head of the listing is not visible
**                        and the cursor is on the first line in the window.
*/

static void scroll_down_one_line(DList *dl)
{
    if (lines_displayed(dl) == rows() - 2)
        //
        // Must update lastLine.  We previously had a screenfull of lines.
        //
        dl->setLast(dl->lastLine()->prev());

    dl->setFirst(dl->firstLine()->prev());
    dl->setCurrLine(dl->firstLine());

    if (CS)
    {
        scroll_listing_down_one();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else if (AL || SR)
    {
        clear_modeline();
        scroll_screen_down_one();
        update_modeline();
        cursor_home();
        display_string(dl->currLine()->line(), dl->currLine()->length());
    }
    else
        redisplay();

    dl->saveYXPos(0, goal_column(dl));
    move_cursor(0, dl->savedXPos());
}

/*
** scroll_up_full_window - scroll listing up one full window,
**                         leaving one line of overlap.  This routine
**                         is only called when we know that the tail
**                         of the listing is not currently displayed.
*/

static void scroll_up_full_window(DList *dl)
{
    DLink *ln = dl->lastLine();
    dl->setFirst(ln);
    dl->setCurrLine(ln);

    clear_display_area();

    for (int i = 0; i < rows() - 2 && ln; i++, ln = ln->next())
        display_string(ln->line(), ln->length());

    ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
    dl->saveYXPos(0, goal_column(dl));
    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(0, dl->savedXPos());

    synch_display();
}

/*
** scroll_down_full_window - try to scroll listing down one full window,
**                           with one line of overlap.  This routine is
**                           only called when we KNOW that there is at
**                           least one line "above" the current listing.
**                           Only change the current line if it flows off
**                           the "bottom" of the screen.  This routine is
**                           only called when we know that the head of the
**                           listing is not currently displayed.
*/

static void scroll_down_full_window(DList *dl)
{
    DLink *ln = dl->firstLine();
    for (int y = 0; y < rows() - 3 && ln != dl->head(); y++, ln = ln->prev());
    //
    // y == # of lines preceding firstLine to add to screen
    //
    dl->setFirst(ln);

    clear_display_area();

    for (int j = 0; j < rows()-2 && ln; j++, ln = ln->next())
        display_string(ln->line(), ln->length());

    if (ln) dl->setLast(ln->prev());

    if (dl->savedYPos()+y >= rows()-2)
    {
        dl->setCurrLine(dl->lastLine());
        dl->saveYXPos(rows()-3, goal_column(dl));
    }
    else
        dl->saveYXPos(dl->savedYPos()+y, dl->savedXPos());

    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());

    synch_display();
}

/*
** scroll_up_half_window - scroll listing up half a window.  This routine
**                         is only called when the tail of the listing
**                         is not being displayed.  We try to leave the
**                         cursor on the file it was on previously,
**                         otherwise it is left on the first file in
**                         the screen.
*/

static void scroll_up_half_window(DList *dl, int y)
{
    if (dl->currLine()->length() > columns()) rightshift_current_line(dl);

    DLink *ln = dl->firstLine();
    for (int i = 0; i < (rows() - 2)/2; i++, ln = ln->next()) ;
    dl->setFirst(ln);

    if (CS || DL || SF || DLN)
    {
        if (CS)
            scroll_listing_up_N((rows()-2)/2);
        else
        {
            clear_modeline();
            scroll_screen_up_N((rows()-2)/2);
            update_modeline();
        }
        move_cursor(rows() - 2 -((rows()-2)/2), 0);
        ln = dl->lastLine()->next();
        for (i = 0; i < (rows() - 2)/2 && ln; i++, ln = ln->next())
            display_string(ln->line(), ln->length());
        ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
    }
    else
    {
        clear_display_area();

        for (i = 0; i < rows() - 2 && ln->next(); i++, ln = ln->next())
            display_string(ln->line(), ln->length());

        if (i != rows()-2)
        {
            //
            // We hit last line before outputing all that we could.
            // Must output lastLine == tail.
            //
            display_string(ln->line(), ln->length());
            dl->setLast(ln);
            i++;  // so we know how many lines have been written
        }
        else
            dl->setLast(ln->prev());
    }

    int pos = y - (rows()-2)/2;
    if (pos < 0) { pos = 0; dl->setCurrLine(dl->firstLine()); }

    dl->saveYXPos(pos, goal_column(dl));
    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(pos, dl->savedXPos());

    synch_display();
}

/*
** scroll_down_half_window - try to scroll listing down half a window.
**                           If freshen is true, which is the default,
**                           the screen is refreshed.  It is important
**                           to note that we may not be able to scroll
**                           down a complete half window, since we
**                           always anchor the head of the listing to
**                           the first line in the screen.  This routine
**                           is only called when the head of the
**                           listing is not being displayed.
*/

static void scroll_down_half_window(DList *dl, int y, int freshen = 1)
{
    if (dl->firstLine() != dl->head())
    {
        //
        // We can scroll down.  Try to leave the cursor on the file
        // it started out on.  Otherwise, leave it on the
        // \(rows\(\)-2\)/2 line, which was the previous firstLine.
        //
        DLink *ln = dl->firstLine();
        for (int i = 0; i < (rows()-2)/2 && ln->prev(); i++, ln = ln->prev()) ;
        dl->setFirst(ln);

        if (dl->currLine()->length() > columns()) rightshift_current_line(dl);

        if (CS || AL || ALN || SR)
        {
            if (CS)
                scroll_listing_down_N(i);
            else
            {
                clear_modeline();
                scroll_screen_down_N(i);
                update_modeline();
                clear_message_line();
            }
            cursor_home();
            for (int j = 0; j < i; j++, ln = ln->next())
                display_string(ln->line(), ln->length());
            ln = dl->firstLine();
            for (int i = 0; i < rows()-2 && ln->next(); i++, ln = ln->next()) ;
            dl->setLast(ln);
        }
        else
        {
            clear_display_area();

            for (int i = 0; i < rows()-2 && ln->next(); i++, ln = ln->next())
                display_string(ln->line(), ln->length());

            if (i != rows() - 2)
            {
                //
                // We hit last line before outputing all that we could.
                // Must output lastLine == tail.
                //
                display_string(ln->line(), ln->length());
                dl->setLast(ln);
                i++;  // so we know how many lines have been written
            }
            else
                dl->setLast(ln->prev());
        }

        int pos = i + y;
        if (pos > rows() - 3)
        {
            pos = rows() - 3;
            dl->setCurrLine(dl->lastLine());
        }

        dl->saveYXPos(pos, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(pos, dl->savedXPos());

        if (freshen) synch_display();
    }
}

/*
** goto_first - position cursor on first line in listing.  This routine
**              is not called if atBegOfList is true.
*/

static void goto_first(DList *dl)
{
    if (dl->head() != dl->firstLine())
        initial_listing(dl);
    else
    {
        if (dl->currLine()->length() > columns()) rightshift_current_line(dl);
        dl->setCurrLine(dl->head());
    }

    dl->saveYXPos(0, goal_column(dl));
    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(0, dl->savedXPos());

    synch_display();
}

/*
** goto_last - position cursor on last file in listing.  This routine is
**             not called if atEndOfList is true.
*/

static void goto_last(DList *dl)
{
    if (dl->currLine()->length() > columns()) rightshift_current_line(dl);

    dl->setCurrLine(dl->tail());

    if (dl->tail() == dl->lastLine())
    {
        //
        // Only need to reposition the cursor.
        //
        dl->saveYXPos(lines_displayed(dl) - 1, goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(dl->savedYPos(), dl->savedXPos());
    }
    else
    {
        //
        // Redisplay end of listing & update our view.
        //
        DLink *ln = dl->tail();
        dl->setLast(ln);

        clear_display_area();

        for (int i = 0; i < rows() - 2; i++, ln = ln->prev())
        {
            move_cursor(rows() - 3 - i, 0);
            display_string(ln->line(), ln->length());
        }
        dl->setFirst(ln->next());
        dl->saveYXPos(rows() - 3,goal_column(dl));
        if (dl->currLine()->length() > columns())
            leftshift_current_line(dl);
        else
            move_cursor(rows() -3 , dl->savedXPos());
    }
    synch_display();
}


/*
** type_any_key_to_continue - ask user to type any key to continue.
**                            Done in standout mode.
*/

static inline void type_any_key_to_continue()
{
    (void)yes_or_no("Press Any Key to Continue", redisplay, Yes, 1);
}

/*
** examine_current_problem - attempt to examine the current problem.
*/

static void examine_current_problem(const DList *dl)
{
    if (examine_problem(get_problem_number(dl)))
        redisplay();
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}

/*
** update_problem_listing - updates the current line in the listing.
**                          number is the number of the problem
**                          in the current line.
*/

static void update_problem_listing(DList *dl, const char *number)
{
    datum key;
    key.dptr  = (char *) number;
    key.dsize = (int)strlen(key.dptr) + 1;
    open_database(GDBM_READER);
    datum data = gdbm_fetch(GdbmFile, key);
    gdbm_close(GdbmFile);
    if (data.dptr)
    {
        char *newline = summary_info(data);
        dl->currLine()->update(&newline);
        free(data.dptr);
    }
}

/*
** append_current_problem - attempt to append to the current problem.
**                          If successful, updates the listing line.
*/

static void append_current_problem(DList *dl)
{
    const char *number = get_problem_number(dl);

    if (append_to_problem(number))
    {
        //
        // Replace old listing line with new.
        //
        update_problem_listing(dl, number);
        redisplay();
    }
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}

/*
** close_current_problem - attempt to close the current problem
*/

static void close_current_problem(DList *dl)
{
    const char *number = get_problem_number(dl);

    if (close_problem(number))
    {
        //
        // Replace old listing line with new.
        //
        update_problem_listing(dl, number);
        redisplay();
    }
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}

/*
** reopen_current_problem - attempt to reopen the current problem
*/

static void reopen_current_problem(DList *dl)
{
    const char *number = get_problem_number(dl);

    if (reopen_problem(number))
    {
        //
        // Replace old listing line with new.
        //
        update_problem_listing(dl, number);
        redisplay();
    }
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}

/*
** modify_current_keywords - attempt to modify the keywords of the
**                           current problem
*/

static void modify_current_keywords(DList *dl)
{
    const char *number = get_problem_number(dl);

    if (modify_keywords(number))
    {
        //
        // Replace old listing line with new.
        //
        update_problem_listing(dl, number);
        redisplay();
    }
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}

/*
** modify_current_severity - attempt to modify the severity of the
**                           current problem.
*/

static void modify_current_severity(DList *dl)
{
    const char *number = get_problem_number(dl);

    if (modify_severity(number))
    {
        //
        // Replace old listing line with new.
        //
        update_problem_listing(dl, number);
        redisplay();
    }
    else
    {
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
    }
}


/*
** remove_listing_line - delete the current line in the DList
**                       and update both the screen and data
**                       structures appropriately.  y is the position
**                       in the window of the current line.  Returns 0
**                       if we have removed the last line in the listing,
**                       otherwise returns 1.
*/

static int remove_listing_line(DList *dl, int y)
{
    if (dl->lastLine() != dl->tail())
    {
        //
        // Last line in listing is not in window - scroll up one line.
        //
        dl->setLast(dl->lastLine()->next());
        dl->deleteLine();

        if (CS || DL)
        {
            if (CS)
                delete_listing_line(y);
            else
            {
                clear_modeline();
                delete_screen_line(y);
                update_modeline();
            }
            move_cursor(rows()-3, 0);
            display_string(dl->lastLine()->line(), dl->lastLine()->length());
        }
        else
        {
            clear_to_end_of_screen(y);
            move_cursor(y, 0);
            DLink *ln = dl->currLine();
            for (int i = y; i < rows()-2; i++, ln = ln->next())
                display_string(ln->line(), ln->length());
            update_modeline();
        }
        dl->saveYXPos(y, goal_column(dl));
    } else
    {
        //
        // Last line of listing is visible in window.
        //
        if (dl->atWindowTop() && dl->atWindowBot())
        {
            //
            // The last line in the window is also the first line.
            //
            if (dl->nelems() == 1)
            {
                cursor_home();
                clear_to_end_of_line();
                return 0;
            }
            scroll_down_half_window(dl, y, 0);
            dl->deleteLine();
            DLink *ln = dl->firstLine();
            for (int pos = 0; ln != dl->tail(); pos++, ln = ln->next()) ;
            dl->saveYXPos(pos, goal_column(dl));
            move_cursor(pos + 1, 0);
            clear_to_end_of_line();
            move_cursor(pos, dl->savedXPos());
        }
        else if (dl->atWindowBot())
        {
            //
            // We want to delete the last line in the window.
            //
            dl->deleteLine();
            move_cursor(y, 0);
            clear_to_end_of_line();
            dl->saveYXPos(y-1, goal_column(dl));
            move_cursor(y-1, dl->savedXPos());
        }
        else
        {
            //
            // We are in the middle of the listing.
            //
            dl->deleteLine();
            if (CS || DL)
            {
                if (CS)
                    delete_listing_line(y);
                else
                {
                    clear_modeline();
                    delete_screen_line(y);
                    update_modeline();
                }
            } else
            {
                clear_to_end_of_screen(y);
                move_cursor(y, 0);
                for (DLink *ln = dl->currLine(); ln; ln = ln->next())
                    display_string(ln->line(), ln->length());
                update_modeline();
            }
            dl->saveYXPos(y, goal_column(dl));
         }
    }
    if (dl->currLine()->length() > columns())
        leftshift_current_line(dl);
    else
        move_cursor(dl->savedYPos(), dl->savedXPos());
    return 1;
}

/*
** transfer_current_problem - attempt to transfer the current problem
**                            from the given area to another.
*/

static int transfer_current_problem(DList *dl)
{
    const char *number = get_problem_number(dl);

    char *area = prompt("New Area --> ", redisplay);

    //
    // Before calling transfer_problem we must guarantee that area
    // is a valid area and it is distict from the current area.
    //

    if (!is_area(area))
    {
        ding();
        message("`%' isn't a valid problem area", area);
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        DELETE area;
        return 1;
    }

    if (strcmp(area, CurrentArea()) == 0)
    {
        ding();
        message("`%' is the same as the current area", area);
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        DELETE area;
        return 1;
    }

    //
    // We have now guaranteed that this will either work, or we will
    // exit due to some unforeseen error.
    //
    (void)transfer_problem(number, area);
    DELETE area;

    //
    // Delete listing line.
    //
    if (remove_listing_line(dl, dl->savedYPos()) == 0)
        //
        // No more lines to view.
        //
        return 0;

    //
    // Replace old listing line with new.
    //
    update_problem_listing(dl, number);
    redisplay();

    return 1;
}

/*
** delete_current_problem - attempt to delete the current problem.  Returns
**                          0 if there are no more problems to view.
*/

static int delete_current_problem(DList *dl)
{
    if (delete_problem(get_problem_number(dl)))
        if (remove_listing_line(dl, dl->savedYPos()) == 0)
            return 0;

    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
    return 1;
}

/*
** save_problem_listing - saves problem listing to a file of the users choice.
**                        If the first character of the filename is ~, the
**                        tilde is replaced by the users home directory.
*/

static void save_problem_listing(const DList *dl)
{
    message_window_dirty = 1; // force message window dirty
    int status;

    char *file = prompt("Filename --> ", redisplay);

    const char *fullname;
    fullname = *file == '~' ? expand_tilde(file) : file;
#ifndef __EMX__
    pid_t pid = fork();
    switch(pid)
    {
      case -1:
        //
        // error
        //
        message("Sorry, can't write, fork() failed");
        break;
      case 0:
        //
        // In the child.
        //
        if (setuid(getuid()) < 0)
            exit(1);
        else
        {
#endif
            FILE *fp = fopen(fullname, "w");
            if (!fp)
                status = 1;
            else
            {
                //
                // Try to write the file.
                //
                for (DLink *ln = dl->head(); ln; ln = ln->next())
                    (void)fprintf(fp, "%s\n", ln->line());
                if (ferror(fp)) exit(1);
		(void)fclose(fp);
                status = 0;
	    }
#ifndef __EMX__
        }
        exit(status);
      default:
        //
        // In the parent.
        //
#ifdef NOWAITPID
        while (wait(&status) != pid) ;
#else
        waitpid(pid, &status, 0);
#endif
#endif
        if (!status)
            message("listing saved to file `%'", fullname);
        else
            message("problem saving listing, sorry");

        DELETE file;
#ifndef __EMX__
        break;
    }
#endif
    move_cursor(dl->savedYPos(), dl->savedXPos());
    synch_display();
}

/*
** help - give some help.  Deal with SIGWINCH and SIGTSTP.
*/

static void help()
{
    String old_modeline(current_modeline);
    update_modeline("----- HELP");

    int position = 0;
    char key;
    do {
        clear_display_area();
        for (int i = 0; i < rows() - 2 && i + position < HELP_FILE_DIM; i++)
            display_string(help_file[position + i]);

        clear_message_line();

        if (position + rows() -2 >= HELP_FILE_DIM)
            // the tail of the help message
            (void)fputs(HELP_MSG[2], stdout);
        else if (position == 0)
            // the head of the help message
            (void)fputs(HELP_MSG[0], stdout);
        else
            //  somewhere in between
            (void)fputs(HELP_MSG[1], stdout);
        synch_display();

        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0 // assume fails only when errno == EINTR
            )
        {
#ifdef SIGWINCH
            if (windowSizeChanged) { windowSizeChanged = 0; adjust_window(); }
#endif
            resumingAfterSuspension = 0;
            redisplay();
        }
        else if (key == KEY_SPC)
        {
            if (position >= HELP_FILE_DIM - 1) break;
            position += rows() - 2;
        }
        else if (key == *BC)
        {
            if (position == 0) break;
            position -= rows() - 2;
        }
        else
            break;  // return to the listing
    }
    while (position < HELP_FILE_DIM - 1);

    update_modeline(old_modeline);
    redisplay();
}

/*
** shell_command - execute a shell command.
**                 If *cmd == 0, start up a shell.
**                 If *cmd == !, reexecute most recent shell command.
*/

static void shell_command(DList *dl)
{
    static String saved_cmd;
    static String saved_shell;

    char *cmd  = prompt("!", redisplay);

    if (*cmd == 0)
    {
        //
        // Start up a shell.
        //
        if (saved_shell == "") saved_shell = getenv("SHELL");
#ifdef __EMX__
        if (saved_shell == "") saved_shell = "cmd.exe";
#else
        if (saved_shell == "") saved_shell = "sh";
#endif

        saved_cmd = saved_shell;

        const char *args[2];
        args[0] = saved_shell;
        args[1] = 0;

        message("Starting interactive shell ...");
        cursor_wrap();

        execute(saved_shell, args);
    }
    else if (*cmd == '!')
    {
        //
        //  Re-execute previously saved command.
        //
        if (saved_cmd != "")
        {
            message(saved_cmd);
            cursor_wrap();

            const char **args = tokenize(saved_cmd, " \t");

            execute(args[0], args, 1);
        }
        else
        {
            ding();
            message("No previous shell command");
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            DELETE cmd;
            return;
        }
    } else
    {
        //
        // Execute command.
        //
        saved_cmd = cmd;
        message(saved_cmd);
        cursor_wrap();

        const char **args = tokenize(saved_cmd, " \t");

        execute(args[0], args, 1);
    }

    DELETE cmd;
    redisplay();
}

/*
** read_from_keybd - read a key from the keyboard, taking care
**                   of SIGTSTPs and SIGWINCHs.  If the read fails,
**                   we assume it is because we caught a SIGTSTP
**                   or SIGWINCH.  In that case we redraw the screen
**                   with redisplay.
*/

static int read_from_keybd()
{
    char key;
    while (1)
    {
        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0)  // assume only fails when errno==EINTR
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            redisplay();
            continue;
        }
        return key;
    }
}

/*
** get_key - reads a key and then clears the message window,
**           if it needs to be cleared. Used only by read_commands in the
**           main switch statement so that message does not need to sleep
**           and clear on the messages that get written.  This way, the
**           message window is cleared after each keypress within the main
**           loop, when necessary.  We also check for and deal with window
**           size changes and the UP and DOWN arrow keys here.
*/

struct arrow_key {
    int len;
    int *seq;
    arrow_key(const char *);
};

arrow_key::arrow_key(const char *str)
{
    if (str == 0)
    {
        //
        // The capability is not defined.
        //
        len = 0;
        seq = 0;
        return;
    }

    seq = new int[12]; // should be ample

    int i = 0;
    do
    {
        switch (*str)
        {
          case '\\':
          {
              int c = *++str;
              switch (c)
              {
                case 'E':  seq[i++] = 0x1b; break;
                case 'b':  seq[i++] = '\b'; break;
                case 'f':  seq[i++] = '\f'; break;
                case 'n':  seq[i++] = '\n'; break;
                case 'r':  seq[i++] = '\r'; break;
                case 't':  seq[i++] = '\t'; break;
                case 'v':  seq[i++] = '\v'; break;
                case '\\': seq[i++] = '\\'; break;
                case '\'': seq[i++] = '\''; break;
                case '\"': seq[i++] = '\"'; break;
                case '^':  seq[i++] = '^';  break;
                default:
                    error("invalid escape in /etc/termcap for arrow key: \\%c", c);
                    break;
              }
              break;
          }
          case '^':
          {
              int c = *++str;
              if (isalpha(c))
              {
                  seq[i] = (c > 'a' ? c - 'a' : c - 'A') + 1;
                  i++;  // g++ 2.2.2 chokes if I write seq\[i++\] = ...
              }
              else
                  switch(c)
                  {
                    case '[':  seq[i++] = 0x1b; break;
                    case '\\': seq[i++] = 0x1c; break;
                    case ']':  seq[i++] = 0x1d; break;
                    case '^':  seq[i++] = 0x1e; break;
                    case '_':  seq[i++] = 0x1f; break;
                    default:
                        error("invalid control sequence for arrow key: ^%c", c);
                        break;
                  }
              }
              break;
          default: seq[i++] = *str; break;
        }
    } while (*++str);
    len = i;
}

static int get_key(DList *dl)
{
    int key, index = 0;
    static arrow_key up(KU), down(KD);
    static int *keys;
    static int remaining = 0;

    if (keys ==0) keys = new int[max(up.len, down.len)];

    if (remaining)
    {
        //
        // We have some characters left over from a partial match
        // of an arrow key; use them up.
        //
        key = keys[0];
        remaining--;
        for (int i = 0; i < remaining; i++) keys[i] = keys[i+1];
        return key;
    }
    else
        key = read_from_keybd();

    if (message_window_dirty)
    {
        clear_message_line();
        move_cursor(dl->savedYPos(), dl->savedXPos());
        synch_display();
        message_window_dirty = 0;
    }

    //
    // Now deal with potential arrow keys.
    //
    if (KU || KD)
    {
        for (index = 0; (index < up.len && up.seq[index] == key) ||
             (index < down.len && down.seq[index] == key); index++)
        {
            if ((up.len - 1) == index && up.seq[index] == key)
                return KEY_ARROW_UP;
            if ((down.len - 1) == index && down.seq[index] == key)
                return KEY_ARROW_DOWN;
            if (index == (max(up.len, down.len) - 1)) break;
            keys[index] = key;
            key = read_from_keybd();
        }
        if (index == 0)
            return key; // no initial match -- the most usual case
        else
        {
            //
            // We had a partial match, but not a complete one.
            // We must return the characters which we have read in
            // the proper order so that the main command loop can
            // check for matches.  The problem here is the potential
            // ambiguity between what the terminal claims to be arrow
            // keys and what has been hardcoded as commands.
            //
            keys[index] = key;
            key = keys[0];  // what we will return to the command loop
            for (int i = 0; i < index; i++) keys[i] = keys[i+1];
            remaining = index;
            return key;
        }
    }
    else
        return key;
}

/*
** read_commands - the command loop
*/

void lister_cmd_loop(DList *dl)
{
    int key;
    for (;;)
    {
        switch (key = get_key(dl))
        {
          case KEY_j:
          case KEY_n:
          case KEY_CTL_N:
          case KEY_SPC:
          case KEY_CR:
          case KEY_ARROW_DOWN:
            if (dl->atEndOfList())
            {
                ding();
                break;
            }
            if (dl->currLine()->length() > columns())
                rightshift_current_line(dl);
            if (dl->savedYPos() < rows() - 3)
            {
                // There are still more lines below us in the window
                // so we just move the cursor down one line.
                dl->setCurrLine(dl->currLine()->next());
                int x = goal_column(dl);
                if (x == dl->savedXPos())
                    cursor_down();
                else
                    move_cursor(dl->savedYPos() + 1, x);
                dl->saveYXPos(dl->savedYPos() + 1, x);
            }
            else
                //
                // We are on the last line on the screen and there
                // are more lines to display.  Scroll up one line
                // and leave the cursor on the next logical line.
                //
                scroll_up_one_line(dl);
            if (dl->currLine()->length() > columns())
                leftshift_current_line(dl);
            synch_display();
            break;

          case KEY_k:
          case KEY_p:
          case KEY_CTL_P:
          case KEY_CTL_Y:
          case KEY_ARROW_UP:
            if (dl->atBegOfList())
            {
                ding();
                break;
            }
            if (dl->currLine()->length() > columns())
                rightshift_current_line(dl);
            if (dl->savedYPos() != 0)
            {
                //
                // We are not at the top of the window so can move up.
                //
                dl->setCurrLine(dl->currLine()->prev());
                int x = goal_column(dl);
                if (x == dl->savedXPos() && UP)
                    cursor_up();
                else
                    move_cursor(dl->savedYPos() - 1, x);
                dl->saveYXPos(dl->savedYPos() - 1, x);
            }
           else
               //
               // We are on the first line of the window and there are
               // lines preceding us in the directory listing.
               //
               scroll_down_one_line(dl);
            if (dl->currLine()->length() > columns())
                leftshift_current_line(dl);
            synch_display();
            break;

          case KEY_CTL_F:
          case KEY_CTL_V:
            if (dl->lastLine() == dl->tail())
            {
                ding();
                break;
            }
            scroll_up_full_window(dl);
            break;

          case KEY_b:
          case KEY_CTL_B:
          case KEY_CTL_Z:
            if (dl->firstLine() == dl->head())
            {
                ding();
                break;
            }
            scroll_down_full_window(dl);
            break;

          case KEY_CTL_D:
            if (dl->lastLine() == dl->tail())
            {
                ding();
                break;
            }
            scroll_up_half_window(dl, dl->savedYPos());
            break;

          case KEY_CTL_U:
            if (dl->firstLine() == dl->head())
            {
                ding();
                break;
            }
            scroll_down_half_window(dl, dl->savedYPos());
            break;

          case KEY_TOP:
            if (dl->atBegOfList())
            {
                ding();
                break;
            }
            goto_first(dl);
            break;

          case KEY_BOT:
            if (dl->atEndOfList())
            {
                ding();
                break;
            }
            goto_last(dl);
            break;

          case KEY_e:
          case KEY_m:
          case KEY_v:
            examine_current_problem(dl); break;

          case KEY_a:
            append_current_problem(dl); break;

          case KEY_c:
            close_current_problem(dl); break;

          case KEY_d:
            if (delete_current_problem(dl) == 0)
                //
                // No more problems to view.
                //
                return;
            break;

          case KEY_r:
            reorganize_database(0);
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            break;

          case KEY_M:
            modify_current_keywords(dl); break;

          case KEY_P:
            modify_current_severity(dl); break;

          case KEY_R:
            reopen_current_problem(dl); break;

          case KEY_S:
            save_problem_listing(dl); break;

          case KEY_T:
            if (transfer_current_problem(dl) == 0)
                //
                // No more problems to view.
                //
                return;
            break;

          case KEY_QM: case KEY_H:
            help(); break;

          case KEY_BANG:
            shell_command(dl); break;

          case KEY_V:
            message(Version);
            move_cursor(dl->savedYPos(), dl->savedXPos());
            synch_display();
            break;

          case KEY_CTL_L:
            redisplay(); break;

          case KEY_q:
            return;

          case KEY_ESC:
            //
            // Some Emacs ESC key bindings.
            //
            switch(get_key(dl))
            {
              case KEY_v:
                if (dl->firstLine() == dl->head())
                {
                    ding();
                    break;
                }
                scroll_down_full_window(dl);
                break;

              case KEY_TOP:
                if (dl->atBegOfList())
                {
                    ding();
                    break;
                }
                goto_first(dl);
                break;

              case KEY_BOT:
                if (dl->atEndOfList())
                {
                    ding();
                    break;
                }
                goto_last(dl);
                break;

              default:
                ding();
                break;
            }
            break;

          default: ding(); break;
        }
    }
}
