/*
** utilities.C - utility functions
**
** utilities.C utilities.C 1.65   Delta\'d: 08:41:30 11/4/92   Mike Lijewski, CNSF
**
** Copyright \(c\) 1991, 1992 Cornell University
** All rights reserved.
**
** Redistribution and use in source and binary forms are permitted
** provided that: \(1\) source distributions retain this entire copyright
** notice and comment, and \(2\) distributions including binaries display
** the following acknowledgement:  ``This product includes software
** developed by Cornell University\'\' in the documentation or other
** materials provided with the distribution and in all advertising
** materials mentioning features or use of this software. Neither the
** name of the University nor the names of its contributors may be used
** to endorse or promote products derived from this software without
** specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS\'\' AND WITHOUT ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <ctype.h>

#ifndef _IBMR2
#include <libc.h>
#endif

#include <fcntl.h>

#include <osfcn.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>

#ifndef _IBMR2
#include <unistd.h>
#endif

// access is prototyped in <sys/access.h> on RS/6000s
#ifdef _IBMR2
#include <sys/access.h>
#include <sys/lockf.h>
#endif

#include <sys/stat.h>
#include <sys/types.h>
#if defined(ESIX) || defined(__EMX__)
typedef int pid_t;
#endif
#include <sys/wait.h>
#include <string.h>

#ifdef __EMX__
#include <errno.h>
#include <io.h>
#include <process.h>
#define OS2EMX
#include <os2.h>
#else
#include <sys/errno.h>
#endif

// this header file is badly busted on RS/6000s
#ifndef _IBMR2
#include <unistd.h>
#endif

#if defined FLOCK || !defined(R_OK)
#include <sys/file.h>
#endif

#include "classes.h"
#include "display.h"
#include "lister.h"
#include "problem.h"
#include "utilities.h"

/* remove this once GNU gets it fixed -- this stuff should be in <unistd.h> */
#ifdef __GNUG__
extern "C" int lockf(int, int, long);
extern "C" int flock(int, int);
#ifndef LOCK_UN
#define LOCK_UN 8
#endif
#ifndef LOCK_EX
#define LOCK_EX 2
#endif
#ifndef F_LOCK
#define F_LOCK   1
#endif
#ifndef F_ULOCK
#define F_ULOCK  0
#endif
#endif /*__GNUG__*/

const char KEY_CR    = '\r';   // carriage return
const char KEY_BKSP  = '\b';   // backspace
const char KEY_CTL_L = '\f';   // repaint screen -- CTR-L
const char KEY_DEL   = 127;    // ASCII DELETE

/*
** fgetline - returns a pointer to the start of a line read from fp,
**            or the null pointer if we hit eof or get an error from
**            fgets. Exits if new fails. Strips the newline from
**            the line. Caller should free memory if desired. size
**            is the expected length of the line to be read.
*/

char *fgetline(FILE *fp, int size)
{
    char *buffer = new char[size];

    char *result= fgets(buffer, size, fp);
    if (result == 0)
    {
        //
        // Either error or at eof.
        //
        DELETE buffer;
        return 0;
    }

    if (buffer[strlen(buffer) - 1] != '\n' && !feof(fp))
    {
        //
        // Longer line than buffer can hold.
        //
        char *restofline = fgetline(fp, size);

        if (restofline == 0) return 0; // eof or error

        char *longline = new char[strlen(buffer) + strlen(restofline) + 1];
        (void)strcat(strcpy(longline, buffer), restofline);

        DELETE restofline;
        DELETE buffer;

        if (longline[strlen(longline) - 1] == '\n')
            longline[strlen(longline) - 1] = 0;

        return longline;
    }
    else
    {
        if (buffer[strlen(buffer) - 1] == '\n')
            buffer[strlen(buffer) - 1] = 0;
        return buffer;
    }
}

/*
** display_string - prints a string to the given the display, guaranteeing not
**                  to print more than columns characters.  If the string
**                  exceeds the width of the window, a ! is placed in
**                  the final column.  len is the length of the string,
**                  if known, which defaults to zero. offset, which
**                  defaults to zero, is non-zero in those cases where we have
**                  already printed offset characters to the screen line.
**                  We never call this when trying to write to the last
**                  row on the screen.  That is the dominion of message.
*/

void display_string(const char *str, int length, int offset)
{
    int len = (length == 0 ? (int) strlen(str) : length) + offset;

    if (len < columns())
    {
        (void)fputs(str, stdout);
        cursor_wrap();
    }
    else if (len > columns())
    {
        (void)printf("%*.*s%c", columns() - offset - 1, columns() - offset - 1,
                     str, '!');
        if (!AM || XN) cursor_wrap();
    }
    else
    {
        (void)fputs(str, stdout);
        if (!AM || XN) cursor_wrap();
    }
}

/*
** error - Prints error message so it can be read.  This is the error
**         function we call once we have initialized the display.
*/

void error(const char *format, ...)
{
    va_list ap;
    va_start(ap, format);
    move_to_message_line();
    clear_to_end_of_line();
    deinit_screen_and_kbdr();
    (void) vfprintf(stdout, format, ap);
    putchar('\n');
    va_end(ap);
    exit(1);
}

/*
** execute - executes command using exec.  Returns 1 if the exec
**           went OK, otherwise it returns 0.  Assumes that the execd
**           program wants our open file descriptors.
**           Forces the effective uid to the real uid in the child.
**           If prompt is true, we prompt for a keypress before returning.
**           Prompt defaults to false.
*/

int execute(const char *file, const char *argv[], int prompt)
{
    deinit_screen_and_kbdr();
    unset_signals();

    int status;
#ifdef __EMX__
    status = spawnvp(P_WAIT, file, (char *const *)argv);
    set_signals();
    setraw();
    return status == 0 ? 1 : 0;
#else
    pid_t pid = fork();

    switch(pid)
    {
      case -1:
        //
        // error
        //
        return 0;
      case 0:
        //
        // in the child
        //
        if (setuid(getuid()) < 0)
            error("file %s, line %d, setuid() failed" __FILE__, __LINE__);

        execvp(file, (char *const *)argv);

        //
        // exec failed
        //
        exit(1);
      default:
        //
        // in the parent
        //
#ifdef NOWAITPID
        while (wait(&status) != pid) ;
#else
        waitpid(pid, &status, 0);
#endif

        set_signals();
        setraw();

        if (prompt)
            //
            // eat a character  -- print in standout mode
            //
            (void)yes_or_no("Press Any Key to Continue", 0, Yes, 1);

        enter_cursor_addressing_mode();
        enter_visual_mode();
        enable_keypad();
        synch_display();

        return status == 0 ? 1 : 0;
    }
#endif
}

#ifdef SIGWINCH

/*
** winch - set flag indicating window size changed.
*/

int windowSizeChanged;  // should be a sig_atomic_t

void winch(int)
{
    (void)signal(SIGWINCH, SIG_IGN);
    windowSizeChanged = 1;
    (void)signal(SIGWINCH, winch);
}

#ifdef M_UNIX
#include <sys/unistd.h>
#include <sys/stream.h>
#include <sys/ptem.h>
#endif

#include <sys/ioctl.h>

/*
** adjust_window - called to adjust our window after getting a SIGWINCH
*/

void adjust_window()
{
#ifdef TIOCGWINSZ
    struct winsize w;
    if (ioctl(fileno(stdout), TIOCGWINSZ, (char *)&w) == 0 && w.ws_row > 0)
        LI = w.ws_row;
    if (ioctl(fileno(stdout), TIOCGWINSZ, (char *)&w) == 0 && w.ws_col > 0)
        CO = w.ws_col;
#endif
    if (LI < 5 || CO < 20)
        error("screen too small to be useful");
}

#endif

/*
** prompt - displays msg prompt and then collects the response.
**          The keys of the response are echoed as they are collected.
**          The response should be deleted when no longer needed.
**          A response can contain any graphical character. Backspace
**          works as expected. Carriage return indicates the end of
**          response. Non-graphical characters are ignored.  If we
**          get suspended and resumed in prompt, redisplay is
**          the function to call to fixed up all but the message line
**          of the display.  We rely on signals interrupting read.
*/

char *prompt(const char *msg, void (*redisplay)())
{
    size_t written = 0;  // number of characters written to message line
    size_t len = strlen(msg);
    String nmsg(msg);

    clear_message_line();

    if (len < columns())
    {
        (void)fputs(nmsg, stdout);
        written = len;
    }
    else
    {
        //
        // Leave space for columns2 + 1 characters.
        //
        (void)fputs((const char *)nmsg + (len-columns()/2+1), stdout);
        written = columns()/2 - 1;
    }
    synch_display();

    //
    // We never echo into the last position in the message window.
    //
    size_t space_available = columns() - written; // available spaces in line
    char *response = new char[space_available + 1];
    size_t pos = 0;  // index of next character in response

    char key;
    for (;;)
    {
        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0    || // assume only fails when errno == EINTR
            key == KEY_CTL_L)
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            redisplay();
            clear_message_line();  // make sure we are on the message line
            response[pos] = 0;
            if (pos + len < columns())
            {
                //
                // Output message and response-to-date.
                //
                (void)fputs(nmsg, stdout);
                (void)fputs(response, stdout);
                space_available = columns() - pos - len;
            }
            else if (pos < columns())
            {
                //
                // Display the response.
                //
                (void)fputs(response, stdout);
                space_available = columns() - strlen(response);
            }
            else
            {
                //
                // Display the backend of the response.
                //
                (void)fputs(&response[pos - columns()/2 + 1], stdout);
                space_available = columns()/2 + 1;
            }
            synch_display();
        }
        else if (isprint(key))
        {
            //
            // echo character to message window and wait for another
            //
            response[pos++] = key;
            space_available--;
            if (!space_available)
            {
                //
                // Need to allocate more room for the response.
                // Note that strlen\(response\) == pos
                //
                space_available = columns()/2 + 1;
                char *nresponse = new char[pos + space_available + 1];
                response[pos] = 0;  // stringify response
                (void)strcpy(nresponse, response);

                DELETE response;
                response = nresponse;

                //
                // Shift prompt in message window so we
                // always have the end in view to which we are
                // adding characters as they are typed.
                //
                clear_message_line();
                (void)fputs(&response[pos - columns()/2 + 1], stdout);
                key = 0;  // nullify key
            }
            else
            {
                putchar(key);
                key = 0;  // nullify key
            }
            synch_display();
        }
        else
            switch (key)
            {
              case KEY_CR: // we have the complete response
                response[pos] = 0;
                clear_message_line();
                synch_display();
                return response;
              case KEY_DEL:
              case KEY_BKSP: // back up one character
                if (pos == 0)
                {
                    ding();
                    break;
                }
                backspace();
                DC ? delete_char_at_cursor() : clear_to_end_of_line();
                --pos;
                ++space_available;
                if (space_available == columns())
                {
                    //
                    // The only way this can happen is if we
                    // had previously shifted the response to the left.
                    // Now we must shift the response to the right.
                    //
                    clear_message_line();
                    response[pos] = 0;
                    if (pos + len < columns())
                    {
                        //
                        // Output message and response-to-date.
                        //
                        (void)fputs(nmsg, stdout);
                        (void)fputs(response, stdout);
                        space_available = columns() - pos - len;
                    }
                    else if (pos < columns())
                    {
                        //
                        // Display the response.
                        //
                        (void)fputs(response, stdout);
                        space_available = columns() - strlen(response);
                    }
                    else
                    {
                        //
                        // Display the backend of the response
                        //
                        (void)fputs(&response[pos - columns()/2 + 1], stdout);
                        space_available = columns()/2 + 1;
                    }
                }
                synch_display();
                break;
              default: ding(); break; // ignore other characters
            }
    }
}

/*
** message - prints a message on the last line of the screen.
**           It is up to the calling process to put the cursor
**           back where it belongs.  Synchs the display.  It can
**           be called as either:
**
**                message\(msg\);
**           or
**                message\(fmt, str\);
**
**           In the later case it must be the case that the format fmt
**           has exactly one % into which the str will be substituted
**           as in the ?printf functions.  Prints in standout mode.
*/

//
// the definition -- declared in utilities.h
//
int message_window_dirty = 0;

void message(const char *fmt, const char *str)
{
    String msg;          // the complete message to be output

    clear_message_line();

    if (str)
    {
        const char *token = strchr(fmt, '%');
        if (token == 0)
            //
            // This should not happen.  But if it does, let us
            // just print the format fmt.
            //
            msg = fmt;
        else
        {
            msg  = String(fmt, token - fmt);
            msg += str;
            msg += token + 1;
        }
    }
    else
        msg = fmt;

    if (msg.length() < columns())
        (void)fputs(msg, stdout);
    else
        (void)printf("%*.*s", columns() - 1, columns() - 1, (const char *)msg);

    synch_display();
    message_window_dirty = 1;
}

/*
** yes_or_no - returns true if a y or Y is typed in response to
**             the msg. We deal with being suspended and resumed.
**
**             defResponse is the assumed default response.
**
**                 defResponse == Yes ==> that return value is true unless
**                                        n or N is typed.
**
**                 defResponse == No  ==> that return value is true only if
**                                        y or Y is typed; else it
**                                        return false.
**
**             If standout is true the message is displayed in standout mode.
**
**             It is assumed that the message that is printed somehow indicates
**             whether the default response is true or false.
*/

int yes_or_no(const char *msg,
              void (*redisplay)(),
              Response defResponse,
              int standout)
{
    if (standout) enter_standout_mode();
    message(msg);
    if (standout) end_standout_mode();

    char key;
    while (1)
        //
        // read a character dealing with interruption
        //
        if (resumingAfterSuspension ||
#ifdef SIGWINCH
            windowSizeChanged       ||
#endif
            get_key(&key) < 0    || // assume only fails when errno==EINTR 
            key == KEY_CTL_L)
        {
#ifdef SIGWINCH
            if (windowSizeChanged)
            {
                windowSizeChanged = 0;
                adjust_window();
            }
#endif
            resumingAfterSuspension = 0;
            if (redisplay) redisplay();
            if (standout) enter_standout_mode();
            message(msg);
            if (standout) end_standout_mode();
        }
        else
            break;

    clear_message_line();
    synch_display();

    switch (defResponse)
    {
      case Yes:
        return !(key == 'n' || key == 'N');
      case No:
        return   key == 'y' || key == 'Y';
    }
}

/*
** lock_file - lock the file opened with file descriptor fd.
**             Exits on error.
*/

void lock_file(int fd)
{
#ifdef __EMX__
    FILELOCK flock;
    flock.lOffset = 0;
    flock.lRange = 256;
    if (DosSetFileLocks(fd, 0, &flock, 1000000, 0) < 0) error("lock failed");
#else
    if (lseek(fd, 0, 0) < 0) error("lseek() failed");
#ifdef FLOCK
    if (flock(fd, LOCK_EX) < 0) error("flock - LOCK_EX");
#else
    if (lockf(fd, F_LOCK, 0) < 0) error("lockf - F_LOCK");
#endif
#endif
}

/*
** unlock_file - unlock file with file descriptor fd.
**               Exits on error.
*/

void unlock_file(int fd)
{
#ifdef __EMX__
    FILELOCK flock;
    flock.lOffset = 0;
    flock.lRange = 256;
    if (DosSetFileLocks(fd, &flock, 0, 10000, 0) < 0) error("unlock failed");
#else
    if (lseek(fd, 0, 0) < 0) error("lseek() failed");
#ifdef FLOCK
    if (flock(fd, LOCK_UN) < 0) error("flock - LOCK_UN");
#else
    if (lockf(fd, F_ULOCK, 0) < 0) error("lockf - F_ULOCK");
#endif
#endif
}

/*
** quit - cleanup and exit.  Called after a SIGHUP, SIGTERM, SIGQUIT
**        or SIGINT, or on normal termination.  sig defaults to 0.
*/

void quit(int sig) { deinit_screen_and_kbdr(); exit(sig); }

/*
** username - returns the username pertaining to the real uid.
**            Exits on error;
*/

const char *fullname()
{
#ifdef __EMX__
    return getenv("FULLNAME");
#else
    static String user;
    if (user == "")
    {
        struct passwd *entry = getpwuid(getuid());
        if (!entry)
            error("file %s, line %d, getpwuid() failed", __FILE__, __LINE__);
        user =  entry->pw_gecos;
    }
    return user;
#endif
}

const char *username()
{
#ifdef __EMX__
    return getenv("LOGNAME");
#else
    static String user;
    if (user == "")
    {
        struct passwd *entry = getpwuid(getuid());
        if (!entry)
            error("file %s, line %d, getpwuid() failed", __FILE__, __LINE__);
        user =  entry->pw_name;
    }
    return user;
#endif
}

const char *mailname()
{
  static char path[256];
  strcpy(path, username());
  char *host = getenv("HOSTNAME");
  if (host)
  {
    strcat(path, "@");
    strcat(path, host);
  }
  return path;
}

/*
** write_to_pipe - writes the data to the pipe on the file descriptor.
**                 Exits on error.
*/

void write_to_pipe(int fd, const char *data, int size)
{
    int nwritten = 0;
    while (size > 0)
    {
        nwritten = write(fd, data, size);
        if (nwritten <= 0)
            error("file %s, line %d, write() failed", __FILE__, __LINE__);
        size -= nwritten;
        data += nwritten;
    }

}

/*
** read_file - reads the file pointed to by fp into the array lines.
**             lines must have been previously allocated in the caller,
**             to size size. len is the expected length of the lines
**             in the file. pos is the position in lines
**             to which we start adding the lines; it defaults to zero.
**             In general, this will be zero, but in one case when I am
**             building an argv for an execv, I want a non-zero offset.
**             Returns the number of lines in lines or -1 on error.
**             lines is null-terminated.
**
**             This is used only for reading files containing
**             relatively few lines.  It skips over lines with a "#"
**             in the first column and blank lines.
*/

int read_file(FILE *fp, char** &lines, int size, int linelen, int pos)
{
    const int chunksize = 20;  // chunksize to grow by
    int nlines = 0;            // number of lines added to lines

    char *line = fgetline(fp, linelen);
    for (; line; line = fgetline(fp, linelen))
    {
        //
        // Skip comment lines and empty lines.
        //
        if (*line == '#' || strcmp(line, "") ==  0)
        {
            DELETE line;
            continue;
        }

        //
        // strip off newline
        //
        if (line[strlen(line) - 1] == '\n') line[strlen(line) - 1] = 0;
        lines[pos++] = line;
        nlines++;
        if (pos == size)
        {
            //
            // grow Areas
            //
            char **newspace = new char*[size += chunksize];
            for (int i = 0; i < pos; i++) newspace[i] = lines[i];
            DELETE lines;
            lines = newspace;
        }
    }
    if (feof(fp) && !ferror(fp))
    {
        //
        // null terminate lines
        //
        if (pos == size)
        {
            //
            // got to grow one more
            //
            char **newspace = new char*[size + 1];
            for (int i = 0; i < pos; i++) newspace[i] = lines[i];
            DELETE lines;
            lines = newspace;
        }
        lines[pos] = 0;
        return nlines;
    }
    else
        return -1;
}

/*
** tokenize - returns a null-terminated vector of the words in line
**            The vector and its elements are in volatile storage
**            which we manage here.
*/

const char **tokenize(const char *line, const char *separators)
{
    //
    // Since strtok modifies its argument, we use a copy of line.
    //
    static char *newline;     // volatile storage of vector elements
    DELETE newline;
    newline = new char[strlen(line) + 1];
    (void)strcpy(newline, line);

    const int chunksize = 5;  // chunksize to grow by
    int size   = chunksize;   // total size of vector
    int nwords = 0;           // number of words in vector
    static char **words;      // volatile storage for the word pointers
    DELETE words;
    words = new char*[chunksize];

    if ((words[nwords++] = strtok(newline, separators)) == 0)
        return (const char **)words;

    while (words[nwords++] = strtok(0, separators))
        if (nwords == size)
        {
            //
            // Grow words.
            //
            char **newspace = new char*[size += chunksize];
            for (int i = 0; i < nwords; i++) newspace[i] = words[i];
            DELETE words;
            words = newspace;
        }
    return (const char **)words;
}

/*
** read_and_exec_perm - returns non-zero if we have read and execute
**                      permission on the file, otherwise 0.
**                      Returns 0 on error.
*/

int read_and_exec_perm(const char *file)
{
    return access(file, R_OK | X_OK) == -1 ? 0 : 1;
}

/*
** expand_tilde - expects a string of the form "~ ...".
**                Returns a new string in volatile storage
**                with the user\'s home directory in place of the ~.
**                The user\'s home directory is always appended
**                in the form: "/usr/staff/mjlx"; a slash is not added to
**                the end of the home directory string.  Returns the original
**                string if we cannot get the user\'s home directory.
*/

const char *expand_tilde(char *str)
{
    static char *home = getenv("HOME");
    if (home == NULL)
    {
        struct passwd *user = getpwuid(getuid());
        if (user == NULL) return str;
        home = user->pw_dir;
    }
    if (*str != '~') return str;
    static String expansion;
    expansion  = home;
    expansion += (str + 1);
    return expansion;
}

/*
** update_screen_line
**
**     oldline is what is currently on the screen in row y
**     newline is what we want on the screen in row y
**
**     We make a good attempt to optimize the output of characters to
**     the screen.  We want to display newline on the screen,
**     assuming oldline is what is currently displayed.  This
**     will be "good" if oldline and newline are quite similar.
**     That is to say, this should only be called when there is an
**     expectation that oldline and newline are "almost" the same.
*/

void update_screen_line(const char *oldline, const char *newline, int y)
{
    if (strcmp(oldline, newline) == 0) return;

    size_t olen = strlen(oldline);
    size_t nlen = strlen(newline);
    size_t  len = olen < nlen ? olen : nlen;

    //
    // Never display more than columns characters.
    //
    int chop = 0;  // do we need to chop off the tail?
    if (len > columns()) { chop = 1; len = columns(); }

    char *equal = new char[len];

    //
    // How similar are the two strings?
    //
    int differences = 0;
    for (int i = 0; i < len; i++) equal[i] = 1;
    for (i = 0; i < len; i++)
        if (oldline[i] != newline[i]) { differences++; equal[i] = 0; }

    if (differences > columns()/2)
    {
        //
        // We just display the new line.
        //
        clear_to_end_of_line();
        (void)fputs(newline, stdout);
        DELETE equal;
        return;
    }

    if (!OS)
    {
        //
        // We can just overwrite the old with the new.
        //
        int last = -2;  // position of last character written
        for (i = 0; i < len; i++)
        {
            if (equal[i]) continue;
            if (i - 1 != last) move_cursor(y, i);
            (i == len - 1 && chop) ? putchar('!') : putchar(newline[i]);
            last = i;
        }
        if (nlen > olen)
        {
            //
            // Have more characters to output.
            //
            chop = len > columns();
            move_cursor(y, i);
            for (i = (int)len; i < nlen && i < columns(); i++)
                (i == columns()-1 && chop) ? putchar('!') : putchar(newline[i]);
        }
        else if (nlen < olen)
        {
            move_cursor(y, i);
            clear_to_end_of_line();
        }
    }
    else
    {
        //
        // We can not overwrite.  Truncate at first difference.
        //
        int first = 0;
        for (i = 0; i < len; i++)
            if (!equal[i])
            {
                first = i;
                break;
            }
        move_cursor(y, i);
        clear_to_end_of_line();
        for (; i < nlen && i < columns(); i++)
            (i == columns() - 1) ? putchar('!') : putchar(newline[i]);
    }
    DELETE equal;
}

/*
** update_modeline - this routine concatenates the two strings
**                   into the modeline.  The modeline
**                   is displayed in standout mode if possible.
**                   We never put more than columns characters into
**                   the modeline.  The modeline is the penultimate
**                   line on the terminal screen.  It does not
**                   synch the display.  If head == tail == 0, we
**                   just display the old modeline.  This happens
**                   if for some reason we had to clear the screen.
*/

//
// the current modeline
//
char *current_modeline;

void update_modeline(const char *head, const char *tail)
{
    move_to_modeline();
    enter_reverse_mode();

    if (head == 0)   // actually, head == tail == 0
    {
        //
        // Redisplay old modeline.
        //
        (void)fputs(current_modeline, stdout);
        end_reverse_mode();
        return;
    }

    int len = (int) strlen(head);
    char *new_modeline = new char[columns() + 1];
    (void)strncpy(new_modeline, head, columns());
    new_modeline[columns()] = 0;  // ensure it is null-terminated

    if (len < columns())
    {
        //
        // Write exactly columns characters to modeline.
        //
        for (int i = len; i < columns() - 1 && tail && *tail; i++, tail++)
            new_modeline[i] = *tail;
        if (i < columns() - 1)
        {
            new_modeline[i++] = ' ';
            for (; i < columns(); i++) new_modeline[i] = '-';
        }
        else if (tail && *tail)
            //
            // The tail was overly long.  Put a ! in the last space
            // on the modeline to signify truncation.
            //
            new_modeline[columns() - 1] = '!';
        else
            //
            // Here len == columns-1 && there is nothing else in tail.
            //
            new_modeline[columns() - 1] = ' ';
    }
    else if (len > columns())
        new_modeline[columns() - 1] = '!';

    if (current_modeline)
    {
        update_screen_line(current_modeline, new_modeline, rows() - 2);
        DELETE current_modeline;
    }
    else
        (void)fputs(new_modeline, stdout);

    current_modeline = new_modeline;
    end_reverse_mode();
}

/*
** lines_displayed - returns the number of lines in the DList
**                   currently displayed on the screen.
*/

int lines_displayed(DList *dl)
{
    DLink *ln = dl->firstLine();
    for (int i = 1; ln != dl->lastLine(); i++, ln = ln->next()) ;
    return i;
}

/*
** get_problem_number - returns the prob# of the current line of the DList
**                      in volatile storage.  The line is of the form:
**
**                         prob# ...
**
**                      We know INTIMATELY how summary_lines
**                      formats the output.
*/

const char *get_problem_number(const DList *dl)
{
    static String number; // our volatile storage

    const char* begin = dl->currLine()->line();

    //
    // step over any spaces preceding Prob #
    //
    while (*begin == ' ') begin++;

    number = String(begin, strchr(begin, ' ') - begin);

    return (const char *)number;
}

/*
** leftshift_current_line - shifts the current line in DList left until
**                          its tail is visible.
*/

void leftshift_current_line(DList *dl)
{
    int inc = dl->currLine()->length()-columns()+1;
    move_cursor(dl->savedYPos(), 0);
    clear_to_end_of_line();
    display_string(&(dl->currLine()->line())[inc],columns()-1);
    dl->saveYXPos(dl->savedYPos(), max(goal_column(dl)-inc, 0));
    move_cursor(dl->savedYPos(), dl->savedXPos());
}

/*
** rightshift_current_line - rightshifts current line to "natural" position.
*/

void rightshift_current_line(DList *dl)
{
    move_cursor(dl->savedYPos(), 0);
    clear_to_end_of_line();
    display_string(dl->currLine()->line(), dl->currLine()->length());
    dl->saveYXPos(dl->savedYPos(), goal_column(dl));
    move_cursor(dl->savedYPos(), dl->savedXPos());
}

/*
** initial_listing - prints the initial listing screen.
**                   Adjusts firstLine, lastLine and currLine.
*/

void initial_listing(DList *dl)
{
    DLink *ln = dl->head();
    dl->setFirst(ln);
    dl->setCurrLine(ln);

    clear_display_area();

    for (int i = 0; i < rows() - 2 && ln; ln = ln->next(), i++)
        display_string(ln->line(), ln->length());

    ln ? dl->setLast(ln->prev()) : dl->setLast(dl->tail());
}

/*
** seconds_in_date - returns the number of seconds in a date string
**                   of the form:
**
**                       "mmm dd mm:hh:ss yyyy"
**
**                   This is the format returned by ctime, with
**                   the leading "day" name chopped off.  According to ANSI C,
**                   a long must be able to hold values up to at least
**                   2147483647.  This will keep this function working for
**                   many years into the future.  Eventually, CurrentYear will
**                   have to be incremented.
*/

long seconds_in_date(const char *date)
{
    //
    // days\[i\] is the number of days preceding month i,
    // where 0 <= i <= 11
    //
    static const int days[] = { 0,31,59,90,120,151,181,212,243,273,304,334 };

    //
    // mhash\[*date + *\(date+1\) + *\(date+2\) - 268\] hashes to an integer
    // in the range 0-11 signifying the month.
    //
    static char mhash[] = { 11, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                            0, 0, 7, 0, 0, 2, 0, 0, 3, 0, 0, 9, 4, 8, 0,
                            0, 6, 0, 5, 0, 0, 0, 0, 0, 10 };

    const long SecondsPerDay  = 86400L;
    const long SecondsPerYear = 31536000L;
    const int CurrentYear = 1992;
    const char *const fmt = "%d";
    long total = 0;
    int tmp, mon = mhash[*date + *(date + 1) + *(date + 2) - 268];

    (void)sscanf(date + 13, fmt, &tmp);   // seconds
    total += tmp;
    (void)sscanf(date + 10, fmt, &tmp);   // minutes
    total += tmp * 60;
    (void)sscanf(date + 7 , fmt, &tmp);   // hours
    total += tmp * 3600;
    (void)sscanf(date + 4 , fmt, &tmp);   // day of the month
    total += (tmp - 1) * SecondsPerDay;
    total += days[mon] * SecondsPerDay;   // days in months preceding this one
    (void)sscanf(date + 16, fmt, &tmp);   // the year
    // leap year adjustment
    if (tmp % 4 == 0 && tmp % 100 != 0 && mon >= 2) total += SecondsPerDay;
    total += (tmp - CurrentYear) * SecondsPerYear;
    return total;
}

/*
** temporary_file - returns the name of a temporary file.  The temporary
**                  is forced to have permission 666.  
*/

const char *temporary_file()
{
    static char fname[256];
    char *file = tempnam(NULL, "p");
    if (file == 0)
        error("file %s, line %d, tmpnam() failed", __FILE__, __LINE__);
    int fd;
    if ((fd = open(file, O_RDWR|O_CREAT, 0666)) < 0)
        error("file %s, line %d, open(%s) failed",
              __FILE__, __LINE__, file);
    (void)close(fd);
    strcpy(fname, file);
    free(file);
    return fname;
}

/*
** set_signals - set up our signal handlers
*/

void set_signals()
{
    (void)signal(SIGHUP,  quit);
    (void)signal(SIGINT,  quit);
    (void)signal(SIGQUIT, quit);
    (void)signal(SIGTERM, quit);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, termstop);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, winch);
#endif
}

/*
** unset_signals - set signals back to defaults
*/

void unset_signals()
{
    (void)signal(SIGHUP,  SIG_DFL);
    (void)signal(SIGINT,  SIG_DFL);
    (void)signal(SIGQUIT, SIG_DFL);
    (void)signal(SIGTERM, SIG_DFL);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, SIG_DFL);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, SIG_DFL);
#endif
}

/*
** block_tstp_and_winch - block SIGTSTP and SIGWINCH
*/

#ifdef BSDSIGS
static int oldmask;
#elif POSIXSIGS
static sigset_t oldset;
#endif

void block_tstp_and_winch()
{
#ifdef BSDSIGS
    int oldmask = sigblock(sigmask(SIGTSTP)
#ifdef SIGWINCH
                           | sigmask(SIGWINCH)
#endif
                           );
#elif POSIXSIGS
    sigset_t newset;
    sigemptyset(&newset);
#ifdef SIGTSTP
    sigaddset(&newset, SIGTSTP);
#endif
#ifdef SIGWINCH
    sigaddset(&newset, SIGWINCH);
#endif
    if (sigprocmask(SIG_BLOCK, &newset, &oldset) < 0)
        error("file %s, line %d, sigprocmask(SIG_BLOCK) failed\n",
              __FILE__, __LINE__);
#else
    //
    // We use ANSI C signals.  These can be "lost" but it is the
    // best we can do.
    //
#ifdef SIGTSTP
    (void)signal(SIGTSTP, SIG_IGN);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, SIG_IGN);
#endif
#endif
}

/*
** unblock_tstp_and_winch - unblock SIGTSTP and SIGWINCH
*/

void unblock_tstp_and_winch()
{
#ifdef BSDSIGS
    (void)sigsetmask(oldmask);
#elif POSIXSIGS
    if (sigprocmask(SIG_SETMASK, &oldset, 0) < 0)
        error("file %s, line %d, sigprocmask(SIG_SETMASK) failed\n",
              __FILE__, __LINE__);
#else
#ifdef SIGTSTP
    (void)signal(SIGTSTP, termstop);
#endif
#ifdef SIGWINCH
    (void)signal(SIGWINCH, winch);
#endif
#endif
}


#ifdef __EMX__
extern "C" int _read_kbd(int, int, int);
#endif

int get_key(char *key)
{
#ifdef __EMX__
again:
  int c = _read_kbd(0, 1, 0);
  if (c == 0)
    switch (_read_kbd(0, 1, 0))
    {
    case 'G':
      *key = '<';
      break;
    case 'O':
      *key = '>';
      break;
    case 'H':
      *key = 'P' - 64;
      break;
    case 'P':
      *key = 'N' - 64;
      break;
    case 'I':
      *key = 'Z' - 64;
      break;
    case 'Q':
      *key = 'V' - 64;
      break;
    default:
      goto again;
    }
  else
    *key = c;
  return *key == -1 ? -1 : 1;
#else
  return read(0, key, 1);
#endif  
}

