#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#define INCL_GPIBITMAPS
#define INCL_GPIPRIMITIVES
#define INCL_WINRECTANGLES
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "sprite.h"

#define ACCSEM_SET               0
#define ACCSEM_CLEAR             1
#define ACCSEM_ALREADYSET        2
#define ACCSEM_NOTSET            3
#define ACCSEM_ERROR             4

#define HSTATUS_INLIBRARY        0x00000001L

static ULONG queryHandleType(PVOID pvHandle);
static USHORT accessSem(PHEADER phHandle,USHORT usAction);
static HPS getMemHps(HAB habAnchor);
static BOOL clipBltPoints(HAB habAnchor,PPOINTL pptlArray,PSIZEL pszlPlay);
static BOOL drawSpriteAt(HPS hpsDraw,
                         HSPRITE hsSprite,
                         PSIZEL pszlSize,
                         PPOINTL pptlPos);
static BOOL drawBackAt(HPS hpsDraw,
                       HPLAYGROUND hpgPlay,
                       PRECTL prclDest,
                       PSIZEL pszlDest,
                       PRECTL prclSrc);

static ULONG queryHandleType(PVOID pvHandle)
//-------------------------------------------------------------------------
// This function returns a QH_* constant specifying the handle type.  It
// will be replaced by CmnQueryHandle() when this subsystem is integrated
// into Common/2.
//
// Input:  pvHandle - points the the handle to query
// Returns:   QH_ERROR if error, QH_* constant otherwise
//-------------------------------------------------------------------------
{
   if (pvHandle==NULL) {
      return QH_ERROR;
   } /* endif */

   switch (((PHEADER)pvHandle)->ulSig) {
   case SIG_HSPRITE:
      return QH_HSPRITE;
   case SIG_HPLAYGROUND:
      return QH_HPLAYGROUND;
   default:
      return QH_ERROR;
   } /* endswitch */
}

static USHORT accessSem(PHEADER phHandle,USHORT usAction)
//-------------------------------------------------------------------------
// This function provides semaphore access for mutual exclusion of private
// data access.
//
// Input:  phHandle - points to the handle header
//         usAction - specifies the action to perform:
//            ACCSEM_SET - requests access to the handle
//            ACCSEM_CLEAR - relinquishes access to the handle
//            ACCSEM_ALREADYSET - not used
//            ACCSEM_NOTSET - not used
// Returns:  ACCSEM_ERROR if an error occurred, else the action to take
//           on the next call to this function
//-------------------------------------------------------------------------
{
   switch (usAction) {
   case ACCSEM_SET:
      if ((phHandle->ulStatus & HSTATUS_INLIBRARY)!=0) {
         return ACCSEM_ALREADYSET;
      } /* endif */

      DosRequestMutexSem(phHandle->hsmAccess,SEM_INDEFINITE_WAIT);
      phHandle->ulStatus|=HSTATUS_INLIBRARY;
      return ACCSEM_CLEAR;
   case ACCSEM_CLEAR:
      if ((phHandle->ulStatus & HSTATUS_INLIBRARY)==0) {
         return ACCSEM_NOTSET;
      } /* endif */

      DosReleaseMutexSem(phHandle->hsmAccess);
      phHandle->ulStatus&=~HSTATUS_INLIBRARY;
      return ACCSEM_SET;
   case ACCSEM_ALREADYSET:
      return ACCSEM_NOTSET;
   case ACCSEM_NOTSET:
      return ACCSEM_ALREADYSET;
   default:
      return ACCSEM_ERROR;
   } /* endswitch */
}

static HPS getMemHps(HAB habAnchor)
//-------------------------------------------------------------------------
// This function creates an HPS associated with a memory HDC.  The
// HDC handle can be retrieved using the GpiQueryDevice() function.
//
// Input:  habAnchor - anchor block of the calling thread.
// Returns:  HPS handle if successful, NULLHANDLE otherwise
//-------------------------------------------------------------------------
{
   HDC hdcMem;
   SIZEL szlHps;
   HPS hpsMem;

   hdcMem=DevOpenDC(habAnchor,OD_MEMORY,"*",0,NULL,NULLHANDLE);
   if (hdcMem==NULLHANDLE) {
      return NULLHANDLE;
   } /* endif */

   szlHps.cx=0;
   szlHps.cy=0;

   hpsMem=GpiCreatePS(habAnchor,
                      hdcMem,
                      &szlHps,
                      PU_PELS|GPIT_MICRO|GPIA_ASSOC);
   if (hpsMem==NULLHANDLE) {
      DevCloseDC(hdcMem);
   } /* endif */

   return hpsMem;
}

static BOOL clipBltPoints(HAB habAnchor,PPOINTL pptlArray,PSIZEL pszlPlay)
//-------------------------------------------------------------------------
// This function clips the first two points in pptlArray to a rectangle
// of size pszlPlay.  The last two points in pptlArray are then adjusted
// by the amount clipped.
//
// It is assumed that the first two points refer to a coordinate space
// of size pszlPlay and that the two rectangles formed by the first and
// last pair of points in pptlArray are of the same size.
//
// Input:  habAnchor - anchor block of the calling thread.
//         pptlArray - points to array of 4 points for GpiBitBlt()
//         pszlPlay - points to the size of the playground to clip to
// Output:  pptlArray - points to adjusted array
// Returns:  TRUE if at least one pel was *not* clipped, FALSE if all
//           points fell outside of the clipping region.
//-------------------------------------------------------------------------
{
   RECTL rclPlay;
   RECTL rclDest;
   RECTL rclInter;
   RECTL rclDelta;

   rclPlay.xLeft=0;
   rclPlay.yBottom=0;
   rclPlay.xRight=pszlPlay->cx-1;
   rclPlay.yTop=pszlPlay->cy-1;

   rclDest.xLeft=pptlArray[0].x;
   rclDest.yBottom=pptlArray[0].y;
   rclDest.xRight=pptlArray[1].x;
   rclDest.yTop=pptlArray[1].y;

   WinIntersectRect(habAnchor,&rclInter,&rclPlay,&rclDest);

   //----------------------------------------------------------------------
   // If the result is an empty rectangle, return FALSE to indicate so.
   //----------------------------------------------------------------------
   if (WinIsRectEmpty(habAnchor,&rclInter)) {
      return FALSE;
   } /* endif */

   rclDelta.xLeft=rclDest.xLeft-rclInter.xLeft;
   rclDelta.yBottom=rclDest.yBottom-rclInter.yBottom;
   rclDelta.xRight=rclDest.xRight-rclInter.xRight;
   rclDelta.yTop=rclDest.yTop-rclInter.yTop;

   pptlArray[0].x-=rclDelta.xLeft;
   pptlArray[0].y-=rclDelta.yBottom;
   pptlArray[1].x-=rclDelta.xRight;
   pptlArray[1].y-=rclDelta.yTop;
   pptlArray[2].x-=rclDelta.xLeft;
   pptlArray[2].y-=rclDelta.yBottom;
   pptlArray[3].x-=rclDelta.xRight;
   pptlArray[3].y-=rclDelta.yTop;
   return TRUE;
}

static BOOL drawSpriteAt(HPS hpsDraw,
                         HSPRITE hsSprite,
                         PSIZEL pszlSize,
                         PPOINTL pptlPos)
//-------------------------------------------------------------------------
// This function draws the sprite at the specified position.  It is assumed
// that the background has already been drawn into hpsDraw before this
// function is called.
//
// Input:  hpsDraw - handle of the presentation space to draw in
//         hsSprite - handle of the sprite to draw
//         pszlSize - points to the size of hpsDraw.  If NULL, the size
//                    of the playground is used.
//         pptlPos - points to the point specifying the position.  If
//                   NULL, the sprite's current position is used.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   POINTL ptlUse;
   SIZEL szlUse;
   POINTL aptlPoints[4];

   if (!hsSprite->hpgPlay->bUpdate) {
      return TRUE;
   } /* endif */

   //----------------------------------------------------------------------
   // Initialize the local variables with either what was passed in or
   // the defaults as noted above in the function prologue
   //----------------------------------------------------------------------
   if (pptlPos==NULL) {
      ptlUse=hsSprite->ptlPos;
   } else {
      ptlUse=*pptlPos;
   } /* endif */

   if (pszlSize==NULL) {
      SprQueryPlaygroundSize(hsSprite->hpgPlay,&szlUse);
   } else {
      szlUse=*pszlSize;
   } /* endif */

   aptlPoints[0].x=ptlUse.x;
   aptlPoints[0].y=ptlUse.y;
   aptlPoints[1].x=aptlPoints[0].x+hsSprite->bmihMask.cx-1;
   aptlPoints[1].y=aptlPoints[0].y+hsSprite->bmihMask.cy-1;
   aptlPoints[2].x=0;
   aptlPoints[2].y=0;
   aptlPoints[3].x=aptlPoints[2].x+hsSprite->bmihMask.cx;
   aptlPoints[3].y=aptlPoints[2].y+hsSprite->bmihMask.cy;

   if (clipBltPoints(hsSprite->habAnchor,aptlPoints,&szlUse)) {
      //-------------------------------------------------------------------
      // Blit the mask and then the bitmap
      //-------------------------------------------------------------------
      GpiWCBitBlt(hpsDraw,
                  hsSprite->hbmMask,
                  4,
                  aptlPoints,
                  ROP_SRCAND,
                  BBO_IGNORE);

      GpiWCBitBlt(hpsDraw,
                  hsSprite->hbmBitmap,
                  4,
                  aptlPoints,
                  ROP_SRCPAINT,
                  BBO_IGNORE);
   } /* endif */

   return TRUE;
}

static BOOL drawBackAt(HPS hpsDraw,
                       HPLAYGROUND hpgPlay,
                       PRECTL prclDest,
                       PSIZEL pszlDest,
                       PRECTL prclSrc)
//-------------------------------------------------------------------------
// This function draws the background in the specified presentation space.
//
// Input:  hpsDraw - handle of the presentation space to draw in
//         hpgPlay - handle of the playground containing the background
//         prclDest - points to the destination rectangle.  If NULL, the
//                    value of prclSrc is used.
//         pszlDest - points to the size of hpsDraw.  If NULL, the size of
//                    the playground is used.
//         prclSrc - points to the source rectangle.  If NULL, the entire
//                   background is painted.
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   RECTL rclUseSrc;
   RECTL rclUseDest;
   SIZEL szlUse;
   POINTL aptlPoints[4];

   if (!hpgPlay->bUpdate) {
      return TRUE;
   } /* endif */

   if (prclSrc==NULL) {
      rclUseSrc.xLeft=0;
      rclUseSrc.yBottom=0;
      rclUseSrc.xRight=hpgPlay->bmihBack.cx;
      rclUseSrc.yTop=hpgPlay->bmihBack.cy;
   } else {
      rclUseSrc=*prclSrc;
   } /* endif */

   if (prclDest==NULL) {
      rclUseDest=rclUseSrc;
      rclUseDest.xRight--;
      rclUseDest.yTop--;
   } else {
      rclUseDest=*prclDest;
   } /* endif */

   if (pszlDest==NULL) {
      szlUse.cx=hpgPlay->bmihBack.cx;
      szlUse.cy=hpgPlay->bmihBack.cy;
   } else {
      szlUse=*pszlDest;
   } /* endif */

   aptlPoints[0].x=rclUseDest.xLeft;
   aptlPoints[0].y=rclUseDest.yBottom;
   aptlPoints[1].x=rclUseDest.xRight;
   aptlPoints[1].y=rclUseDest.yTop;
   aptlPoints[2].x=rclUseSrc.xLeft;
   aptlPoints[2].y=rclUseSrc.yBottom;
   aptlPoints[3].x=rclUseSrc.xRight;
   aptlPoints[3].y=rclUseSrc.yTop;

   if (clipBltPoints(hpgPlay->habAnchor,aptlPoints,&szlUse)) {
      //-------------------------------------------------------------------
      // If there is a background bitmap, blit it, otherwise black out the
      // area.
      //-------------------------------------------------------------------
      if (hpgPlay->hbmBack!=NULLHANDLE) {
         GpiWCBitBlt(hpsDraw,
                     hpgPlay->hbmBack,
                     4,
                     aptlPoints,
                     ROP_SRCCOPY,
                     BBO_IGNORE);
      } else {
         //----------------------------------------------------------------
         // WinFillRect() excludes the top and right of the rectangle
         //----------------------------------------------------------------
         rclUseDest.xRight++;
         rclUseDest.yTop++;
         WinFillRect(hpsDraw,&rclUseDest,hpgPlay->lBackColor);
      } /* endif */
   } /* endif */

   return TRUE;
}

SPRERROR EXPENTRY SprCreatePlayground(HAB habAnchor,PHPLAYGROUND phpgPlay)
//-------------------------------------------------------------------------
// This function creates a playground to which sprites can be added.
//
// Input:  habAnchor - anchor block of the calling thread.
// Output:  phpgPlay - points to the variable with the HPLAYGROUND handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   BITMAPINFOHEADER2 bmihInfo;
   LONG lValue;

   *phpgPlay=calloc(1,sizeof(PLAYGROUND));
   if (*phpgPlay==NULL) {
      *phpgPlay=NULL;
      return SPR_ERR_NOMEMORY;
   } /* endif */

   (*phpgPlay)->ulSig=SIG_HPLAYGROUND;
   (*phpgPlay)->ulStatus=0;

   if (DosCreateMutexSem(NULL,&(*phpgPlay)->hsmAccess,0,FALSE)) {
      free(*phpgPlay);
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phpgPlay)->habAnchor=habAnchor;

   (*phpgPlay)->hpsWork=getMemHps(habAnchor);
   if ((*phpgPlay)->hpsWork==NULLHANDLE) {
      free(*phpgPlay);
      *phpgPlay=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phpgPlay)->hdcWork=GpiQueryDevice((*phpgPlay)->hpsWork);

   bmihInfo.cbFix=16;
   bmihInfo.cx=MAX_SPRITE_CX*2;
   bmihInfo.cy=MAX_SPRITE_CY*2;
   bmihInfo.cPlanes=1;

   DevQueryCaps((*phpgPlay)->hdcWork,CAPS_COLOR_BITCOUNT,1,&lValue);
   bmihInfo.cBitCount=lValue;

   (*phpgPlay)->hbmWork=GpiCreateBitmap((*phpgPlay)->hpsWork,
                                        &bmihInfo,
                                        0,
                                        NULL,
                                        NULL);
   if ((*phpgPlay)->hbmWork==NULLHANDLE) {
      GpiDestroyPS((*phpgPlay)->hpsWork);
      DevCloseDC((*phpgPlay)->hdcWork);
      free(*phpgPlay);
      *phpgPlay=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   GpiSetBitmap((*phpgPlay)->hpsWork,(*phpgPlay)->hbmWork);

   (*phpgPlay)->lBackColor=CLR_BLACK;
   (*phpgPlay)->bUpdate=TRUE;

   (*phpgPlay)->hbmBack=NULLHANDLE;
   (*phpgPlay)->ulNumMembers=0;
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprDestroyPlayground(HPLAYGROUND hpgPlay)
//-------------------------------------------------------------------------
// This function destroys the playground including any sprites that are
// still members of it.  All resources consumed by the playground,
// including the back bitmap, are returned to the system.
//
// Input:  hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   ULONG ulIndex;
   HSPRITE hsSprite;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      GpiDeleteBitmap(hpgPlay->hbmBack);
   } /* endif */

   for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
      hsSprite=hpgPlay->ahsSprites[ulIndex];
      SprRemoveSprite(hpgPlay,hsSprite);
      SprDestroySprite(hsSprite);
   } /* endfor */

   GpiSetBitmap(hpgPlay->hpsWork,NULLHANDLE);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      GpiDeleteBitmap(hpgPlay->hbmBack);
   } /* endif */

   GpiDestroyPS(hpgPlay->hpsWork);
   DevCloseDC(hpgPlay->hdcWork);
   accessSem((PHEADER)hpgPlay,usAction);
   free(hpgPlay);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprAddSprite(HPLAYGROUND hpgPlay,HSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function labels a sprite as a "member" of the specified playground.
// Doing so allows the application to control the sprite's position,
// visibility, etc. on a drawing surface.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to add
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } else
   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hsSprite->hpgPlay!=NULL) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_HASPLAYGROUND;
   } else
   if (hpgPlay->ulNumMembers==MAX_SPRITES) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_PLAYGROUNDFULL;
   } /* endif */

   hpgPlay->ahsSprites[hpgPlay->ulNumMembers]=hsSprite;
   hpgPlay->ulNumMembers++;

   hsSprite->hpgPlay=hpgPlay;

   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprRemoveSprite(HPLAYGROUND hpgPlay,HSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function removes the sprite from the membership list of the
// specified playground.  The sprite can then be added to another
// playground, or this one at a later time.
//
// Since there is a limited number of sprites that can be members of
// a playground, this function can be used to temporarily remove unused
// sprites from a playground so that others can be used.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to remove
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   ULONG ulIndex;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } else
   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
      if (hpgPlay->ahsSprites[ulIndex]==hsSprite) {
         break;
      } /* endif */
   } /* endfor */

   if (ulIndex==hpgPlay->ulNumMembers) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   //----------------------------------------------------------------------
   // Adjust the member array by moving all of the sprites after the one
   // being removed to the slot just before there current position.  Then,
   // decrement the number of members and we're done.
   //----------------------------------------------------------------------
   hpgPlay->ulNumMembers--;

   while (ulIndex<hpgPlay->ulNumMembers) {
      hpgPlay->ahsSprites[ulIndex]=hpgPlay->ahsSprites[ulIndex+1];
      ulIndex++;
   } /* endwhile */

   hpgPlay->ahsSprites[ulIndex]=NULL;
   hsSprite->hpgPlay=NULL;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetPlaygroundBack(HPLAYGROUND hpgPlay,
                                       HBITMAP hbmNew,
                                       HBITMAP *phbmOld)
//-------------------------------------------------------------------------
// This function sets the background bitmap of the playground.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application unless the bitmap is "unset" from the playground (by
// calling this function again with a different handle).
//
// Input:  hpgPlay - handle to the playground
//         hbmNew - handle to the new bitmap to used as the background
// Output:  phbmOld - points to the handle to the old background bitmap.
//          This can be NULL, meaning that the application isn't interested
//          in receiving this value.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (phbmOld!=NULL) {
      *phbmOld=hpgPlay->hbmBack;
   } /* endif */

   hpgPlay->hbmBack=hbmNew;

   //----------------------------------------------------------------------
   // We're only interested in the cx and cy fields
   //----------------------------------------------------------------------
   hpgPlay->bmihBack.cbFix=16;
   GpiQueryBitmapInfoHeader(hpgPlay->hbmBack,&hpgPlay->bmihBack);

   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQueryPlaygroundBack(HPLAYGROUND hpgPlay,
                                         HBITMAP *phbmBack)
//-------------------------------------------------------------------------
// This function returns the handle of the background bitmap currently in
// use.
//
// Input:  hpgPlay - handle to the playground
// Output:  phbmBack - points to the handle to the background bitmap.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);
   *phbmBack=hpgPlay->hbmBack;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetPlaygroundSize(HPLAYGROUND hpgPlay,PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function sets the playground size for playgrounds that do not have
// a bitmap set as the background.
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the size of the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   hpgPlay->bmihBack.cx=pszlSize->cx;
   hpgPlay->bmihBack.cy=pszlSize->cy;

   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQueryPlaygroundSize(HPLAYGROUND hpgPlay,PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function returns the size of the playground.  For playgrounds with
// bitmaps set as the background, the returned value is the size of the
// bitmap.  Otherwise, the returned value is that which was specified on
// the last call to SprSetPlaygroundSize().
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the variable to receive the size of the
//                    playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);
   pszlSize->cx=hpgPlay->bmihBack.cx;
   pszlSize->cy=hpgPlay->bmihBack.cy;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetPlaygroundColor(HPLAYGROUND hpgPlay,LONG lBackColor)
//-------------------------------------------------------------------------
// This function sets the new background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         lBackColor - specifies the new background color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   hpgPlay->lBackColor=lBackColor;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQueryPlaygroundColor(HPLAYGROUND hpgPlay,
                                          PLONG plBackColor)
//-------------------------------------------------------------------------
// This function returns the background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         plBackColor - points to the variable to receive the background
//                       color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hpgPlay->hbmBack!=NULLHANDLE) {
      accessSem((PHEADER)hpgPlay,usAction);
      return SPR_ERR_HASBACKGROUND;
   } /* endif */

   *plBackColor=hpgPlay->lBackColor;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetUpdateFlag(HPLAYGROUND hpgPlay,BOOL bUpdate)
//-------------------------------------------------------------------------
// This function sets the update flag for the playground.  If FALSE, no
// drawing actually takes place in any of the functions requiring an HPS,
// and the value of the HPS handle may be NULLHANDLE.  If TRUE, updating
// is reenabled, but you should still call SprDrawPlayground() to refresh
// the screen with the current contents.
//
// Input:  hpgPlay - handle to the playground
//         bUpdate - specifies the new update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);
   hpgPlay->bUpdate=bUpdate;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQueryUpdateFlag(HPLAYGROUND hpgPlay,PBOOL pbUpdate)
//-------------------------------------------------------------------------
// This function returns the setting of the update flag.  See the notes
// for SprSetUpdateFlag() for more information about this setting.
//
// Input:  hpgPlay - handle to the playground
//         pbUpdate - points to the variable to receive the update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);
   *pbUpdate=hpgPlay->bUpdate;
   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprDrawPlayground(HPS hpsDraw,HPLAYGROUND hpgPlay)
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   ULONG ulIndex;

   if (queryHandleType(hpgPlay)!=QH_HPLAYGROUND) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hpgPlay,ACCSEM_SET);

   if (hpgPlay->bUpdate) {
      drawBackAt(hpsDraw,hpgPlay,NULL,NULL,NULL);

      for (ulIndex=0; ulIndex<hpgPlay->ulNumMembers; ulIndex++) {
         SprDrawSprite(hpsDraw,hpgPlay->ahsSprites[ulIndex]);
      } /* endfor */
   } /* endif */

   accessSem((PHEADER)hpgPlay,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprCreateSprite(HAB habAnchor,
                                  HBITMAP hbmBitmap,
                                  PHSPRITE phsSprite)
//-------------------------------------------------------------------------
// This function creates a sprite from the specified bitmap.  The sprite
// cannot be moved, shown, etc., however, until it is associated with a
// playground.
//
// The color black is used as the transparency color.  If you need to use
// black in the bitmap without it becoming transparent, use the next
// closest color.  <grin>
//
// New sprites are initialized as being at position (0,0) and hidden.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application or else unpredictable results will occur.
//
// Input:  habAnchor - anchor block of the calling thread.
//         hbmBitmap - handle to the bitmap
// Output:  phsSprite - points to the sprite handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   HPS hpsMem;
   HDC hdcMem;
   POINTL aptlPoints[4];

   *phsSprite=calloc(1,sizeof(SPRITE));
   if (*phsSprite==NULL) {
      *phsSprite=NULL;
      return SPR_ERR_NOMEMORY;
   } /* endif */

   (*phsSprite)->ulSig=SIG_HSPRITE;
   (*phsSprite)->ulStatus=0;

   if (DosCreateMutexSem(NULL,&(*phsSprite)->hsmAccess,0,FALSE)) {
      free(*phsSprite);
      return SPR_ERR_RESOURCE;
   } /* endif */

   (*phsSprite)->habAnchor=habAnchor;

   (*phsSprite)->hbmBitmap=hbmBitmap;

   (*phsSprite)->ptlPos.x=0;
   (*phsSprite)->ptlPos.y=0;
   (*phsSprite)->bVisible=FALSE;

   (*phsSprite)->bmihBitmap.cbFix=16;
   GpiQueryBitmapInfoHeader((*phsSprite)->hbmBitmap,&(*phsSprite)->bmihBitmap);

   //----------------------------------------------------------------------
   // Get an OD_MEMORY HDC and HPS to create the mask in.  Since we will
   // save the bitmap handle, but don't give a $%#@ about the HDC/HPS, they
   // can be local variables.
   //----------------------------------------------------------------------
   hpsMem=getMemHps(habAnchor);
   if (hpsMem==NULLHANDLE) {
      free(*phsSprite);
      *phsSprite=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   hdcMem=GpiQueryDevice(hpsMem);

   (*phsSprite)->bmihMask=(*phsSprite)->bmihBitmap;
   (*phsSprite)->bmihMask.cPlanes=1;
   (*phsSprite)->bmihMask.cBitCount=1;

   (*phsSprite)->hbmMask=GpiCreateBitmap(hpsMem,
                                         &(*phsSprite)->bmihMask,
                                         0,
                                         NULL,
                                         NULL);
   if ((*phsSprite)->hbmMask==NULLHANDLE) {
      GpiDestroyPS(hpsMem);
      DevCloseDC(hdcMem);
      free(*phsSprite);
      *phsSprite=NULL;
      return SPR_ERR_RESOURCE;
   } /* endif */

   GpiSetBitmap(hpsMem,(*phsSprite)->hbmMask);

   aptlPoints[0].x=0;
   aptlPoints[0].y=0;
   aptlPoints[1].x=aptlPoints[0].x+(*phsSprite)->bmihMask.cx-1;
   aptlPoints[1].y=aptlPoints[0].y+(*phsSprite)->bmihMask.cy-1;
   aptlPoints[2].x=0;
   aptlPoints[2].y=0;
   aptlPoints[3].x=aptlPoints[2].x+(*phsSprite)->bmihBitmap.cx;
   aptlPoints[3].y=aptlPoints[2].y+(*phsSprite)->bmihBitmap.cy;

   //----------------------------------------------------------------------
   // Set the foreground to white and the background to black so that this
   // works.  The resulting behavior in the GpiWCBitBlt() call is
   // inconsistent with the docs, so I don't know what to think.
   //----------------------------------------------------------------------
   GpiSetColor(hpsMem,CLR_WHITE);
   GpiSetBackColor(hpsMem,CLR_BLACK);

   GpiWCBitBlt(hpsMem,
               (*phsSprite)->hbmBitmap,
               4,
               aptlPoints,
               ROP_SRCCOPY,
               BBO_IGNORE);
   GpiSetBitmap(hpsMem,NULLHANDLE);
   GpiDestroyPS(hpsMem);
   DevCloseDC(hdcMem);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprDestroySprite(HSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function destroys the sprite and returns all resources to the
// system.
//
// Input:  hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay!=NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASPLAYGROUND;
   } /* endif */

   GpiDeleteBitmap(hsSprite->hbmBitmap);
   GpiDeleteBitmap(hsSprite->hbmMask);
   accessSem((PHEADER)hsSprite,usAction);
   free(hsSprite);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetSpritePosition(HPS hpsDraw,
                                       HSPRITE hsSprite,
                                       PPOINTL pptlNew)
//-------------------------------------------------------------------------
// This function changes the position of the sprite.  This function is
// optimized so that, if the rectangle bounding the sprite at the new
// position overlaps the old, only one "bit blit" to the specified HPS
// is done, eliminating flicker.
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in once it is
//                   moved
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   SIZEL szlPlay;
   SIZEL szlWork;
   RECTL rclOld;
   RECTL rclNew;
   RECTL rclUnion;
   RECTL rclSrc;
   RECTL rclDest;
   POINTL ptlWork;
   POINTL aptlPoints[4];

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if ((hsSprite->bVisible) && (hsSprite->hpgPlay->bUpdate)) {
      szlWork.cx=MAX_SPRITE_CX*2;
      szlWork.cy=MAX_SPRITE_CY*2;

      SprQueryPlaygroundSize(hsSprite->hpgPlay,&szlPlay);

      SprQuerySpriteRect(hsSprite,&rclOld);
      hsSprite->ptlPos=*pptlNew;
      SprQuerySpriteRect(hsSprite,&rclNew);

      WinUnionRect(hsSprite->habAnchor,&rclUnion,&rclOld,&rclNew);

      if ((rclUnion.xRight-rclUnion.xLeft>MAX_SPRITE_CX*2) ||
          (rclUnion.yTop-rclUnion.yBottom>MAX_SPRITE_CY*2)) {

         rclSrc.xLeft=rclOld.xLeft;
         rclSrc.yBottom=rclOld.yBottom;
         rclSrc.xRight=rclSrc.xLeft+hsSprite->bmihBitmap.cx;
         rclSrc.yTop=rclSrc.yBottom+hsSprite->bmihBitmap.cy;

         drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSrc);

         SprDrawSprite(hpsDraw,hsSprite);
      } else {
         rclSrc=rclUnion;
         rclSrc.xRight++;
         rclSrc.yTop++;

         rclDest.xLeft=0;
         rclDest.yBottom=0;
         rclDest.xRight=rclUnion.xRight-rclUnion.xLeft;
         rclDest.yTop=rclUnion.yTop-rclUnion.yBottom;

         drawBackAt(hsSprite->hpgPlay->hpsWork,
                    hsSprite->hpgPlay,
                    &rclDest,
                    &szlWork,
                    &rclSrc);

         ptlWork.x=hsSprite->ptlPos.x-rclUnion.xLeft;
         ptlWork.y=hsSprite->ptlPos.y-rclUnion.yBottom;

         drawSpriteAt(hsSprite->hpgPlay->hpsWork,hsSprite,&szlWork,&ptlWork);

         //----------------------------------------------------------------
         // GpiBitBlt is non-inclusive on source AND target
         //----------------------------------------------------------------
         aptlPoints[0].x=rclUnion.xLeft;
         aptlPoints[0].y=rclUnion.yBottom;
         aptlPoints[1].x=rclUnion.xRight+1;
         aptlPoints[1].y=rclUnion.yTop+1;
         aptlPoints[2].x=0;
         aptlPoints[2].y=0;
         aptlPoints[3].x=rclUnion.xRight-rclUnion.xLeft+1;
         aptlPoints[3].y=rclUnion.yTop-rclUnion.yBottom+1;

         if (clipBltPoints(hsSprite->habAnchor,aptlPoints,&szlPlay)) {
            GpiBitBlt(hpsDraw,
                      hsSprite->hpgPlay->hpsWork,
                      4,
                      aptlPoints,
                      ROP_SRCCOPY,
                      BBO_IGNORE);
         } /* endif */
      } /* endif */
   } else {
      hsSprite->ptlPos=*pptlNew;
   } /* endif */

   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQuerySpritePosition(HSPRITE hsSprite,PPOINTL pptlPos)
//-------------------------------------------------------------------------
// This function returns the current position of the sprite.  Note that
// a sprite has a current position even if it is hidden.
//
// Input:  hsSprite - handle to the sprite
// Output:  pptlPos - points to the current position
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   *pptlPos=hsSprite->ptlPos;
   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQuerySpriteSize(HSPRITE hsSprite,PSIZEL pszlSize)
//-------------------------------------------------------------------------
// This function returns the current size of the sprite.
//
// Input:  hsSprite - handle to the sprite
// Output:  pszlSize - points to the current size
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);
   pszlSize->cx=hsSprite->bmihBitmap.cx;
   pszlSize->cy=hsSprite->bmihBitmap.cy;
   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQuerySpriteRect(HSPRITE hsSprite,PRECTL prclRect)
//-------------------------------------------------------------------------
// This function returns the bounding rectangle of the sprite at its
// current position.
//
// Input:  hsSprite - handle to the sprite
// Output:  prclRect - points to the current bounding rectangle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   prclRect->xLeft=hsSprite->ptlPos.x;
   prclRect->yBottom=hsSprite->ptlPos.y;
   prclRect->xRight=prclRect->xLeft+hsSprite->bmihBitmap.cx-1;
   prclRect->yTop=prclRect->yBottom+hsSprite->bmihBitmap.cy-1;
   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprSetSpriteVisibility(HPS hpsDraw,
                                         HSPRITE hsSprite,
                                         BOOL bVisible)
//-------------------------------------------------------------------------
// This function shows or hides a sprite.
//
// Input:  hpsDraw - handle to the HPS to draw in once the sprite is
//                   shown or hidden
//         hsSprite - handle to the sprite
//         bVisible - new visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   RECTL rclSprite;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if (hsSprite->bVisible!=bVisible) {
      hsSprite->bVisible=bVisible;

      if (hsSprite->hpgPlay->bUpdate) {
         if (hsSprite->bVisible) {
            SprDrawSprite(hpsDraw,hsSprite);
         } else {
            rclSprite.xLeft=hsSprite->ptlPos.x;
            rclSprite.yBottom=hsSprite->ptlPos.y;
            rclSprite.xRight=rclSprite.xLeft+hsSprite->bmihMask.cx;
            rclSprite.yTop=rclSprite.yBottom+hsSprite->bmihMask.cy;

            drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSprite);
         } /* endif */
      } /* endif */
   } /* endif */

   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprQuerySpriteVisibility(HSPRITE hsSprite,PBOOL pbVisible)
//-------------------------------------------------------------------------
// This function returns the visibility state of the sprite
//
// Input:  hsSprite - handle to the sprite
// Output:  pbVisible - points to the visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   *pbVisible=hsSprite->bVisible;
   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}

SPRERROR EXPENTRY SprDrawSprite(HPS hpsDraw,HSPRITE hsSprite)
//-------------------------------------------------------------------------
// This function draws a sprite
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------
{
   USHORT usAction;
   RECTL rclSprite;

   if (queryHandleType(hsSprite)!=QH_HSPRITE) {
      return SPR_ERR_BADHANDLE;
   } /* endif */

   usAction=accessSem((PHEADER)hsSprite,ACCSEM_SET);

   if (hsSprite->hpgPlay==NULL) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_HASNOPLAYGROUND;
   } /* endif */

   if ((!hsSprite->bVisible) || (!hsSprite->hpgPlay->bUpdate)) {
      accessSem((PHEADER)hsSprite,usAction);
      return SPR_ERR_NOERROR;
   } /* endif */

   rclSprite.xLeft=hsSprite->ptlPos.x;
   rclSprite.yBottom=hsSprite->ptlPos.y;
   rclSprite.xRight=rclSprite.xLeft+hsSprite->bmihMask.cx;
   rclSprite.yTop=rclSprite.yBottom+hsSprite->bmihMask.cy;

   drawBackAt(hpsDraw,hsSprite->hpgPlay,NULL,NULL,&rclSprite);
   drawSpriteAt(hpsDraw,hsSprite,NULL,NULL);

   accessSem((PHEADER)hsSprite,usAction);
   return SPR_ERR_NOERROR;
}
