// ***********************************************************************
// CANVAS.HPP
// ***********************************************************************
//
// Written by Michael T. Duffy
//
// Canvas Graphics Object for 256 color modes.
//
// ***********************************************************************

#ifndef CANVAS_HPP
#define CANVAS_HPP

// ***********************************************************************
//  Constant Definitions
// ***********************************************************************

const USHORT       BOUNDS_NONE   = 0x0000;
const USHORT       BOUNDS_SOURCE = 0x0001;
const USHORT       BOUNDS_DEST   = 0x0002;


// ***********************************************************************
//  Class Definitions
// ***********************************************************************

class Canvas
  {
  private:

    // Internal data
    USHORT           usLastErrorCode;

    // Canvas structure
    USHORT           usWidth;
    USHORT           usHeight;
    ULONG            ulSize;
    PBYTE            pbySurface;
    PBYTE *          apbyRowStart;
    LONG             lRowChange;

    LONG             lDwordWidth;
    BOOL             bTopDown;

    // Drawing parameters

    BYTE             byDrawColor;


  public:

    // Constructors
    Canvas
      (
      USHORT         usWidthIn    = 320,
      USHORT         usHeightIn   = 200,
      BOOL           bTopDownIn   = TRUE
      );

    // Destructor
    ~Canvas ();

    // Member Functions

    VOID SetDrawColor
      (
      BYTE             byColor
      );

    BYTE QueryDrawColor
      (
      VOID
      );

    USHORT QueryWidth
      (
      VOID
      );

    USHORT QueryHeight
      (
      VOID
      );

    PBYTE QuerySurface
      (
      VOID
      );

    PBYTE QueryRowAddress
      (
      ULONG            ulIndex
      );

    PBYTE * QueryRowAddressArray
      (
      VOID
      );

    LONG QueryRowIncrement
      (
      VOID
      );

    VOID Clear
      (
      BYTE             byColor    = 0
      );

    VOID PutPixel
      (
      SHORT            sXPos,
      SHORT            sYPos
      );

    VOID HorizLine
      (
      SHORT            sX1,
      SHORT            sX2,
      SHORT            sY
      );

    VOID VertLine
      (
      SHORT            sX,
      SHORT            sY1,
      SHORT            sY2
      );

    VOID Rectangle
      (
      SHORT            sX1,
      SHORT            sY1,
      SHORT            sX2,
      SHORT            sY2
      );

    VOID XORHorizLine
      (
      SHORT            sX1,
      SHORT            sX2,
      SHORT            sY
      );

    VOID XORVertLine
      (
      SHORT            sX,
      SHORT            sY1,
      SHORT            sY2
      );

    VOID XORRectangle
      (
      SHORT            sX1,
      SHORT            sY1,
      SHORT            sX2,
      SHORT            sY2
      );

    VOID CopyDirtyRec
      (
      SHORT            sSourceX,
      SHORT            sSourceY,
      SHORT            sDestX,
      SHORT            sDestY,
      SHORT            sRecWidth,
      SHORT            sRecHeight,
      Canvas *         pcnvDest,
      USHORT           usBoundsCheck
      );


    BOOL Validate
      (
      VOID
      );

    USHORT QueryLastErrorCode
      (
      VOID
      ) {return (usLastErrorCode);};

  // End of Canvas definition
  };

typedef Canvas *   PCANVAS;


// ***********************************************************************
//  Inline Functions
// ***********************************************************************

//........................................................................
inline VOID Canvas::SetDrawColor
//........................................................................
  (
  BYTE             byColor
  )
{
byDrawColor = byColor;
};

//........................................................................
inline BYTE Canvas::QueryDrawColor
//........................................................................
  (
  VOID
  )
{
return (byDrawColor);
};

//........................................................................
inline USHORT Canvas::QueryWidth
//........................................................................
  (
  VOID
  )
{
return (usWidth);
};

//........................................................................
inline USHORT Canvas::QueryHeight
//........................................................................
  (
  VOID
  )
{
return (usHeight);
};

//........................................................................
inline PBYTE Canvas::QuerySurface
//........................................................................
  (
  VOID
  )
{
return (pbySurface);
};

//........................................................................
inline PBYTE Canvas::QueryRowAddress
//........................................................................
  (
  ULONG            ulIndex
  )
{
return (apbyRowStart [ulIndex]);
};

//........................................................................
inline PBYTE * Canvas::QueryRowAddressArray
//........................................................................
  (
  VOID
  )
{
return (apbyRowStart);
};

//........................................................................
inline LONG Canvas::QueryRowIncrement
//........................................................................
  (
  VOID
  )
{
return (lRowChange);
};

//........................................................................
inline BOOL Canvas::Validate
//........................................................................
  (
  VOID
  )
{
// This routine simply makes sure the canvas exists with valid data
if ((pbySurface == NULL) ||
    (usWidth  == 0) ||
    (usHeight == 0) ||
    (ulSize  == 0))
  {
  return (FALSE);
  }
else
  {
  return (TRUE);
  };
};


#endif // CANVAS_HPP
