;;; CLASS BROWSER FOR C++
;;; $Id: br-macro.el,v 3.1 1995/02/17 18:19:36 mmann Exp $
;;;
;;; **********************************************************************
;;; Copyright (C) 1993, 1994 Gerd Moellmann. All rights reserved.
;;; Altenbergstr. 6, D-40235 Duesseldorf, Germany
;;; 100025.3303@COMPUSERVE.COM
;;; Suggestions, comments and requests for improvements are welcome.
;;; **********************************************************************
;;;
;;; This version works with both Emacs version 18 and 19, and I want
;;; to keep it that way. It requires the CL-19 Common Lisp compatibility
;;; package for Emacs 18 and 19.
;;;
;;; This file contains macros used in the browse package.
;;; 

;; This file may be made part of the Emacs distribution at the option
;; of the FSF.

;; This code is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this code, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

(require 'cl-19 "cl")
(require 'backquote)

;;;
;;; Running under Emacs 19?
;;; 

(defmacro browse-emacs-19-p ()
  (` (memq 'emacs-19 browse-options)))

;;;
;;; Set face of a region (Emacs19 and WINDOW-SYSTEM, only).
;;; 

(defmacro browse-set-face (start end face)
  (when (and (browse-emacs-19-p) window-system)
    (` (when (and browse-hilit-on-redisplay (, face))
	 (overlay-put (make-overlay (, start) (, end)) 'face (, face))))))

;;; 
;;; Assign a text property depending on the circumstances.
;;;

(defmacro browse-put-text-property (start end prop value)
  (when (browse-emacs-19-p)
    (` (put-text-property (, start) (, end) (, prop) (, value)))))

(defmacro browse-get-text-property (point property)
  (when (browse-emacs-19-p)
    (` (get-text-property (, point) (, property)))))

;;;
;;; Execute BODY in a non read-only buffer
;;; 

(defmacro browse-output (&rest body)
  (let ((read-only (gensym "--browse-output--"))
	(modified (gensym "--browse-output--")))
    (` (let (((, read-only) buffer-read-only)
	     ((, modified) (buffer-modified-p)))
	 (unwind-protect
	     (progn
	       (setf buffer-read-only nil)
	       (,@ body))
	   (setf buffer-read-only (, read-only))
	   (set-buffer-modified-p (, modified)))))))

(put 'browse-output 'lisp-indent-hook 0)

;;;
;;; Like DOLIST performing BODY over all trees in a tree.
;;;
;;; ??? Beginning with Emacs 19.27 (EMX, OS/2) we sometimes get symbols
;;; ??? out of the obarray that were never put into it, like 
;;; ??? `change-major-mode-hook'. It seems that this release has a minor
;;; ??? bug. Whatsoever that is the reason why the property value is
;;; ??? tested for being NIL.
;;; 

(defmacro dotrees (spec &rest body)
  "(dotrees (tree vector) . body). Perform BODY over all trees in a 
tree structure."
  (let ((var (gensym "--dotrees--"))
        (spec-var (car spec))
        (array (cadr spec)))
    (` (loop for (, var) being the symbols of (, array)
	     as (, spec-var) = (get (, var) 'browse-root) do
	     (when (vectorp (, spec-var))
	       (,@ body))))))

(put 'dotrees 'lisp-indent-hook 1)

;;;
;;; Get screen or frame width depending on emacs version
;;; 

(defmacro browse-frame-width ()
  (` (if (browse-emacs-19-p)
         (frame-width)
       (screen-width))))

;;;
;;; Move to column COLUMN by inserting spaces and tabs.
;;; 

(defmacro browse-move-to-column (column)
  (` (indent-to (, column))))

;;;
;;; Replace all occurrences of OLD-LETTER by
;;; NEW-LETTER in string STR.  OLD-LETTER and NEW-LETTER have
;;; to be strings.
;;; 

(defmacro browse-replace-letter (str old new)
  (` (mapconcat (function
                 (lambda (c)
                   (let ((s (char-to-string c)))
                     (if (string= s (, old)) (, new) s))))
                (, str) "")))

;;;
;;; Return the line the point is in.
;;; 

(defmacro browse-current-line ()
  (` (+ (count-lines (point-min) (point))
        (if (zerop (current-column)) 1 0)
        -1)))

;;;
;;; Perform BODY with COMPLETION-IGNORE-CASE set to T. Currently, this
;;; macro relies on Emacs Lisp's dynamic scoping. It will be changed
;;; when (let's hope) it changes to lexical scoping.
;;; 

(defmacro browse-completion-ignoring-case (&rest body)
  (` (let ((completion-ignore-case t))
       (,@ body))))

(put 'browse-completion-ignoring-case 'lisp-indent-hook 0)

;;;
;;; Perform BODY with SELECTIVE-DISPLAY restored to old
;;; value after BODY is done.
;;; 

(defmacro browse-save-selective (&rest body)
  (let ((var (gensym "--browse-save--")))
    (` (let (((, var) selective-display))
	 (unwind-protect (progn (,@ body))
	   (setq selective-display (, var)))))))

(put 'browse-save-selective 'lisp-indent-hook 0)

;;;
;;; Return T if the given TREE is the one for global functions, variables
;;; etc.
;;; 

(defmacro browse-global-tree-p (tree)
  (` (string= (class-name (tree-class (, tree))) browse-global-tree-name)))

(provide 'br-macro)
