;;; rexx.el  -  Major mode for editing rexx source in emacs.

;;; Copyright (C) 1993, 1994 Free Software Foundation, Inc.

;;; Author: Espen Skoglung, Ralf Grohmann (ralf@ubka.uni-karlsruhe.de)
;;; Keywords: languages

;;; This was originally PASCAL.EL by 
;;;          Espen Skoglund (espensk@stud.cs.uit.no)
;;; I (Ralf) have modified it to be useable with REXX-Programs
;;;   ... which means I have deleted most of the features of pascal-mode...

;;; This file is part of GNU Emacs.

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.

;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;;; USAGE
;;; =====

;;; Emacs should enter Rexx mode when you find a Rexx source file.
;;; When you have entered Rexx mode, you may get more info by pressing
;;; C-h m. You may also get online help describing various functions by:
;;; C-h f <Name of function you want described>

;;; KNOWN BUGS / BUGREPORTS
;;; =======================
;;; I suppose there are. Drop me a mail (ralf@ubka.uni-karlsruhe.de)
;;; or - EVEN BETTER - fix it and send me the diffs!

;;; Code:

(defconst rexx-mode-version "0.01a"
  "Version of `rexx-mode.el'.")

(defvar rexx-mode-abbrev-table nil
  "Abbrev table in use in Rexx-mode buffers.")
(define-abbrev-table 'rexx-mode-abbrev-table ())

(defvar rexx-mode-map ()
  "Keymap used in Rexx mode.")

(if rexx-mode-map
    ()
  (setq rexx-mode-map (make-sparse-keymap))
  (define-key rexx-mode-map ";"        'electric-rexx-semi-or-dot)
  (define-key rexx-mode-map ":"        'electric-rexx-colon)
  (define-key rexx-mode-map "="        'electric-rexx-equal)
  (define-key rexx-mode-map "\r"       'electric-rexx-terminate-line)
  (define-key rexx-mode-map "\t"       'electric-rexx-tab)
  (define-key rexx-mode-map "\177"     'backward-delete-char-untabify)
  (define-key rexx-mode-map "\C-cb"    'rexx-insert-block)
  (define-key rexx-mode-map "\M-*"     'rexx-star-comment)
  (define-key rexx-mode-map "\C-c\C-c" 'rexx-comment-area)
  (define-key rexx-mode-map "\C-c\C-u" 'rexx-uncomment-area)
;;; Commands to change the whole buffer 
  (define-key rexx-mode-map "\C-cd"    'rexx-downcase-keywords)
  (define-key rexx-mode-map "\C-cu"    'rexx-upcase-keywords)
  (define-key rexx-mode-map "\C-cc"    'rexx-capitalize-keywords)
  )
  

(defvar rexx-keywords
  '("address" "arg" "by" "call" "do" "drop" "end" "else" "exit" "expose" 
    "function" "forever" "if" "interpret" "iterate" "leave" "nop" "numeric" 
    "off" "on" "options" "otherwise" "parse" "procedure" "pull" "push" 
    "queue" "return" "say" "select" "signal" "source" "trace" "then" "to" 
    "until" "upper" "value" "var" "version" "when" "with" "while" 
    "abbrev" "abs" "beep" "bitand" "bitor" "bitxor" "b2x" "center" "charin" 
    "charout" "chars" "c2d" "c2x" "datatype" "date" "delstr" "delword" 
    "digits" "d2c" "d2x" "directory" "errortext" "endlocal" "filespec" 
    "form" "format" "fuzz" "insert" "lastpos" "left" "length" "linein" 
    "lineout" "lines" "max" "min" "overlay" "pos" "queued" "random" 
    "reverse" "right" "setlocal" "sign" "sourceline" "space" "stream"
    "strip" "substr" "subword" "symbol" "time" "translate" "trunc" 
    "value" "verify" "word" "wordindex" "wordlength" "wordpos" "words" 
    "xrange" "x2b" "x2c" "x2d" "rxqueue" ))

;;; Strings used to mark beginning and end of excluded text
(defconst rexx-exclude-str-start "/\*--- EXCLUDED ---")
(defconst rexx-exclude-str-end " --- EXCLUDED ---\*/")

(defvar rexx-mode-syntax-table nil
  "Syntax table in use in Rexx-mode buffers.")
(if rexx-mode-syntax-table
    ()
  (setq rexx-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\\ "\\"  rexx-mode-syntax-table)
  (modify-syntax-entry ?( "()"  rexx-mode-syntax-table)  
  (modify-syntax-entry ?) ")("  rexx-mode-syntax-table)
  (modify-syntax-entry ?/ ". 14"  rexx-mode-syntax-table)  
  (modify-syntax-entry ?* ". 23" rexx-mode-syntax-table)
  (modify-syntax-entry ?+ "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?- "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?= "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?% "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?< "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?> "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?& "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?| "."    rexx-mode-syntax-table)
  (modify-syntax-entry ?_ "w"    rexx-mode-syntax-table)
  (modify-syntax-entry ?\' "\""  rexx-mode-syntax-table))

(defvar rexx-indent-level 3
  "*Indentation of Rexx statements with respect to containing block.")
(defvar rexx-auto-newline nil
  "*Non-nil means automatically newline after semicolons
after an end.")

;;;
;;;  Macros
;;;

(defsubst rexx-get-beg-of-line (&optional arg)
  (save-excursion
    (beginning-of-line arg)
    (point)))

(defsubst rexx-get-end-of-line (&optional arg)
  (save-excursion
    (end-of-line arg)
    (point)))
  
(defsubst rexx-within-string ()
  (save-excursion
    (nth 3 (parse-partial-sexp (rexx-get-beg-of-line) (point)))))


;;;###autoload
(defun rexx-mode ()
  "Major mode for editing Rexx code. \\<rexx-mode-map>

Useful functions are:
\\[rexx-insert-block]\t- insert do ... end;
\\[rexx-star-comment]\t- insert /* ...*... */
\\[rexx-comment-area]\t- Put marked area in a comment
\\[rexx-uncomment-area]\t- Uncomment an area commented with \
\\[rexx-comment-area].
\\[rexx-downcase-keywords] - Make all Keywords lower case
\\[rexx-upcase-keywords] - As above
\\[rexx-capitalize-keywords] - As above

 rexx-auto-newline      (default nil)
    Non-nil means automatically newline after simcolons 
    after an end.

Turning on Rexx mode calls the value of the variable rexx-mode-hook with
no args, if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map rexx-mode-map)
  (setq major-mode 'rexx-mode)
  (setq mode-name "Rexx")
  (setq local-abbrev-table rexx-mode-abbrev-table)
  (set-syntax-table rexx-mode-syntax-table)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments t)
  (make-local-variable 'case-fold-search)
  (setq case-fold-search t)
  (run-hooks 'rexx-mode-hook))



;;;
;;;  Electric functions
;;;
(defun electric-rexx-terminate-line ()
  "Terminate line."
  (interactive)
  (delete-horizontal-space) ; Removes trailing whitespaces
  (newline)
  )


(defun electric-rexx-semi-or-dot ()
  "Insert `;' or `.' character."
  (interactive)
  (insert last-command-char)
  (save-excursion
    (beginning-of-line)
  (if rexx-auto-newline
      (electric-rexx-terminate-line))))

(defun electric-rexx-colon ()
  "Insert `:'."
  (interactive)
  (insert last-command-char)
  ;; Do nothing if within string.
  (if (rexx-within-string)
      ()
    (save-excursion
      (beginning-of-line)
    )))

(defun electric-rexx-equal ()
  "Insert `='."
  (interactive)
  (insert last-command-char)
  )

(defun electric-rexx-tab ()
  "Function called when TAB is pressed in Rexx mode."
  (interactive)
  ;; Do nothing if within a string.
  (if (rexx-within-string)
      (insert "\t")
    ) )



;;;
;;; Interactive functions
;;;
(defun rexx-insert-block ()
  "Insert Rexx do ... end; block in the code with right indentation."
  (interactive)
  (insert "do")
  (electric-rexx-terminate-line)
  (save-excursion
    (insert "end;")
    (beginning-of-line)
    ))

;;; RalfG: Geht
(defun rexx-star-comment ()
  "Insert Rexx star comment at point."
  (interactive)
  (insert "/*")
  (electric-rexx-terminate-line)
  (insert "* ")
(save-excursion
  (electric-rexx-terminate-line)
  (delete-horizontal-space)
  (insert "*/"))
  (insert "  "))


(defun rexx-comment-area (start end)
  "Put the region into a Rexx comment.

The commented area starts with `rexx-exclude-str-start', and ends with
`rexx-include-str-end'.  But if you change these variables,
\\[rexx-uncomment-area] won't recognize the comments."
  (interactive "r")
  (save-excursion
    ;; Insert start and endcomments
    (goto-char end)
    (if (and (save-excursion (skip-chars-forward " \t") (eolp))
	     (not (save-excursion (skip-chars-backward " \t") (bolp))))
	(forward-line 1)
      (beginning-of-line))
    (insert rexx-exclude-str-end)
    (setq end (point))
    (newline)
    (goto-char start)
    (beginning-of-line)
    (insert rexx-exclude-str-start)
    (newline)
    ;; Replace end-comments within commented area
;;  (goto-char end)
;;  (save-excursion
;;    (while (re-search-backward "\\*/" start t)
;;	(replace-match "!/*" t t)))
	))

(defun rexx-uncomment-area ()
  "Uncomment a commented area.
This command does nothing if the pointer is not in a commented
area.  See also `rexx-comment-area'."
  (interactive)
  (save-excursion
    (let ((start (point))
	  (end (point)))
      ;; Find the boundaries of the comment
      (save-excursion
	(setq start (progn (search-backward rexx-exclude-str-start nil t)
			   (point)))
	(setq end (progn (search-forward rexx-exclude-str-end nil t)
			 (point))))
      ;; Check if we're really inside a comment
      (if (or (equal start (point)) (<= end (point)))
	  (message "Not standing within commented area.")
	(progn
	  ;; Remove endcomment
	  (goto-char end)
	  (beginning-of-line)
	  (let ((pos (point)))
	    (end-of-line)
	    (delete-region pos (1+ (point))))
	  ;; Change comments back to normal
;;	  (save-excursion
;;	    (while (re-search-backward "!/\\*" start t)
;;	      (replace-match "*/" t t)))
	  ;; Remove startcomment
	  (goto-char start)
	  (beginning-of-line)
	  (let ((pos (point)))
	    (end-of-line)
	    (delete-region pos (1+ (point)))))))))


(defun rexx-downcase-keywords ()
  "Downcase all Rexx keywords in the buffer."
  (interactive)
  (rexx-change-keywords 'downcase-word))

(defun rexx-upcase-keywords ()
  "Upcase all Rexx keywords in the buffer."
  (interactive)
  (rexx-change-keywords 'upcase-word))

(defun rexx-capitalize-keywords ()
  "Capitalize all Rexx keywords in the buffer."
  (interactive)
  (rexx-change-keywords 'capitalize-word))

;; Change the keywords according to argument.
(defun rexx-change-keywords (change-word)
  (save-excursion
    (let ((keyword-re (concat "\\<\\("
			      (mapconcat 'identity rexx-keywords "\\|")
			      "\\)\\>")))
      (goto-char (point-min))
      (while (re-search-forward keyword-re nil t)
	(funcall change-word -1)))))

;;; rexx.el ends here
