/*

    NTBIND.C - converts a.out/coff files to Win32 PE-format

        supported compilers/format/system:

            - EMX,RSXNT a.out   DOS,OS/2,Win32
            - DJGPP     coff    DOS
            - Gcc       coff    Linux

        gcc must support:

            - __attribute__((stdcall))
            - relocations in output file (required for DLL's and Win32s)
                (patch coff LD for relocated output)

        gcc should also support:

            - stack probes  (emx: -mprobe; gcc 2.8: -fstack-check)
            - pragma pack() (Win32 SDK)
            - C++ comments in C-headers (Win32 SDK; gcc 2.7.x ok)

*/

/* ntbind.c - converts a.out/coff files to Win32 PE-format

   Copyright (c) 1995-1999 Rainer Schnitker

   This file is part of RSXNT.

   RSXNT is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   RSXNT is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with RSXNT; see the file COPYING.  If not, write to
   the Free Software Foundation, 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA. */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>
#include <fcntl.h>
#include "port.h"
#include "moddef.h"
#include "ntbind.h"

#define RSXNT_VERSION "NTBIND version 1.50 (c) 1996-1999 Rainer Schnitker"

#define COFF_SECTIONS 6
#define TEXT        0
#define DATA        1
#define DEBUG       2
#define IDATA       3
#define EDATA       4
#define RELOCSEC    5

/* NT must have a strict segment align */
static unsigned long segment_size;
#undef SEGMENT_SIZE
#define SEGMENT_SIZE segment_size

#define SECTION_SIZE     segment_size  /* win32 = 0x1000 */
#define FILEALIGN        0x200
#define ROUND_PAGE(x)    (((x)+0xFFF)&~0xFFF)
#define ROUND_SECTION(x) (((x)+(SECTION_SIZE-1))&~(SECTION_SIZE-1))
#define ROUND_FILE(x)    (((x)+(FILEALIGN-1))&~(FILEALIGN-1))

/****************************************************************************\
 *                                                                          *
 *   prototypes                                                             *
 *                                                                          *
\****************************************************************************/

static unsigned long aout_find_export_symbol (char *export);
static unsigned long coff_find_export_symbol (char *export);

/****************************************************************************\
 *                                                                          *
 *   helper functions                                                       *
 *                                                                          *
\****************************************************************************/

static void out_of_mem(void)
{
    write(1 , "out of memory", 13);
    exit(1);
}

static void *xmalloc (size_t n)
{
    long *q;
    q = malloc (n);
    if (q == NULL)
        out_of_mem ();
    return q;
}

static void *xrealloc (void *p, size_t n)
{
    long *q;
    q = realloc (p, n);
    if (q == NULL)
        out_of_mem ();
    return (q);
}

static char *xstrdup (const char *p)
{
    char *q;

    q = strdup (p);
    if (q == NULL)
        out_of_mem ();
    return (q);
}

static void error(char *err)
{
    puts(err);
    exit(1);
}

/****************************************************************************\
 *                                                                          *
 *   internal structures / functions for linked lists                       *
 *                                                                          *
\****************************************************************************/

typedef struct exportlist {
    char *  name;
    DWORD   addr;
    int     ord;
} EXPORT_LIST;

typedef struct importlist {
    char *  name;
    DWORD   addr;
    int     hint;
    DWORD   impsec_off;
} IMPORT_LIST;

typedef struct dlllist {
    char *      name;
    int         entries;
    IMPORT_LIST *thunknames;
} DLL_LIST;

typedef struct {
    int     size;
    int     lastoff;
    char    *mem;
} RELOC_LIST;

typedef struct relocs{
        DWORD   vaddr;
        DWORD   size;
        WORD    relocs[1];
} IMAGE_BASERELOC;

typedef enum {
    MODE_UNDEF, MODE_EXE, MODE_DLL
} OUTPUT_MODE;
typedef enum {
    SUB_UNDEF = 0, SUB_NATIVE, SUB_GUI, SUB_CUI, SUB_RES1, SUB_OS2, SUB_POSIX
} SUBSYSTEM;

typedef struct tagdefaults {
    char        *pename;
    char        *aoutname;
    char        *stubname;
    char        *description;
    long        heapsize;
    long        stacksize;
    SUBSYSTEM   win32_subsystem;
    OUTPUT_MODE output_mode;
    char        opt_nosyscall;
    char        opt_printall;
    char        opt_emxstyle;
    char        opt_strip;
} DEFAULT_PARAM;

DEFAULT_PARAM default_param = {
    NULL, NULL, "dosstub.dos", NULL,
    0x400000, 0x400000,
    SUB_CUI, MODE_EXE,
    0,0,0
};

static struct {
    long emx, aout, pe, image;
} pe_off;

static int emx_source = 1;

static char MagicImport[] = "__ImpNT";

static EXPORT_LIST *exp_list;
static int nexp=0;
static int expmin=0xffff;
static int expmax=0;

static DLL_LIST *dll_list;
static int ndll=0;

static RELOC_LIST reloc_list;

static char *impbuf;
static int impbuf_size;
static int secimp_size = sizeof(IMP_DESC);
static int secimp_strsize = 0;

static int secexp_strsize = 0;

/****************************************************************************\
 *                                                                          *
 *   memory functions                                                       *
 *                                                                          *
\****************************************************************************/

static void init_export_entry()
{
    exp_list = (EXPORT_LIST *) xmalloc(sizeof(EXPORT_LIST));
}

static void add_export_entry(char *name, int ord)
{
    exp_list = xrealloc(exp_list, sizeof(EXPORT_LIST) * ++nexp);
    exp_list[nexp-1].name = name;
    exp_list[nexp-1].addr = 0;
    exp_list[nexp-1].ord = ord;

    secexp_strsize += strlen(name) + 1;

    if (expmin > ord)
        expmin = ord;
    if (expmax < ord)
        expmax = ord;
}

static void init_dll_entry(void)
{
    dll_list = (DLL_LIST *) xmalloc(sizeof(DLL_LIST));
    dll_list[0].entries = 0;
}

static void add_dll_entry(char *dll, char *name, DWORD addr, int hint)
{
    int i, n;

    /* search for dll name */
    for (i = 0; i< ndll; i++)
        if (strcmp(dll, dll_list[i].name) == 0)
            break;

    /* not found, add new */
    if (i >= ndll) {
        secimp_size += 2 * sizeof(DWORD);       /* add NULL thunk */
        secimp_size += sizeof(IMP_DESC);
        secimp_strsize += strlen(dll)+1;
        i = ndll++;
        dll_list = (DLL_LIST *) xrealloc(dll_list, ndll * sizeof(DLL_LIST));
        dll_list[i].name = dll;
        dll_list[i].entries = 0;
        dll_list[i].thunknames = (IMPORT_LIST *) xmalloc(sizeof(IMPORT_LIST));
    }

    /* add function to list */
    secimp_size += 2 * sizeof(DWORD);
    secimp_strsize += strlen(name)+1+sizeof(WORD); /* Hint */
    n = dll_list[i].entries ++;
    dll_list[i].thunknames = (IMPORT_LIST *)
            xrealloc(dll_list[i].thunknames, (n+1) * sizeof(IMPORT_LIST));
    dll_list[i].thunknames[n].name = name;
    dll_list[i].thunknames[n].addr = addr;
    dll_list[i].thunknames[n].hint = hint;
}

static void init_reloc_entry(void)
{
    DWORD *dw;
    reloc_list.mem = xmalloc(8);
    reloc_list.size = 0;
    reloc_list.lastoff = 0;
    dw = (DWORD *) reloc_list.mem;
    dw[0] = 0;
    dw[1] = 0;
}

static void add_reloc_entry(DWORD addr)
{
    static WORD HighLow = 0x3000;
    WORD *wd;
    DWORD *dw;

    dw = (DWORD *) (reloc_list.mem + reloc_list.lastoff);
    if (dw[0] != (addr & ~0xfff)) {
        if (reloc_list.size % 4) {
            HighLow = 0;
            add_reloc_entry(dw[0]);
            HighLow = 0x3000;
        }
        reloc_list.size += 8;
        reloc_list.mem = xrealloc(reloc_list.mem, reloc_list.size);
        reloc_list.lastoff = reloc_list.size - 8;
        dw = (DWORD *) (reloc_list.mem + reloc_list.lastoff);
        dw[0] = addr & ~0xfff;
        dw[1] = 8;
    }
    dw[1]+=2;
    reloc_list.size+=2;
    reloc_list.mem = xrealloc(reloc_list.mem, reloc_list.size);
    wd = (WORD *) (reloc_list.mem + reloc_list.size - 2);
    *wd = HighLow | (addr & 0xfff);
}

static void init_all_lists(void)
{
    init_dll_entry();
    init_export_entry();
    init_reloc_entry();
}

/****************************************************************************\
 *                                                                          *
 *  file headers / functions to add values                                  *
 *                                                                          *
\****************************************************************************/

static struct exec_info ihdr;                       /* portable header */
static FILEHDR          file_hdr;                   /* coff */
static NTOPTHDR         ntopt_hdr;                  /* coff */
static SCNHDR           scnd_hdr[COFF_SECTIONS];    /* coff */
static struct exec      gnu_hdr;                    /* a.out */
static unsigned long    BaseOfImage;                /* for '-defbase' */

static char *          symstr;
static struct nlist *  nl;

static void handle_stabs(int fhandle)
{
    int i;
    int win32_linked = 0;
    int sysdll = 0;
    long this_off = 0, next_off = 0;

    /* search nlist Entries for import calls */
    for (i = 0; i < ihdr.sym_entries; i++) {

        if (!emx_source && !nl[i].n_type) {
            this_off += next_off;
            next_off = nl[i].n_value;
        }

        if (nl[i].n_type == 36 || nl[i].n_type == 38) { /* function/data */
            char *name = symstr + nl[i].n_un.n_strx + this_off;

            if (memcmp(name, MagicImport, sizeof(MagicImport)-1) == 0) {
                /* '__ImpNT_Function=DLL.Ord' Entries */
                char *dll;

                if (!(dll=strchr(name, '='))) {
                    printf("bad import entry '%s'\n", name);
                    exit(1);
                }
                *dll++ = 0;
                /* forget underscore in function (not size -1) */
                if (strcmp(dll, "RSXNT.dll") == 0)
                    sysdll = 1;

                if (!(sysdll && default_param.opt_nosyscall))
                    add_dll_entry(dll, name+sizeof(MagicImport), nl[i].n_value, nl[i].n_desc);
                else
                    printf("-nosyscall: exclude DLL '%s' at %lX\n", dll, nl[i].n_value);
            }
        } else if (nl[i].n_type == 5) {         /* function, test WINMAIN */
            char *name = symstr + nl[i].n_un.n_strx + this_off;
            if (strcmp(name, "_WinMain") == 0 || strcmp(name, "_LibMain") == 0)
                ntopt_hdr.Subsystem = default_param.win32_subsystem = SUB_GUI;
        } else if (nl[i].n_type == 6) {         /* test __win32_app */
            char *name = symstr + nl[i].n_un.n_strx + this_off;
            if (strcmp(name, "__win32_app") == 0)
                win32_linked = 1;
        }
    }

    if (emx_source)
    if (!(sysdll && default_param.opt_nosyscall) && !win32_linked) {
        puts("This programs is not link with win32.a");
        exit(1);
    }

    /* search nlist entries for export statements */
    if (nexp) {
        int e;
        for (e = 0; e < nexp; ++e) {
            if (default_param.opt_printall)
                printf("Search export entry %s\n", exp_list[e].name);

            exp_list[e].addr = (emx_source) ?
                        aout_find_export_symbol (exp_list[e].name)
                        : coff_find_export_symbol (exp_list[e].name);

            if (!exp_list[e].addr)
                printf("error cannot find symbol %s\n", exp_list[e].name);
        }
    }
    impbuf_size = ROUND_FILE(secimp_size + secimp_strsize);
}

static void print_entries(int fhandle)
{
    int i;
    WORD *wd;


    printf("\nIMPORT ENTRIES:\n");
    for (i=0; i<ndll; i++) {
        int j;
        IMPORT_LIST *imp_list;

        printf("DLL = %s Number of Functions %d\n",
            dll_list[i].name,
            dll_list[i].entries
            );

        imp_list = dll_list[i].thunknames;
        for (j = 0; j < dll_list[i].entries; j++)
            printf("\tfunction %40s addr = %8lX\n"
                    , imp_list[j].name
                    , imp_list[j].addr);
    }
    printf("imp size %d strings %d\n", secimp_size, secimp_strsize);

    printf("\nEXPORT ENTRIES:\n");
    for (i=0; i<nexp; i++)
        printf("%20s %lX %3d\n",
            exp_list[i].name,
            exp_list[i].addr,
            exp_list[i].ord);

    printf("\nRELOC ENTRIES size = %d:\n", reloc_list.size);
    wd = (WORD *) reloc_list.mem;
    for (;;) {
        DWORD *dw = (DWORD *)(wd);
        if ((int)dw >= (int)(reloc_list.mem + reloc_list.size))
            break;
        printf("vaddr %lX entries %lX\n", dw[0], (dw[1]-8)/2);
        wd += 4;
        for (i = 0; i < (dw[1]-8)/2; i++, wd++) {
            printf("offset %X\n", *wd & 0xfff);
        }
    }
}

/* write default values to PE-header */

static void init_nthdr(int fhandle)
{
    struct exe_hdr exehdr;
    int doshandle;
    long doslen;
    struct timeb mytime;

    if ((doshandle = open(default_param.stubname,
                        O_RDONLY | O_BINARY)) == -1) {
        perror("open dos stubname");
        exit(1);
    }
    doslen = filelength(doshandle);
    read(doshandle, &exehdr, sizeof(struct exe_hdr));
    if (exehdr.signatur == 0x5a4d) {
        struct emx_hdr emxhdr;
        pe_off.emx  = ((DWORD) exehdr.hdr_para) * 16;
        lseek(doshandle, pe_off.emx , SEEK_SET);
        read(doshandle, &emxhdr, sizeof(emxhdr));
        if (memcmp(emxhdr.sig, "emx", 3) != 0)
            pe_off.emx = 0;
    }
    close(doshandle);

    if (pe_off.emx) {   /* build bound file */
        pe_off.aout = ROUND_FILE(doslen);
        pe_off.pe   = pe_off.aout + sizeof(struct exec);
        pe_off.image = pe_off.aout + 1024;
        default_param.opt_emxstyle = 1;
    } else {
        pe_off.aout = 0;
        pe_off.pe   = (doslen + 15) & ~15;
        pe_off.image = (doslen + 0x700) & ~0x3FF;
        default_param.opt_emxstyle = 0;
    }

    ftime(&mytime);

    file_hdr.f_magic = 0x14C;
    file_hdr.f_nscns = 2;
    file_hdr.f_timdat = mytime.time + mytime.timezone;
    file_hdr.f_symptr = 0;
    file_hdr.f_nsyms = 0;
    file_hdr.f_opthdr = 0xE0;
    file_hdr.f_flags = 0x010E;
    if (default_param.output_mode == MODE_DLL)
        file_hdr.f_flags |= 0x2000; /* IMAGE_FILE_DLL */
    if (reloc_list.size == 0)
        file_hdr.f_flags |= 0x0001; /* IMAGE_FILE_RELOCS_STRIPPED */

    /* set names */
    strcpy(scnd_hdr[TEXT].s_name, ".text");
    strcpy(scnd_hdr[DATA].s_name, ".data");

    /* set paddr = virtual size of section (Win32) */
    scnd_hdr[TEXT].s_paddr = ihdr.text_size;
    scnd_hdr[DATA].s_paddr = ihdr.data_size + ihdr.bss_size;

    scnd_hdr[TEXT].s_size = ihdr.text_size;
    scnd_hdr[DATA].s_size = ihdr.data_size;

    /* set scnptr = real size of section (NT) */
    scnd_hdr[TEXT].s_scnptr = pe_off.image;
    scnd_hdr[DATA].s_scnptr = pe_off.image + ihdr.text_size;

    /* set virtual addr */
    scnd_hdr[TEXT].s_vaddr = ihdr.text_vaddr;
    scnd_hdr[DATA].s_vaddr = ihdr.data_vaddr;

    /* set flags */
    scnd_hdr[TEXT].s_flags  = 0x60000020L;      /* read, exe  ; code */
    scnd_hdr[DATA].s_flags  = 0xC0000040L;      /* read, write; data */
    scnd_hdr[RELOCSEC].s_flags = 0x42000040L;   /* read, discd; data */
    scnd_hdr[IDATA].s_flags = 0x40000040L;      /* read, data */
    scnd_hdr[EDATA].s_flags = 0xC0000040L;      /* read, write; data */
    scnd_hdr[DEBUG].s_flags = 0x42000040L;      /* read, discd; data */

    /* Standard fields */
    ntopt_hdr.Magic  = 0x10B;
    ntopt_hdr.MajorLinkerVersion = 3;
    ntopt_hdr.MinorLinkerVersion = 10;
    ntopt_hdr.SizeOfCode = ihdr.text_size;
    ntopt_hdr.SizeOfInitializedData  = ihdr.data_size + ROUND_FILE(ihdr.bss_size);
    ntopt_hdr.SizeOfUninitializedData = 0;
    ntopt_hdr.AddressOfEntryPoint = ihdr.entry - BaseOfImage;
    ntopt_hdr.BaseOfCode = scnd_hdr[TEXT].s_vaddr;
    ntopt_hdr.BaseOfData = scnd_hdr[DATA].s_vaddr;
    /* NT additional fields */
    ntopt_hdr.ImageBase = BaseOfImage;
    ntopt_hdr.SectionAlignment = SECTION_SIZE;
    ntopt_hdr.FileAlignment = FILEALIGN;
    ntopt_hdr.MajorOperatingSystemVersion = 0x01;
    ntopt_hdr.MinorOperatingSystemVersion = 0x00;
    ntopt_hdr.MajorImageVersion = 0x01;
    ntopt_hdr.MinorImageVersion = 0x00;
    ntopt_hdr.MajorSubsystemVersion = 0x04;     /* 3.10 = old style */
    ntopt_hdr.MinorSubsystemVersion = 0x00;     /* 4.00 = new style */
    ntopt_hdr.Reserved1 = 0;
    ntopt_hdr.SizeOfImage = 0;                  /* T+D+B+R+I+E */
    ntopt_hdr.SizeOfHeaders = pe_off.image;     /* DOS + NT headers */
    ntopt_hdr.CheckSum = 0;
    ntopt_hdr.Subsystem = default_param.win32_subsystem; /* 2=Gui 3=Char */
    ntopt_hdr.DllCharacteristics = 0;
    ntopt_hdr.SizeOfStackReserve = 0x100000;    /* 1 MB stack */
    ntopt_hdr.SizeOfStackCommit = 0x2000;       /* commit stack = 8 KB */
    ntopt_hdr.SizeOfHeapReserve = 0x100000;     /* heap = VirtalAlloc() */
    ntopt_hdr.SizeOfHeapCommit = 0x1000;
    ntopt_hdr.LoaderFlags = 0;
    ntopt_hdr.NumberOfRvaAndSizes = 16;
}

/****************************************************************************\
 *                                                                          *
 * fill PE-herder with debug, idata, export and reloc section               *
 *                                                                          *
\****************************************************************************/

void fill_ntheader(void)
{
    DWORD vaddr, fileptr;

    vaddr = ROUND_SECTION(ihdr.data_vaddr + ihdr.data_size + ihdr.bss_size);
    fileptr = pe_off.image + ihdr.text_size + ihdr.data_size;

    if (default_param.opt_emxstyle && !default_param.opt_strip) {
        strcpy(scnd_hdr[DEBUG].s_name , "EMXDBG");

        file_hdr.f_nscns ++;

        scnd_hdr[DEBUG].s_paddr = ihdr.sym_size + ihdr.str_size;
        scnd_hdr[DEBUG].s_vaddr = vaddr;
        scnd_hdr[DEBUG].s_size = ROUND_FILE(scnd_hdr[DEBUG].s_paddr);
        scnd_hdr[DEBUG].s_scnptr = fileptr;

        vaddr += ROUND_SECTION(scnd_hdr[DEBUG].s_size);
        fileptr += scnd_hdr[DEBUG].s_size;
        ntopt_hdr.SizeOfInitializedData += scnd_hdr[DEBUG].s_size;
    }

    if (ndll) {
        strcpy(scnd_hdr[IDATA].s_name , ".idata");

        file_hdr.f_nscns ++;

        scnd_hdr[IDATA].s_paddr = impbuf_size;
        scnd_hdr[IDATA].s_vaddr = vaddr;
        scnd_hdr[IDATA].s_size = ROUND_FILE(impbuf_size);
        scnd_hdr[IDATA].s_scnptr = fileptr;

        ntopt_hdr.DataDirectory[DIR_IMPORT].VirtualAddress =
            scnd_hdr[IDATA].s_vaddr;
        ntopt_hdr.DataDirectory[DIR_IMPORT].Size =
            scnd_hdr[IDATA].s_paddr;

        vaddr += ROUND_SECTION(scnd_hdr[IDATA].s_size);
        fileptr += scnd_hdr[IDATA].s_size;
        ntopt_hdr.SizeOfInitializedData += scnd_hdr[IDATA].s_size;
    }

    if (nexp) {
        int nOrdinals = expmax - expmin + 1;
        DWORD expbuf_size = sizeof(EXP_DESC)
                    + nexp * 6                                      // ord+name
                    + ((nOrdinals > nexp) ? nOrdinals : nexp) * 4   // funcs
                    + strlen(default_param.pename) + 1 + secexp_strsize;

        strcpy(scnd_hdr[EDATA].s_name , ".edata");
        file_hdr.f_nscns ++;

        scnd_hdr[EDATA].s_paddr = expbuf_size;
        scnd_hdr[EDATA].s_size = ROUND_FILE(expbuf_size);
        scnd_hdr[EDATA].s_vaddr = vaddr;
        scnd_hdr[EDATA].s_scnptr = fileptr;

        ntopt_hdr.DataDirectory[DIR_EXPORT].VirtualAddress =
            scnd_hdr[EDATA].s_vaddr;
        ntopt_hdr.DataDirectory[DIR_EXPORT].Size =
            scnd_hdr[EDATA].s_paddr;

        vaddr += ROUND_SECTION(scnd_hdr[EDATA].s_size);
        fileptr += scnd_hdr[EDATA].s_size;
        ntopt_hdr.SizeOfInitializedData += scnd_hdr[EDATA].s_size;
    }

    if (reloc_list.size) {
        strcpy(scnd_hdr[RELOCSEC].s_name , ".reloc");

        file_hdr.f_nscns ++;

        scnd_hdr[RELOCSEC].s_vaddr = vaddr;
        scnd_hdr[RELOCSEC].s_size = ROUND_FILE(reloc_list.size);
        scnd_hdr[RELOCSEC].s_scnptr = fileptr;
        scnd_hdr[RELOCSEC].s_paddr = reloc_list.size;

        ntopt_hdr.DataDirectory[DIR_BASERELOC].VirtualAddress =
            scnd_hdr[RELOCSEC].s_vaddr;
        ntopt_hdr.DataDirectory[DIR_BASERELOC].Size =
            scnd_hdr[RELOCSEC].s_paddr;

        vaddr += ROUND_SECTION(scnd_hdr[RELOCSEC].s_size);
        fileptr += scnd_hdr[RELOCSEC].s_size;
        ntopt_hdr.SizeOfInitializedData += scnd_hdr[RELOCSEC].s_size;
    }

    ntopt_hdr.SizeOfImage = vaddr;
}


/****************************************************************************\
 *                                                                          *
 *   functions to write PE-file                                             *
 *                                                                          *
\****************************************************************************/

static int export_compare(const void *x1, const void *x2)
{
    EXPORT_LIST *e1, *e2;

    e1 = (EXPORT_LIST *) x1;
    e2 = (EXPORT_LIST *) x2;

    return strcmp(e1->name, e2->name);
}

static int export_compare_ord(const void *x1, const void *x2)
{
    EXPORT_LIST *e1, *e2;

    e1 = (EXPORT_LIST *) x1;
    e2 = (EXPORT_LIST *) x2;

    if (e1->ord == e2->ord)
        return 0;
    else if (e1->ord > e2->ord)
        return 1;
    else
        return -1;
}

static void write_export(int pehandle)
{
    char *expbuf;
    EXP_DESC *expd;
    unsigned int str_off, func_off, names_off, ord_off, outname_off;
    DWORD *names;
    DWORD *functions;
    WORD *ord;
    int i;
    DWORD vaddr = scnd_hdr[EDATA].s_vaddr;
    DWORD size;
    EXPORT_LIST * funcs_by_ord;
    int nOrdinals = expmax - expmin + 1;
    int iMustResort = 0;

    qsort(exp_list, nexp, sizeof(EXPORT_LIST), export_compare);

    /* test all export entries */
    for (i = 0; i < nexp; i++) {
        if (exp_list[i].addr == 0) {
            printf("can't find function %s in symbol list\n", exp_list[i].name);
            exit(1);
        }
        if (nOrdinals == 1)
            exp_list[i].ord = i + 1;
        else if (exp_list[i].ord != i + 1)
            iMustResort = 1;
    }

    if (nOrdinals == 1)
        nOrdinals = nexp;
    else if (nOrdinals < nexp)
    {
        puts ("Error: less ordinal numbers than exported functions");
        exit(1);
    }

    func_off = sizeof(EXP_DESC);
    names_off = func_off + nOrdinals * 4;
    ord_off = names_off + nexp * 4;
    outname_off = ord_off + nexp * 2;
    str_off = outname_off + strlen(default_param.pename) + 1;

    size = ROUND_FILE(str_off + secexp_strsize);

    expbuf = xmalloc(size);
    memset(expbuf, 0, size);

    expd = (EXP_DESC *) expbuf;
    expd->Characteristics = 0;
    expd->TimeDateStamp = 0;
    expd->MajorVersion = 0;
    expd->MinorVersion = 0;
    expd->Name = vaddr + outname_off;
    expd->Base = (expmin == 0) ? 1 : expmin;
    expd->NumberOfFunctions = nOrdinals;
    expd->NumberOfNames = nexp;
    expd->AddressOfFunctions = (DWORD **)(vaddr + func_off);
    expd->AddressOfNames = (DWORD **)(vaddr + names_off);
    expd->AddressOfNameOrdinals = (WORD **)(vaddr + ord_off);

    strcpy(expbuf + outname_off, default_param.pename);

    functions = (DWORD *) (expbuf + func_off);
    names = (DWORD *) (expbuf + names_off);
    ord = (WORD *) (expbuf + ord_off);

    for (i = 0; i < nexp; i++) {
        int l;
        functions[i] = exp_list[i].addr;
        ord[i] = exp_list[i].ord - expd->Base;
        names[i] = vaddr + str_off;
        l = strlen(exp_list[i].name) + 1;
        memcpy(expbuf + str_off, exp_list[i].name, l);
        str_off += l;
    }

    if (iMustResort) /* resort function addresses in ordinal order */
    {
        funcs_by_ord = (EXPORT_LIST *) malloc (sizeof(EXPORT_LIST) * nOrdinals);
        memset (funcs_by_ord, 0, sizeof(EXPORT_LIST) * nOrdinals);
        for (i = 0; i < nexp; ++i) {
            funcs_by_ord[ord[i]].addr = functions[i];
            funcs_by_ord[ord[i]].ord = ord[i];
        }
        for (i = 0; i < nOrdinals; ++i)
            if (!funcs_by_ord[i].ord)
                funcs_by_ord[i].ord = i;
        qsort(funcs_by_ord, nOrdinals, sizeof(EXPORT_LIST), export_compare_ord);
        for (i = 0; i < nOrdinals; i++)
            functions[i] = funcs_by_ord[i].addr;
        free (funcs_by_ord);
    }

    write(pehandle, expbuf, size);
    free(expbuf);
}

#if 0
static int import_compare(const void *x1, const void *x2)
{
    IMPORT_LIST * i1 = (IMPORT_LIST *) x1;
    IMPORT_LIST * i2 = (IMPORT_LIST *) x2;

    return strcmp(i1->name, i2->name);
}
        IMPORT_LIST *imp_list = dll_list[i].thunknames;
        qsort(imp_list, dll_list[i].entries, sizeof(IMPORT_LIST), import_compare);
#endif

static void write_import(int pehandle)
{
    IMP_DESC *impd;
    unsigned int str_off,thunk_off;
    int i, j, l;
    DWORD vaddr = scnd_hdr[IDATA].s_vaddr;

    impbuf = xmalloc(impbuf_size);
    memset(impbuf, 0, impbuf_size);

    impd = (IMP_DESC *) impbuf;
    thunk_off = (ndll+1) * sizeof(IMP_DESC);
    str_off = secimp_size;

    for (i=0; i<ndll; i++) {
        l = strlen(dll_list[i].name) + 1;
        memcpy(impbuf + str_off, dll_list[i].name, l);
        impd->TimeDateStamp = 0;
        impd->ForwarderChain = 0;
        impd->Name = vaddr + str_off;
        impd->FirstThunk = vaddr + thunk_off;
        impd->Characteristics = impd->FirstThunk + (dll_list[i].entries+1) * 4;
        impd++;
        thunk_off += 2 * (dll_list[i].entries + 1) * sizeof(DWORD);
        str_off += l;
    }
    impd->Characteristics = 0;
    impd->TimeDateStamp = 0;
    impd->ForwarderChain = 0;
    impd->Name = 0;
    impd->FirstThunk = 0;

    impd = (IMP_DESC *) impbuf;
    for (i=0; i<ndll; i++) {
        IMPORT_LIST *imp_list;
        DWORD *thunks;
        DWORD *thunks2;

        imp_list = dll_list[i].thunknames;
        thunk_off = impd->FirstThunk - vaddr;
        thunks = (DWORD *) (impbuf + thunk_off);
        thunks2 = (DWORD *) (impbuf + thunk_off + (dll_list[i].entries+1) * 4);

        for (j=0; j<dll_list[i].entries; j++) {
            *(WORD *) (impbuf + str_off) = imp_list[j].hint;
            l = strlen(imp_list[j].name) + 1;
            memcpy(impbuf + str_off + 2, imp_list[j].name, l);
            imp_list[j].impsec_off = thunk_off + j * sizeof(DWORD);
            thunks[j] = vaddr + str_off;
            thunks2[j] = vaddr + str_off;
            str_off += l + sizeof(WORD);
        }
        thunks[j] = 0;
        impd++;
    }

    lseek(pehandle, scnd_hdr[IDATA].s_scnptr, SEEK_SET);
    write(pehandle, impbuf, impbuf_size);

    free(impbuf);
}

static void write_import_fixup(int pehandle)
{
    int i, j;
    IMPORT_LIST *imp_list;
    DWORD imp_vaddr = scnd_hdr[IDATA].s_vaddr;
    DWORD sec_ptr, sec_vaddr;
    DWORD t1, t2, d1, d2;
    int which;

    t1 = scnd_hdr[TEXT].s_vaddr;
    t2 = t1 + scnd_hdr[TEXT].s_size;
    d1 = scnd_hdr[DATA].s_vaddr;
    d2 = d1 + scnd_hdr[DATA].s_size;

    for (i = 0; i < ndll; i++) {
        imp_list = dll_list[i].thunknames;

        for (j = 0; j < dll_list[i].entries; j++) {
            DWORD off, value;
            DWORD addr = imp_list[j].addr - BaseOfImage;

            if (addr >= t1 && addr <= t2)
                which = TEXT;
            else if (addr >= d1 && addr <= d2)
                which = DATA;
            else {
                printf("%lX, bad import fixup!\n", addr);
                continue;
            }

            sec_ptr = scnd_hdr[which].s_scnptr;
            sec_vaddr = scnd_hdr[which].s_vaddr;

            off = imp_list[j].addr - BaseOfImage - sec_vaddr;
            value = imp_list[j].impsec_off + imp_vaddr + BaseOfImage;

            lseek(pehandle, sec_ptr + off, SEEK_SET);
            write(pehandle, &value, sizeof(DWORD));

            if (default_param.opt_printall)
                printf("import call at text offset %lX to %lX\n", off, value);

            /* shorten referenz : code -> data -> import */
            /*                      code -> import */
            if (which == DATA) {
                DWORD v;
                int i;
                WORD *wd = (WORD *) reloc_list.mem;

                for (;;) {
                    DWORD *dw = (DWORD *)(wd);
                    DWORD dat;

                    if ((int)dw >= (int)(reloc_list.mem + reloc_list.size))
                        break;
                    v = dw[0];

                    wd += 4;
                    for (i = 0; i < (dw[1]-8)/2; i++, wd++) {
                        DWORD test_off = v + (*wd & 0xfff);

                        if (test_off >= t1 && test_off <= t2)
                            which = TEXT;
                        else if (test_off >= d1 && test_off <= d2)
                            which = DATA;
                        else
                            printf("importing data: ?? %lX", test_off);

                        sec_ptr = scnd_hdr[which].s_scnptr
                                  + test_off
                                  - scnd_hdr[which].s_vaddr;

                        lseek(pehandle, sec_ptr, SEEK_SET);
                        read(pehandle, &dat, sizeof(DWORD));

                        if (dat == addr + BaseOfImage) {
                            if (default_param.opt_printall)
                                printf("data import: fix %lX to %lX\n", dat, value);
                            lseek(pehandle, sec_ptr, SEEK_SET);
                            write(pehandle, &value, sizeof(DWORD));
                        }
                    }
                }
            }
        }
    }
}

struct layout
{
    DWORD text_base;              /* Base address of the text segment */
    DWORD text_end;               /* End address of the text segment */
    DWORD data_base;              /* Base address of initialized data */
    DWORD data_end;               /* End address of initialized data */
    DWORD bss_base;               /* Base address of uninitialized data */
    DWORD bss_end;                /* End address of uninitialized data */
};

static void make_pe(int fhandle)
{
    DWORD pe_magic = 0x4550;
    int pehandle, doshandle;
    long len, size, headsize;
    char *buf;
    struct layout data_layout;

    fill_ntheader();

    headsize = (pe_off.image > 4096) ? pe_off.image : 4096;
    buf = xmalloc(headsize);

    remove(default_param.pename);
    if ((pehandle = open(default_param.pename,
                        O_RDWR | O_CREAT | O_TRUNC | O_BINARY,
                        S_IREAD | S_IWRITE)) == -1) {
        perror("creat output ntfile");
        exit(1);
    }

    if ((doshandle = open(default_param.stubname,
                        O_RDONLY | O_BINARY)) == -1) {
        perror("open dos stubname");
        exit(1);
    }

    memset(buf, 0, headsize);
    write(pehandle, buf, 0x800);  /* zero PE header */

    size = filelength(doshandle);

    lseek(pehandle, 0, SEEK_SET);
    while (size > 0) {
        len = size;
        if (len > 4096)
            len = 4096;
        read(doshandle, buf, len);
        write(pehandle, buf, len);
        size -= len;
    }
    close(doshandle);

    /* write new offset entry */
    lseek(pehandle, 0x3C, SEEK_SET);
    write(pehandle, &pe_off.pe, 4);

    if (pe_off.emx) {
        /* write emx bind header */
        lseek(pehandle, pe_off.emx + 18, SEEK_SET);
        write(pehandle, &pe_off.aout, 4);

        /* write aout header */
        gnu_hdr.a_trsize = 0;
        gnu_hdr.a_drsize = 0;
        if (default_param.opt_strip)
            gnu_hdr.a_syms = 0;
        lseek(pehandle, pe_off.aout, SEEK_SET);
        write(pehandle, &gnu_hdr, sizeof(struct exec));
    }

    /* write nt header */
    lseek(pehandle, pe_off.pe, SEEK_SET);
    write(pehandle, &pe_magic, 4);
    write(pehandle, &file_hdr, sizeof(FILEHDR));
    write(pehandle, &ntopt_hdr, sizeof(NTOPTHDR));
    write(pehandle, &scnd_hdr[TEXT], sizeof(SCNHDR));
    write(pehandle, &scnd_hdr[DATA], sizeof(SCNHDR));
    if (default_param.opt_emxstyle && !default_param.opt_strip)
        write(pehandle, &scnd_hdr[DEBUG], sizeof(SCNHDR));
    if (ndll)
        write(pehandle, &scnd_hdr[IDATA], sizeof(SCNHDR));
    if (nexp)
        write(pehandle, &scnd_hdr[EDATA], sizeof(SCNHDR));
    if (reloc_list.size)
        write(pehandle, &scnd_hdr[RELOCSEC], sizeof(SCNHDR));

    /* write code */
    size = scnd_hdr[TEXT].s_size;
    lseek(fhandle, ihdr.text_fileptr, 0);
    lseek(pehandle, scnd_hdr[TEXT].s_scnptr, SEEK_SET);
    while (size > 0) {
        len = size;
        if (len > 4096)
            len = 4096;
        read(fhandle, buf, len);
        write(pehandle, buf, len);
        size -= len;
    }

    if (emx_source) {
        /* read emx os2 bind header (bad address) */
        lseek(fhandle, ihdr.data_fileptr, 0);
        read(fhandle, &data_layout, sizeof(struct layout));
    }

    /* write data */
    size = scnd_hdr[DATA].s_size;
    lseek(fhandle, ihdr.data_fileptr, 0);
    lseek(pehandle, scnd_hdr[DATA].s_scnptr, SEEK_SET);
    while (size > 0) {
        len = size;
        if (len > 4096)
            len = 4096;
        read(fhandle, buf, len);
        write(pehandle, buf, len);
        size -= len;
    }

    if (emx_source) {
        /* modify emx os2 bind header (bad bss address) */
        data_layout.data_end = ihdr.bss_vaddr + BaseOfImage;
        data_layout.bss_base = data_layout.data_end;
        data_layout.bss_end = data_layout.bss_base + ihdr.bss_size;
        if (ihdr.bss_size)
            data_layout.bss_end -= 4;

        lseek(pehandle, scnd_hdr[DATA].s_scnptr, SEEK_SET);
        write(pehandle, &data_layout, sizeof(struct layout));
    }

    /* write debug */
    if (default_param.opt_emxstyle && !default_param.opt_strip) {
        lseek(pehandle, scnd_hdr[DEBUG].s_scnptr, SEEK_SET);
        /* symbols */
        write(pehandle, nl, ihdr.sym_size);
        /* strings */
        write(pehandle, &ihdr.str_size, sizeof(DWORD));
        write(pehandle, symstr+4, ihdr.str_size-4);
        size = scnd_hdr[DEBUG].s_size - scnd_hdr[DEBUG].s_paddr;
        memset(buf, 0, size);
        write(pehandle, buf, size);
    }

    if (nexp) {
        lseek(pehandle, scnd_hdr[EDATA].s_scnptr, SEEK_SET);
        write_export(pehandle);
    }

    /* write relocs */
    if (reloc_list.size) {
        lseek(pehandle, scnd_hdr[RELOCSEC].s_scnptr, SEEK_SET);
        write(pehandle, reloc_list.mem, reloc_list.size);
        /* align section in file */
        size = scnd_hdr[RELOCSEC].s_size - reloc_list.size;
        memset(buf, 0, size);
        write(pehandle, buf, size);
    }

    /* write imports */
    if (ndll) {
        write_import(pehandle);
        write_import_fixup(pehandle);
    }

    free(buf);
    close(pehandle);
}


/****************************************************************************\
 *                                                                          *
 *   AOUT functions                                                         *
 *                                                                          *
\****************************************************************************/

static void save_reloc(RELOC *r, unsigned long base, int fhandle)
{
    if (r->r_extern) {              /* all extern symbols, but not emx imports */
        struct nlist *nlp = nl + r->r_symbolnum;
        if (nlp->n_type != 105)
            add_reloc_entry(base + r->r_address);
    }
    else if (r->r_pcrel == 0) {     /* pcrel=1 are only relative offsets */
        long value = 1;

        if (fhandle >= 0)
        {
            lseek(fhandle, ihdr.text_fileptr + r->r_address, 0);
            read(fhandle, &value, sizeof(long));
        }
        if (value != 0) /* fix bug for weak symbol */
            add_reloc_entry(base + r->r_address);
    }
}

static int reloc_compare(const void *x1, const void *x2)
{
    unsigned long r1, r2;

    r1 = ((RELOC *)x1)->r_address;
    r2 = ((RELOC *)x2)->r_address;

    return (r1 == r2) ? 0 : (r1 < r2) ? -1 : +1;
}

#ifdef FIX_EMX09B_LD_BUG
/* look for fix in emx09c */
static int test_duplicate_reloc(DWORD test)
{
    DWORD v;
    int i;
    WORD *wd = (WORD *) reloc_list.mem;

    for (;;) {
        DWORD *dw = (DWORD *)(wd);

        if ((int)dw >= (int)(reloc_list.mem + reloc_list.size))
            break;
        v = dw[0];

        wd += 4;
        for (i = 0; i < (dw[1]-8)/2; i++, wd++) {
            if (v + (*wd & 0xfff) == test)
                return 1;
        }
    }
    return 0;
}
#endif

static void handle_aout_relocs(int fhandle)
{
    int i, max;
    unsigned long base;

    /* get text relocations */
    if (ihdr.tr_size) {
        RELOC *text_reloc = (RELOC *) xmalloc(ihdr.tr_size);
        lseek(fhandle, ihdr.tr_fileptr, 0);
        read(fhandle, text_reloc, ihdr.tr_size);
        base = ihdr.text_vaddr;
        max = ihdr.tr_size / sizeof(RELOC);
        qsort (text_reloc, max, sizeof(RELOC), reloc_compare);
        for (i=0; i<max; i++)
            save_reloc(text_reloc + i , base, fhandle);
        free(text_reloc);
    }

    /* get data relocations */
    if (ihdr.tr_size) {
        RELOC *data_reloc = (RELOC *) xmalloc(ihdr.dr_size);
        lseek(fhandle, ihdr.dr_fileptr, 0);
        read(fhandle, data_reloc, ihdr.dr_size);
        base = ihdr.data_vaddr;
        max = ihdr.dr_size / sizeof(RELOC);
        qsort (data_reloc, max, sizeof(RELOC), reloc_compare);
        for (i=0; i<max; i++)
            save_reloc(data_reloc + i , base, -1);
        free(data_reloc);
    }

#ifdef FIX_EMX09B_LD_BUG
    /* search nlist Entries for additional relocations
    ** bug in linker: no data relocations for ctor/dtor lists
    **   compare with /emx/src/lib/startup/ctor1.c __ctordtorInit1()
    */
    for (i = 0; i < ihdr.sym_entries; i++) {
        if (nl[i].n_type > 10)
            printf("%d %X hex\n",nl[i].n_type,nl[i].n_type);

        if (nl[i].n_type == 29) {       /* ctorlist */
            DWORD offset = nl[i].n_value - BaseOfImage - ihdr.data_vaddr;
            long size, value;

            back:
            lseek(fhandle, ihdr.data_fileptr + offset, 0);
            read(fhandle, &size, sizeof(long));
            if (size == -1) {           /* other ld bug */
                offset -= 4;
                goto back;
            }
            offset+=4;
            lseek(fhandle,  ihdr.data_fileptr + offset, 0);
            read(fhandle, &value, sizeof(long));
            if (value != -1)
                continue;

            for (offset+=4 ; size>1; --size, offset+=4) {
                RELOC r;

                /* handle fixed ld version */
                if (test_duplicate_reloc(ihdr.data_vaddr + offset)) {
                    i = ihdr.sym_entries;       /* break all; */
                    break;
                }
                r.r_address = offset;
                r.r_symbolnum = 5;
                r.r_pcrel = r.r_extern = r.r_pad = 0;
                r.r_length = 2;
                save_reloc(&r, ihdr.data_vaddr);
            }
        }
    }
#endif
}

static unsigned long aout_find_export_symbol (char *exportname)
{
    unsigned long addr = 0;
    int i;

    for (i = 0; i < ihdr.sym_entries; i++) {

        if (memcmp(exportname, symstr + nl[i].n_un.n_strx + 1,
                strlen(exportname) + 1) == 0) {

            if (addr) {
                printf("more than one entry for export function: %s\n",
                          exportname);
            } else
                addr = nl[i].n_value - BaseOfImage;
        }
    }
    return addr;
}

static void convert_aouthdr(int fhandle)
{
    struct layout emxlayout;

    lseek(fhandle, 0, 0);
    read(fhandle, &gnu_hdr, sizeof(gnu_hdr));

    if (gnu_hdr.a_entry == 0x10000) {           /* origin emx start address */
        BaseOfImage = 0;
        segment_size = 0x10000;
    }
    else if (gnu_hdr.a_entry == 0x401000) {     /* rsxnt 1.2 start address */
        BaseOfImage = 0x400000L;
        segment_size = 0x1000;
    }
    else if (gnu_hdr.a_entry == 0x410000) {     /* rsxnt 1.25 start address */
        BaseOfImage = 0x400000L;
        segment_size = 0x10000;
    }
    else if (gnu_hdr.a_entry >= 0x800000) {     /* DLL start address */
        segment_size = gnu_hdr.a_entry & 0xFFFFF;
        BaseOfImage = gnu_hdr.a_entry - segment_size;
    }
    else {
        printf("This entry is not supported: addr=0x%lX\n", gnu_hdr.a_entry);
        if (gnu_hdr.a_entry == 0x410000)
            printf("You are using the old ld from rsxnt 1.1");
        error("Can't handle this type of a.out file");
    }

    ihdr.text_vaddr = N_TXTADDR(gnu_hdr);
    ihdr.text_size = gnu_hdr.a_text;
    ihdr.text_fileptr = N_TXTOFF(gnu_hdr);
    ihdr.data_vaddr = N_DATADDR(gnu_hdr);
    ihdr.data_size = gnu_hdr.a_data;
    ihdr.data_fileptr = N_DATOFF(gnu_hdr);
    ihdr.bss_vaddr = N_BSSADDR(gnu_hdr);
    ihdr.bss_size = gnu_hdr.a_bss;
    ihdr.tr_size = gnu_hdr.a_trsize;
    ihdr.tr_fileptr = N_TRELOFF(gnu_hdr);
    ihdr.dr_size = gnu_hdr.a_drsize;
    ihdr.dr_fileptr = N_DRELOFF(gnu_hdr);
    ihdr.entry = gnu_hdr.a_entry;

    if (!ihdr.tr_size || !ihdr.dr_size)
        error("No relocations in file: \n"
            "\tyou have not linked the NT library");
/*
    && (ihdr.text_vaddr < 0x400000L)
        puts("Warning: No relocations in file: can't run under Win32s");
*/

    /* test correct base address */
    lseek(fhandle, ihdr.data_fileptr, SEEK_SET);
    read(fhandle, &emxlayout, sizeof(struct layout));

    if (ihdr.data_vaddr != (emxlayout.data_base - BaseOfImage)
          || ihdr.text_vaddr != (emxlayout.text_base - BaseOfImage))
    {
        printf("data %lX emx %lX\n", ihdr.data_vaddr,
                             emxlayout.data_base - BaseOfImage);
        printf("text %lX emx %lX\n", ihdr.text_vaddr ,
                             emxlayout.text_base - BaseOfImage);
        error("You have not used the new LD linker");
    }

    ihdr.sym_size = gnu_hdr.a_syms;
    ihdr.sym_entries = gnu_hdr.a_syms / sizeof(struct nlist);
    ihdr.sym_fileptr = N_SYMOFF(gnu_hdr);
    ihdr.str_fileptr = N_STROFF(gnu_hdr);

    /* get symbols */
    lseek(fhandle, ihdr.sym_fileptr, 0);
    nl = (struct nlist *) xmalloc(ihdr.sym_size);
    read(fhandle, nl, ihdr.sym_size);

    read(fhandle, &ihdr.str_size, sizeof(DWORD));
    symstr = (char *) xmalloc(ihdr.str_size);
    read(fhandle, symstr+4, ihdr.str_size-4);

    handle_aout_relocs(fhandle);
}

/****************************************************************************\
 *                                                                          *
 *   COFF functions                                                         *
 *                                                                          *
\****************************************************************************/

struct coff_relocs {
  unsigned long r_vaddr __attribute__((packed));
  unsigned long r_symndx __attribute__((packed));
  unsigned short r_type;
};

static int coff_reloc_compare(const void *x1, const void *x2)
{
    unsigned long r1, r2;

    r1 = ((struct coff_relocs *)x1)->r_vaddr;
    r2 = ((struct coff_relocs *)x2)->r_vaddr;

    return (r1 == r2) ? 0 : (r1 < r2) ? -1 : +1;
}

static void handle_coff_relocs(int coffhandle)
{
    struct coff_relocs *reloc;
    int             i, n;

    /* relocs in text section */
    if (ihdr.tr_size) {
        /* readin section */
        reloc = (struct coff_relocs *) malloc (ihdr.tr_size);
        lseek (coffhandle, ihdr.tr_fileptr, 0);
        read (coffhandle, reloc, ihdr.tr_size);

        n = ihdr.tr_size / sizeof (struct coff_relocs);
        qsort (reloc, n, sizeof(struct coff_relocs), coff_reloc_compare);

        for (i = 0; i < n; i++)
            if (reloc[i].r_type == 6)
                add_reloc_entry(reloc[i].r_vaddr - BaseOfImage);

        free (reloc);
    }

    /* relocs in data section */
    if (ihdr.dr_size) {
        reloc = (struct coff_relocs *) malloc (ihdr.dr_size);
        lseek (coffhandle, ihdr.dr_fileptr, 0);
        read (coffhandle, reloc, ihdr.dr_size);

        n = ihdr.dr_size / sizeof (struct coff_relocs);
        qsort (reloc, n, sizeof(struct coff_relocs), coff_reloc_compare);

        for (i = 0; i < n; i++)
            if (reloc->r_type == 6)
                add_reloc_entry(reloc[i].r_vaddr - BaseOfImage);

        free (reloc);
    }
}

static int      f_nsyms;
static SYMENT * f_symtab;
static char *   f_string_table;

static void coff_read_symbols (int fhandle, int nsyms, unsigned long symptr)
{
    long strsize;

    f_nsyms = nsyms;

    f_symtab = (SYMENT *) xmalloc(f_nsyms * SYMESZ);
    lseek(fhandle, symptr, 0);
    read(fhandle, f_symtab, f_nsyms * SYMESZ);

    read(fhandle, &strsize, sizeof(strsize));
    f_string_table = (char *) xmalloc(strsize);
    read(fhandle, f_string_table+4, strsize-4);
    f_string_table[0] = 0;
}

static unsigned long coff_find_export_symbol (char *exportname)
{
    int i;
    unsigned long addr = 0;
    char *name;
    char copy[9];

    for (i = 0; i < f_nsyms; i++) {
        SYMENT *p = f_symtab + i;

        if (p->e_sclass != 2)   /* find external */
            continue;

        if (p->e.e.e_zeroes) {
            memcpy (copy, p->e.e_name, 8);
            copy [8] = 0;
            name = copy;
        }
        else
            name = f_string_table + p->e.e.e_offset;

        if (memcmp(exportname, name + 1, strlen(exportname) + 1) == 0) {
            if (addr)
                printf("more than one entry for export function: %s\n",
                          exportname);
            else
                addr = p->e_value - BaseOfImage;
        }
    }
    return addr;
}

static void convert_coffhdr(int fhandle)
{
    FILEHDR     fhdr;
    AOUTHDR     aout_hdr;
    SCNHDR      scnd_hdr[5];
    long        correct;

    lseek(fhandle, 0, 0);
    read(fhandle, &fhdr, sizeof(fhdr));
    read(fhandle, &aout_hdr, sizeof(aout_hdr));
    read(fhandle, &scnd_hdr, sizeof(scnd_hdr));

    if (aout_hdr.entry < 0x400000) {
        printf("This entry is not supported: addr=0x%lX\n", aout_hdr.entry);
        error("Set the LIBRARY_PATH to rsxnt lib");
    }
    if ((aout_hdr.entry & 0xFFFFF) != 0x1000) {
        printf("This entry is not supported: addr=0x%lX\n", aout_hdr.entry);
        error("use -Ttext=0xXXX01000");
    }

    emx_source = 0;
    segment_size = 0x1000;
    BaseOfImage = aout_hdr.entry - segment_size;
    
    correct = scnd_hdr[0].s_vaddr & 0xfff;

    ihdr.text_vaddr = scnd_hdr[0].s_vaddr - correct - BaseOfImage;
    ihdr.text_size = scnd_hdr[0].s_size + correct;
    ihdr.text_fileptr = scnd_hdr[0].s_scnptr - correct;
    ihdr.data_vaddr = scnd_hdr[1].s_vaddr - BaseOfImage;
    ihdr.data_size = scnd_hdr[1].s_size;
    ihdr.data_fileptr = scnd_hdr[1].s_scnptr;
    ihdr.bss_vaddr = scnd_hdr[2].s_vaddr - BaseOfImage;
    ihdr.bss_size = scnd_hdr[2].s_size;
    ihdr.tr_size = scnd_hdr[0].s_nreloc * sizeof (struct coff_relocs);
    ihdr.tr_fileptr = scnd_hdr[0].s_relptr;
    ihdr.dr_size = scnd_hdr[1].s_nreloc * sizeof (struct coff_relocs);
    ihdr.dr_fileptr = scnd_hdr[1].s_relptr;
    ihdr.entry = aout_hdr.entry;

    ihdr.sym_size = scnd_hdr[3].s_size;
    ihdr.sym_entries = scnd_hdr[3].s_size / sizeof(struct nlist);
    ihdr.sym_fileptr = scnd_hdr[3].s_scnptr;

    ihdr.str_fileptr = scnd_hdr[4].s_scnptr;
    ihdr.str_size = scnd_hdr[4].s_size;

    nl = (struct nlist *) xmalloc(ihdr.sym_size);
    lseek(fhandle, ihdr.sym_fileptr, 0);
    read(fhandle, nl, ihdr.sym_size);

    symstr = (char *) malloc(ihdr.str_size);
    lseek(fhandle, ihdr.str_fileptr, 0);
    read(fhandle, symstr, ihdr.str_size);

    default_param.opt_strip = 1;

    gnu_hdr.a_info = 0x10B;
    gnu_hdr.a_text = ihdr.text_size;
    gnu_hdr.a_data = ihdr.data_size;
    gnu_hdr.a_bss = ihdr.bss_size;
    gnu_hdr.a_syms = 0;
    gnu_hdr.a_entry = ihdr.entry;
    gnu_hdr.a_trsize = 0;
    gnu_hdr.a_drsize = 0;

    handle_coff_relocs(fhandle);
    coff_read_symbols (fhandle, fhdr.f_nsyms, fhdr.f_symptr);
}


/****************************************************************************\
 *                                                                          *
 *   Module Definition file                                                 *
 *                                                                          *
\****************************************************************************/

static void def_warning(char *s)
{
    printf("Warning statement not supported :");
    puts(s);
    fflush(stdout);
}

/* callback for emx library function (parse DEF files) */

static int md_callback (struct _md *md, const _md_stmt *stmt, _md_token token,
                        void *arg)
{
    switch (token) {
        case _MD_BASE:
            def_warning ("BASE");
            break;

       case _MD_CODE:
           def_warning ("CODE");
           break;

       case _MD_DATA:
           def_warning ("DATA");
           break;

        case _MD_DESCRIPTION:
            default_param.description = xstrdup (stmt->descr.string);
            break;

        case _MD_EXETYPE:
            def_warning ("EXETYPE");
            break;

        case _MD_EXPORTS:
            add_export_entry(
                    xstrdup (stmt->export.entryname),
                    stmt->export.ordinal);
            break;

        case _MD_HEAPSIZE:
            if (stmt->heapsize.size < 20*1024
                || stmt->heapsize.size > 512*1024*1024)
                puts ("Invalid heap size in module definition file");
            else
                default_param.heapsize = (stmt->heapsize.size + 0xfff) & ~0xfff;
            break;
            break;

        case _MD_IMPORTS:
            def_warning ("IMPORTS");
            break;

        case _MD_LIBRARY:
        /* dll_flag = TRUE; */
            switch (stmt->library.init) {
            case _MDIT_GLOBAL:
                puts("MODDEF: init_global = TRUE");
                break;
            case _MDIT_INSTANCE:
                puts("MODDEF: init_global = FLASE");
                break;
            default:
                break;
            }
            switch (stmt->library.term) {
                case _MDIT_GLOBAL:
                    puts("MODDEF: term_global = TRUE");
                    break;
                case _MDIT_INSTANCE:
                    puts("MODDEF: term_global = TRUE");
                    break;
                default:
                    break;
            }
            break;

        case _MD_NAME:
            if (stmt->name.name[0] != 0)
                default_param.pename = xstrdup(stmt->name.name);

            printf("app_type %d\n",  stmt->name.pmtype);
            break;

        case _MD_OLD:
            def_warning ("OLD");
            break;

        case _MD_PROTMODE:
            def_warning ("PROTMODE");
            break;

        case _MD_REALMODE:
            def_warning ("REALMODE");
            break;

        case _MD_SEGMENTS:
            def_warning ("SEGMENTS");
            break;

        case _MD_STACKSIZE:
            if (stmt->stacksize.size < 20*1024
                || stmt->stacksize.size > 512*1024*1024)
                puts ("Invalid stack size in module definition file");
            else
                default_param.stacksize = (stmt->stacksize.size + 0xfff) & ~0xfff;
            break;

        case _MD_STUB:
            if (!stmt->stub.none)
                default_param.stubname = xstrdup (stmt->stub.name);
            break;

        case _MD_VIRTUAL:
        case _MD_PHYSICAL:
            puts("VIRTUAL DEVICE and PHYSICAL DEVICE statements not supported");

        case _MD_parseerror:
        default:
            printf ("%s (line %ld)", _md_errmsg (stmt->error.code),
                        _md_get_linenumber (md));
            exit (1);
    }
    return (0);
}

static void read_def_file (char *def_fname)
{
  struct _md *md;

  md = _md_open (def_fname);
  if (md == NULL) {
    printf ("cannot open `%s'", def_fname);
    exit(1);
  }
  _md_next_token (md);
  _md_parse (md, md_callback, NULL);
  _md_close (md);
}


/****************************************************************************\
 *                                                                          *
 *   PE main functions                                                      *
 *                                                                          *
\****************************************************************************/

static int build_pe_exe(void)
{
    int fhandle;
    char *app_type;

    if ((fhandle = open(default_param.aoutname, O_RDONLY | O_BINARY)) == -1) {
        perror("open input-file");
        return 1;
    }
    read(fhandle, &file_hdr, sizeof(file_hdr));

    if (file_hdr.f_magic == 0x10B)
        convert_aouthdr(fhandle);
    else if (file_hdr.f_magic == 0x14C) {
        convert_coffhdr(fhandle);
    }
    else {
        puts("not a valid aout/coff file");
        return 1;
    }

    init_nthdr(fhandle);
    handle_stabs(fhandle);
    if (default_param.opt_printall)
        print_entries(fhandle);
    make_pe(fhandle);

    if (default_param.output_mode == MODE_DLL)
        app_type = "DLL";
    else if (default_param.win32_subsystem == SUB_GUI)
        app_type = "GUI";
    else if (!default_param.opt_emxstyle || default_param.opt_nosyscall ||
                pe_off.aout <= 512)
        app_type = "Console";
    else if (BaseOfImage)
        app_type = "Dual(rsx)";
    else
        app_type = "Dual(emx)";

    printf("Ntbind: Building %s Application \"%s\"\n", app_type, default_param.pename);

    close(fhandle);
    return 0;
}

/****************************************************************************\
 *                                                                          *
 *   main / get options                                                     *
 *                                                                          *
\****************************************************************************/

static char usage_text[] =
    RSXNT_VERSION "\n"
    "usage : NTBind aout-file [options]\n"
    "options:\n"
    "-d name.def   : the definition file"
    "-o output.exe : executable name\n"
    "-o output.dll : create DLL\n"
    "-s stubname   : define dos stub file\n"
    "-strip        : strip all symbols\n"
    "-nosyscall    : don't use C runtime (rsxnt.dll)\n"
    "-version      : print ntbind version\n" ;

int main(int argc, char **argv)
{
    static char stub_path[260];
    char *pename = NULL;
    char *aoutname = NULL;
    char *stubname = NULL;
    char *defname = NULL;
    char use_tmp_name = 0;
    int i, r;

    for (i = 1; i < argc; i++) {
        if (argv[i][0] == '-') {
            if (argv[i][1] && !argv[i][2]) {

                /* one letter options + argument */
                if (argv[i+1] == NULL) {
                    printf("option need argument %s\n", argv[i]);
                    return 1;
                }
                if (argv[i][1] == 'o')
                    pename = argv[++i];
                else if (argv[i][1] == 'd')
                    defname = argv[++i];
                else if (argv[i][1] == 's')
                    stubname = argv[++i];
                else {
                    printf("Unknown option: %s\n", argv[i]);
                    error(usage_text);
                }
            }
            else {
                /* more letter options without args*/
                if (strcmp(argv[i], "-version") == 0)
                    puts(RSXNT_VERSION);
                else if (strcmp(argv[i], "-print") == 0)
                    default_param.opt_printall = 1;
                else if (strcmp(argv[i], "-nosyscall") == 0)
                    default_param.opt_nosyscall = 1;
                else if (strcmp(argv[i], "-strip") == 0)
                    default_param.opt_strip = 1;
                else if (strcmp(argv[i], "-remove") == 0)
                    use_tmp_name = 1;
                else {
                    printf("unknown option: %s\n", argv[i]);
                    error(usage_text);
                }
            }
        }
        else if (strstr(argv[i], ".def") != NULL) {
            if (!defname)
                defname = argv[i];
            else {
                puts("RSXNT : more than one def file");
                error(usage_text);
            }
        }
        else if (!aoutname)
            aoutname = argv[i];
        else {
            puts("RSXNT : more than one input file");
            error(usage_text);
        }
    }

    if (!aoutname)
        error(usage_text);
    else
        default_param.aoutname = aoutname;

    init_all_lists();

    /* read def file first, options can overrides this */
    if (defname) {
        if (access(defname,0)<0) {
            printf("cannot found definition file '%s'\n", defname);
            return 1;
        } else
            read_def_file(defname);
    }

    /* override */
    if (pename)
        default_param.pename = pename;
    if (stubname)
        default_param.stubname = stubname;

    /*
        DJGPP compiler perhaps run:
            ntbind name.exe
        should rename to:
            ntbind name.old -o name.exe
    */

    /* no output-name or no exe-extention */
    if (!default_param.pename) {
        char *point;
        int len;

        len = strlen (aoutname) + 1;
        default_param.pename = xmalloc(len + 4);
        memcpy (default_param.pename, aoutname, len);
        point = strrchr(default_param.pename, '.');

        if (point) {
            if (stricmp(point, ".DLL") == 0)
                default_param.output_mode = MODE_DLL;
            else
                default_param.output_mode = MODE_EXE;

            if (stricmp(point, ".DLL") && stricmp(point, ".EXE"))
                strcpy(point, ".exe");
            else {
                /* bad: input exe name == output exe name */
                default_param.aoutname = xstrdup(default_param.pename);
                point = strrchr(default_param.aoutname, '.');
                strcpy (point, ".SYM");
                if (access(default_param.aoutname, 0) == 0)
                    remove (default_param.aoutname);
                rename (default_param.pename, default_param.aoutname);

                if (!defname && default_param.output_mode == MODE_DLL) {
                    defname = xstrdup(default_param.pename);
                    point = strrchr(defname, '.');
                    strcpy (point, ".DEF");
                    read_def_file(defname);
                }
                /* use_tmp_name = 1; */
            }
        }
        else {
            strcat(default_param.pename, ".exe");
            default_param.output_mode = MODE_EXE;
        }
    } else {
        char *teststr;
        int l;

        teststr = xstrdup(default_param.pename);
        strupr(teststr);
        l = strlen(teststr) - 4;
        if (l > 0 && stricmp(teststr + l, ".exe") == 0)
            default_param.output_mode = MODE_EXE;
        else if (l > 0 && stricmp(teststr + l, ".DLL") == 0)
            default_param.output_mode = MODE_DLL;
        else {
            printf("DLL or EXE file : %s\n", default_param.pename);
            return 1;
        }
    }

    if (access(default_param.aoutname, 0) < 0) {
        printf("cannot found emx aout file '%s'\n", default_param.pename);
        return 1;
    }
    if (access(default_param.stubname, 0) < 0) {
        if (_path(stub_path, default_param.stubname) < 0) {
            printf("cannot found dos stub file '%s'\n", default_param.stubname);
            return 1;
        } else
            default_param.stubname = stub_path;
    }

    r = build_pe_exe();
    if (use_tmp_name)
        remove (default_param.aoutname);
    return r;
}
