/*----------------------------------------------------------
   Fetches mail from a POP3 server and stores it locally.
   This program must be run from the "mail root" directory.

           Author:       Peter Moylan (peter@ee.newcastle.edu.au)
           Started:      23 August 1999
           Last revised: 5 November 1999

   Usage:
           popget hostname user password

           where hostname is the node name of the remote POP3 server;
           user is the username and password is the POP3 password.

------------------------------------------------------------*/

/****************************************************************/
/*                       MAIN PROGRAM                           */
/****************************************************************/

parse arg server user password

CALL RxFuncAdd SysLoadFuncs, rexxutil, sysloadfuncs
CALL SysLoadFuncs
CALL RxFuncAdd "SockLoadFuncs","rxSock","SockLoadFuncs"
CALL SockLoadFuncs nul
RB = ""                                       /* Receive buffer */
Globals = "OurSocket RB"

IF ConnectToServer(server, user, password) THEN
    DO
        IF SendCmd("STAT") THEN
            PARSE VALUE ReceiveLine() WITH dummy N bytecount
        ELSE
            N = 0
        SAY N" Messages in mailbox"
        DO j = 1 TO N
            call HandleMessage(user, j)
        END
    END

IF OurSocket \= -1 THEN
    DO
        CALL ExecCmd "QUIT"
        CALL SockSoClose(OurSocket)
    END

EXIT

/****************************************************************/
/*                   HANDLING ONE MESSAGE                       */
/****************************************************************/

HandleMessage: PROCEDURE expose (Globals)

    /* Retrieves one mail item from the server and stores it    */
    /* locally, also deleting it from the server.               */

    parse arg user, j
    IF \ExecCmd("RETR "j) THEN RETURN
    tmpfilename = SysTempFileName("?????.msg")
    recipient = ""
    InHeader = 1

    DO FOREVER
        /* Each time around this loop we retrieve one line of   */
        /* the message.                                         */

        line = ReceiveLine()
        IF LEFT(line,1) = '.' THEN
            DO
                k = LENGTH(line)
                IF k = 1 THEN LEAVE
                line = RIGHT(line,k-1)
            END
        CALL LineOut tmpfilename, line
        IF InHeader THEN
            DO
                IF line = "" THEN InHeader = 0
                ELSE IF LEFT(line,3) = "To:" THEN
                    recipient = ExtractUser(RIGHT(line,LENGTH(line)-3))
            END
    END
    CALL stream tmpfilename, 'C', 'CLOSE'

    /* Delete the message from the server.  (Comment out this   */
    /* line if you want to leave messages on the server.)       */

    call SendCmd "DELE "j

    /* Store the message in a subdirectory whose name is        */
    /* derived from the "To:" line in the mail header.  If we	*/
    /* can't find any such name, default to the username that	*/
    /* we're using to log in to the POP server.  If all else	*/
    /* fails, leave the mail in the current directory.          */

    IF recipient = "" THEN recipient = user

    IF recipient = "" THEN filename = tmpfilename
    ELSE
        DO
            CALL SysMkDir recipient
            filename = SysTempFileName(recipient"\?????.msg")
            '@MOVE 'tmpfilename' 'filename' >nul'
        END
    SAY "Saved "filename
    RETURN

/****************************************************************/
/*           PARSING THE "To:" LINE IN THE HEADER               */
/****************************************************************/

ExtractUser: PROCEDURE

    /* The argument to this procedure is the "To:" line from    */
    /* the message header.  It has the form user@domain, plus   */
    /* possibly some other detail.  We return the 'user' part.  */

    PARSE ARG line
    line = STRIP(line)
    IF LEFT(line,1) = '"' THEN
        PARSE VAR line WITH '"' rubbish '"' line
    IF RIGHT(line,1) = ')' THEN
        PARSE VAR line WITH line '(' rubbish ')'
    line = STRIP(line)
    IF LEFT(line,1) = '<' THEN
        PARSE VAR line WITH '<' line '>' rubbish
    j = POS('@',line)
    IF j > 0 THEN line = LEFT(line,j-1)
    RETURN line

/****************************************************************/
/*                 CONNECTING TO THE SERVER                     */
/****************************************************************/

ConnectToServer: procedure expose (Globals)

    /* This procedure opens a connection to the remote POP      */
    /* server and logs in.                                      */

    parse arg server, user, password
    OurSocket = SockSocket("AF_INET", "SOCK_STREAM", "IPPROTO_TCP")
    IF OurSocket = -1 THEN
      DO
        say "Can't create socket"
        RETURN 0
      END
    IF DATATYPE(LEFT(server,1))='NUM' THEN host.addr = server
    ELSE
      DO
        IF \SockGetHostByName(server,'host.') THEN
          DO
            say "Unknown host"
            DROP host.
            RETURN 0
          END
      END

    target.!family = 'AF_INET'
    target.!port = 110
    target.!addr = host.addr
    DROP host.
    IF SockConnect(OurSocket,"target.!") = -1 THEN
      DO
        say "Failed to connect"
        RETURN 0
      END
    IF \OKReply() THEN
      DO
        say "Server rejected us"
        RETURN 0
      END
    IF \ExecCmd("USER "||user) THEN
      DO
        say "Username rejected"
        RETURN 0
      END
    IF \ExecCmd("PASS "||password) THEN
      DO
        say "Password rejected"
        RETURN 0
      END
    RETURN 1

/****************************************************************/

SendCmd: PROCEDURE expose (Globals)

    /* Sends a one-line command to the server */

    PARSE ARG str
    RETURN (SockSend(OurSocket, str||'0D0A'X) \= -1)

/****************************************************************/

ExecCmd: PROCEDURE expose (Globals)

    /* Like SendCmd, but also checks for a positive reply. */

    PARSE ARG str
    RETURN (SockSend(OurSocket, str||'0D0A'X) \= -1) & OKReply()

/****************************************************************/

OKReply: PROCEDURE expose (Globals)

    /* Checks that server returned an "OK" response */

    RETURN LEFT(ReceiveLine(),1) = '+'

/****************************************************************/
/*              RECEIVING ONE LINE FROM THE SERVER              */
/*                                                              */
/*  Note: the global variable RB is the receive buffer; it      */
/*  holds input that has not yet been processed.                */
/****************************************************************/

ReceiveLine: PROCEDURE expose (Globals)
    line = ''
    DO FOREVER
      len = LENGTH(RB)
      IF len = 0 THEN
        DO
          NullResponseCount = 0
          DO WHILE len = 0
            len = SockRecv(OurSocket, 'RB', 256)
            IF len = 0 THEN
              DO
                NullResponseCount = NullResponseCount+1
                IF NullResponseCount > 20 THEN len = -1
              END
          END /*do-while*/
          IF len = -1 THEN RETURN ''
        END /* if len=0 */
      j0 = POS('0A'X, RB)
      IF j0 = 0 THEN
        DO
          line = STRIP(line||RB, 'T', '0D'X)
          RB = ''
        END
      ELSE
        DO
          line = STRIP(line||LEFT(RB,j0-1), 'T', '0D'X)
          RB = RIGHT(RB,len-j0)
          RETURN line
        END
    END /* do forever */

/****************************************************************/
