MODULE DumpINI;

        (********************************************************)
        (*                                                      *)
        (*          Converts Weasel.INI to a text file          *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            16 July 1998                    *)
        (*  Last edited:        15 October 1998                 *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*       This version is not dumping the host lists     *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2, IOChan, ChanConsts, FileSys, SeqFile, Strings, STextIO;

FROM SYSTEM IMPORT LOC, ADR;

FROM Storage IMPORT ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST Nul = CHR(0);

TYPE
    CardArray = ARRAY [0..1] OF CARDINAL;
    NameString = ARRAY [0..31] OF CHAR;
    EmailAddress = ARRAY [0..511] OF CHAR;
    FilenameString = ARRAY [0..511] OF CHAR;
    BufferIndex = [0..MAX(CARDINAL) DIV 4];

VAR
    (* Anchor block handle for this application.  *)

    hab: OS2.HAB;

    (* INI file handle. *)

    hini: OS2.HINI;

    (* Output file handle. *)

    cid: IOChan.ChanId;

(********************************************************************************)
(*                          OUTPUT TO TNI FILE                                  *)
(********************************************************************************)

PROCEDURE WriteRaw (cid: IOChan.ChanId;  data: ARRAY OF LOC;  amount: CARDINAL);

    (* Writes a string to a file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(data), amount);
    END WriteRaw;

(************************************************************************)

PROCEDURE FWriteChar (cid: IOChan.ChanId;  character: CHAR);

    (* Writes a single character to a file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(character), 1);
    END FWriteChar;

(************************************************************************)

PROCEDURE FWriteCard (cid: IOChan.ChanId;  number: CARDINAL);

    (* Writes a number to a file. *)

    BEGIN
        IF number > 9 THEN
            FWriteCard (cid, number DIV 10);
            number := number MOD 10;
        END (*IF*);
        FWriteChar (cid, CHR(ORD('0') + number));
    END FWriteCard;

(************************************************************************)

PROCEDURE FWriteString (cid: IOChan.ChanId;  string: ARRAY OF CHAR);

    (* Writes a string to a file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(string), LENGTH(string));
    END FWriteString;

(************************************************************************)

PROCEDURE FWriteLn (cid: IOChan.ChanId);

    (* Writes end-of-line to the file. *)

    TYPE TwoChar = ARRAY [0..1] OF CHAR;
    CONST CRLF = TwoChar {CHR(13), CHR(10)};

    BEGIN
        WriteRaw (cid, CRLF, 2);
    END FWriteLn;

(************************************************************************)

PROCEDURE OpenOutputFile (VAR (*OUT*) cid: IOChan.ChanId;
                       outname: ARRAY OF CHAR): BOOLEAN;

    (* Opens the file outname.TNI for output.  If a file of this name   *)
    (* already exists, we create a backup file.                         *)

    VAR filename, BAKname: FilenameString;
        result: ChanConsts.OpenResults;  dummy: BOOLEAN;

    BEGIN
        Strings.Assign (outname, filename);
        Strings.Append (".TNI", filename);
        SeqFile.OpenWrite (cid, filename, ChanConsts.write+ChanConsts.raw, result);
        IF result = ChanConsts.fileExists THEN
            Strings.Assign (outname, BAKname);
            Strings.Append (".BAK", BAKname);
            FileSys.Remove (BAKname, dummy);
            FileSys.Rename (filename, BAKname, dummy);
            SeqFile.OpenWrite (cid, filename, ChanConsts.write+ChanConsts.raw, result);
        END (*IF*);
        RETURN result = ChanConsts.opened;
    END OpenOutputFile;

(************************************************************************)

PROCEDURE INIGet (hini: OS2.HINI;  name1, name2: ARRAY OF CHAR;
                                     VAR (*OUT*) variable: ARRAY OF LOC): BOOLEAN;

    (* Retrieves the value of a variable from the INI file.  Returns FALSE if   *)
    (* the variable was not found.                                              *)

    VAR size: CARDINAL;

    BEGIN
        OS2.PrfQueryProfileSize (hini, name1, name2, size);
        IF size = HIGH(variable)+1 THEN
            OS2.PrfQueryProfileData (hini, name1, name2, ADR(variable), size);
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END (*IF*);
    END INIGet;

(********************************************************************************)

PROCEDURE INIGetString (hini: OS2.HINI;  name1, name2: ARRAY OF CHAR;
                                    VAR (*OUT*) variable: ARRAY OF CHAR): BOOLEAN;

    (* Like INIGet, but we accept any size data that will fit in the variable,  *)
    (* and we add a Nul terminator in the case of a size mismatch.              *)

    VAR size: CARDINAL;

    BEGIN
        OS2.PrfQueryProfileSize (hini, name1, name2, size);
        IF size <= HIGH(variable)+1 THEN
            OS2.PrfQueryProfileData (hini, name1, name2, ADR(variable), size);
            IF size <= HIGH(variable) THEN
                variable[size] := CHR(0);
            END (*IF*);
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END (*IF*);
    END INIGetString;

(************************************************************************)

PROCEDURE TranslateSYSparameters;

    (* Writes the miscellaneous INI data to the output file. *)

    VAR port, MaxUsers, TimeoutLimit: CardArray;
        j: [0..1];
        TransactionLogLevel, ServerEnable: CARDINAL;
        MailRoot, ForwardRelayHost: ARRAY [0..511] OF CHAR;

    (********************************************************************)

    PROCEDURE GetItem (name: ARRAY OF CHAR;
                         VAR (*OUT*) variable: ARRAY OF LOC): BOOLEAN;

        BEGIN
            RETURN INIGet (hini, "$SYS", name, variable);
        END GetItem;

    (********************************************************************)

    PROCEDURE GetStringItem (name: ARRAY OF CHAR;
                         VAR (*OUT*) variable: ARRAY OF CHAR): BOOLEAN;

        BEGIN
            RETURN INIGetString (hini, "$SYS", name, variable);
        END GetStringItem;

    (****************************************************************)

    BEGIN
        TransactionLogLevel := 1;
        IF NOT GetItem ("Enable", ServerEnable) THEN
            ServerEnable := 3;
        END (*IF*);
        IF NOT GetItem ("ServerPort", port) THEN
            port := CardArray {25, 110};
        END (*IF*);
        IF NOT GetItem ("MaxUsers", MaxUsers) THEN
            MaxUsers := CardArray {10, 10};
        END (*IF*);
        IF GetItem ("TransLevel", TransactionLogLevel) THEN
            INC (TransactionLogLevel);
        END (*IF*);
        IF NOT GetItem ("TimeOut", TimeoutLimit) THEN
            TimeoutLimit := CardArray {900, 900};
        END (*IF*);
        IF NOT GetStringItem ("MailRoot", MailRoot) THEN
            MailRoot := "C:\MPTN\ETC\MAIL\";
        END (*IF*);
        IF NOT GetStringItem ("ForwardRelay", ForwardRelayHost) THEN
            ForwardRelayHost := "";
        END (*IF*);

        (* We have the data, now write it to the output file. *)

        FWriteString (cid, "[General]");  FWriteLn (cid);
        FOR j := 0 TO 1 DO
            FWriteCard (cid, port[j]);  FWriteChar (cid, ' ');
            FWriteCard (cid, MaxUsers[j]);  FWriteChar (cid, ' ');
            FWriteCard (cid, TimeoutLimit[j]);  FWriteChar (cid, ' ');
        END (*FOR*);
        FWriteCard (cid, TransactionLogLevel);  FWriteChar (cid, ' ');
        FWriteCard (cid, ServerEnable);  FWriteLn (cid);
        FWriteString (cid, MailRoot);  FWriteLn (cid);
        FWriteString (cid, ForwardRelayHost);  FWriteLn (cid);

    END TranslateSYSparameters;

(************************************************************************)

PROCEDURE TranslateUsers;

    (* Reads all user information from the INI file, writes it to       *)
    (* the TNI file.                                                    *)

    VAR BufferSize: CARDINAL;
        bufptr: POINTER TO ARRAY BufferIndex OF CHAR;
        index: BufferIndex;
        Name, password: ARRAY [0..31] OF CHAR;
        j: [0..31];

    BEGIN
        FWriteString (cid, "[Users]");  FWriteLn (cid);
        IF OS2.PrfQueryProfileSize (hini, NIL, NIL, BufferSize)
                              AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, NIL, NIL, bufptr, BufferSize);

            index := 0;
            LOOP
                IF index >= BufferSize THEN
                    EXIT (*LOOP*);
                END (*IF*);
                j := 0;
                LOOP
                    Name[j] := bufptr^[index];
                    INC(index);  INC(j);
                    IF (Name[j-1] = Nul) OR (index >= BufferSize) THEN
                        EXIT (*LOOP*);
                    END (*IF*);
                END (*LOOP*);
                IF Name[0] = Nul THEN
                    EXIT (*LOOP*);
                END (*IF*);
                IF Name[0] <> '$' THEN
                    FWriteString (cid, Name);
                    IF INIGet (hini, Name, "Password", password) THEN
                        FWriteChar (cid, ' ');
                        FWriteString (cid, password);
                    END (*IF*);
                    FWriteLn (cid);
                END (*IF*);
            END (*LOOP*);

            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);
    END TranslateUsers;

(************************************************************************)

PROCEDURE TranslateOneAlias (name: NameString);

    (* Reads an alias expansion, writes it to the TNI file. *)

    VAR BufferSize: CARDINAL;
        bufptr: POINTER TO ARRAY BufferIndex OF CHAR;
        index: BufferIndex;
        value: EmailAddress;
        j: CARDINAL;

    BEGIN
        IF OS2.PrfQueryProfileSize (hini, "$ALIAS", name, BufferSize)
                              AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, "$ALIAS", name, bufptr, BufferSize);

            IF bufptr^[0] = CHR(1) THEN FWriteChar (cid, '+')
            ELSE FWriteChar (cid, '-')
            END (*IF*);
            FWriteString (cid, name);  FWriteLn (cid);
            index := 1;
            LOOP
                IF index >= BufferSize THEN
                    EXIT (*LOOP*);
                END (*IF*);
                j := 0;
                LOOP
                    value[j] := bufptr^[index];
                    INC(index);  INC(j);
                    IF (value[j-1] = Nul) OR (index >= BufferSize) THEN
                        EXIT (*LOOP*);
                    END (*IF*);
                END (*LOOP*);
                IF value[0] = Nul THEN
                    EXIT (*LOOP*);
                END (*IF*);
                FWriteString (cid, "  ");
                FWriteString (cid, value);
                FWriteLn (cid);
            END (*LOOP*);

            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);
    END TranslateOneAlias;

(************************************************************************)

PROCEDURE TranslateAliases;

    (* Dumps all alias definitions to the TNI file. *)

    VAR BufferSize: CARDINAL;
        bufptr: POINTER TO ARRAY BufferIndex OF CHAR;
        index: BufferIndex;
        Name: NameString;
        j: CARDINAL;

    BEGIN
        FWriteString (cid, "[Aliases]");  FWriteLn (cid);
        IF OS2.PrfQueryProfileSize (hini, "$ALIAS", NIL, BufferSize)
                              AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, "$ALIAS", NIL, bufptr, BufferSize);

            index := 0;
            LOOP
                IF index >= BufferSize THEN
                    EXIT (*LOOP*);
                END (*IF*);
                j := 0;
                LOOP
                    Name[j] := bufptr^[index];
                    INC(index);  INC(j);
                    IF (Name[j-1] = Nul) OR (index >= BufferSize) THEN
                        EXIT (*LOOP*);
                    END (*IF*);
                END (*LOOP*);
                IF Name[0] = Nul THEN
                    EXIT (*LOOP*);
                END (*IF*);
                TranslateOneAlias (Name);
            END (*LOOP*);

            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);
    END TranslateAliases;

(************************************************************************)
(*                              MAIN PROGRAM                            *)
(************************************************************************)

PROCEDURE RunTheProgram;

    BEGIN
        IF OpenOutputFile(cid, "Weasel") THEN
            hini := OS2.PrfOpenProfile (hab, "weasel.ini");
            IF hini = OS2.NULLHANDLE THEN
                STextIO.WriteString ("Could not open WEASEL.INI");
                STextIO.WriteLn;
            ELSE
                TranslateSYSparameters;
                TranslateUsers;
                TranslateAliases;
                OS2.PrfCloseProfile (hini);
            END (*IF*);
            SeqFile.Close (cid);
        ELSE
            STextIO.WriteString ("Could not open output file");
            STextIO.WriteLn;
        END (*IF*);
    END RunTheProgram;

(************************************************************************)

BEGIN
    hab := OS2.WinInitialize (0);
    RunTheProgram;
END DumpINI.

