IMPLEMENTATION MODULE Hosts;

        (********************************************************)
        (*                                                      *)
        (*          Checks on host names and addresses          *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            9 May 1998                      *)
        (*  Last edited:        4 November 1999                 *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

IMPORT Strings, OS2;

FROM SYSTEM IMPORT
    (* type *)  CARD8, LOC,
    (* proc *)  CAST;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM Sockets IMPORT
    (* const*)  AF_INET,
    (* proc *)  sock_init, gethostid;

FROM Internet IMPORT
    (* proc *)  inet_addr;

FROM NetDB IMPORT
    (* type *)  HostEntPtr,
    (* proc *)  gethostname, gethostbyname, gethostbyaddr, tcp_h_errno;

FROM Names IMPORT
    (* type *)  HostNameIndex, HostName, HostCategory;

FROM InetUtilities IMPORT
    (* proc *)  OpenINIFile, IPToString, Swap4, NameIsNumeric;

FROM WildCard IMPORT
    (* proc *)  WildMatch;

FROM SplitScreen IMPORT
    (* proc *)  WriteString, WriteLn, NotDetached;

(************************************************************************)

TYPE
    LabelString = ARRAY [0..9] OF CHAR;
    Label = ARRAY HostCategory OF LabelString;

CONST
    Nul = CHR(0);
    INILabel = Label {"Local", "MayRelay", "RelayDest", "Banned"};

TYPE
    (* In host records the data can be numeric (in numaddr) or textual  *)
    (* (in textaddr), or both.  NOTE: if both are present, it doesn't   *)
    (* necessarily imply that the number and the name refer to the same *)
    (* machine.  We merge records to save space and improve search time.*)

    (* If numeric and range are both TRUE, this is a representation of  *)
    (* the range [numaddr..numaddr2] (inclusive).  In all other cases   *)
    (* numaddr2 is ignored.                                             *)

    HostRecordPtr = POINTER TO HostRecord;
    HostRecord = RECORD
                     next: HostRecordPtr;
                     numeric, range, text: BOOLEAN;
                     numaddr: CARDINAL;         (* network byte order *)
                     numaddr2: CARDINAL;        (* network byte order *)
                     textaddr: HostName;
                 END (*RECORD*);

    HostList = HostRecordPtr;

VAR
    MasterList: ARRAY HostCategory OF HostList;
    LocalListLock: Lock;

(************************************************************************)
(*                           SCREEN OUTPUT                              *)
(************************************************************************)

PROCEDURE WriteIPAddr (addr: CARDINAL);

    (* Writes an IP address to the screen. *)

    VAR IPBuffer: ARRAY [0..16] OF CHAR;

    BEGIN
        IPToString (addr, IPBuffer);
        WriteString (IPBuffer);
    END WriteIPAddr;

(************************************************************************)

PROCEDURE DumpList (j: HostCategory);

    (* Writes the list to the screen. *)

    VAR current: HostRecordPtr;

    BEGIN
        WriteString ("Dumping ");  WriteString (INILabel[j]);
        WriteLn;
        current := MasterList[j];
        WHILE current <> NIL DO
            IF current^.numeric THEN
                WriteIPAddr (current^.numaddr);
            END (*IF*);
            IF current^.text THEN
                WriteString (current^.textaddr);
            END (*IF*);
            WriteLn;
            current := current^.next;
        END (*WHILE*);
    END DumpList;

(************************************************************************)
(*                       SEARCHING A HOSTLIST                           *)
(************************************************************************)

PROCEDURE OldNameMatch (VAR (*IN*) template, second: HostName): BOOLEAN;

    (* Returns TRUE if template and second are equal, with the extra    *)
    (* rules:                                                           *)
    (*   1. Character case is not significant.                          *)
    (*   2. A '.' as the initial character in template means that we    *)
    (*      will return TRUE if template matches the tail of second.    *)
    (* Special case: if template = "." then it matches anything.        *)

    VAR j1, j2: CARDINAL;

    BEGIN
        j1 := LENGTH (template);
        j2 := LENGTH (second);

        (* Check the "match anything" special case. *)

        IF (j1 = 0) OR ((j1 = 1) AND (template[0] = '.')) THEN
            RETURN TRUE;
        END (*IF*);

        (* Now the general case.  We check from the tail of the strings *)
        (* so as to pick up tail matches more easily.                   *)

        LOOP
            IF j1 = 0 THEN
                RETURN j2 = 0;
            ELSIF j2 = 0 THEN
                RETURN (j1 = 1) AND (template[0] = '.');
            ELSE
                DEC (j1);  DEC (j2);
                IF (j1 = 0) AND (template[0] = '.') THEN
                    RETURN second[j2] = '.';
                ELSIF CAP(template[j1]) <> CAP(second[j2]) THEN
                    RETURN FALSE;
                END (*IF*);
            END (*IF*);
        END (*LOOP*);

    END OldNameMatch;

(************************************************************************)

PROCEDURE NameMatch (VAR (*IN*) input, template: HostName): BOOLEAN;

    (* Returns TRUE if input matches template under either the old      *)
    (* rules (initial '.' is wild) or the new rules ('*' is wild).      *)

    BEGIN
        RETURN OldNameMatch(template, input) OR WildMatch(input, template);
    END NameMatch;

(************************************************************************)

PROCEDURE MatchByName (L: HostList;  VAR (*IN*) host: HostName): BOOLEAN;

    (* Returns TRUE iff host is in list L. *)

    VAR found: BOOLEAN;

    BEGIN
        found := FALSE;
        WHILE (L <> NIL) AND NOT found DO
            IF L^.text THEN
                found := NameMatch (host, L^.textaddr);
            END (*IF*);
            IF NOT found THEN
                L := L^.next;
            END (*IF*);
        END (*WHILE*);
        RETURN found;
    END MatchByName;

(************************************************************************)

PROCEDURE MatchNumeric (L: HostList;  IPAddress: CARDINAL): BOOLEAN;

    (* Returns TRUE if the address IPAddress is in list L. *)
    (* IPAddress is in network byte order.                 *)

    VAR found: BOOLEAN;  temp: CARDINAL;

    BEGIN
        found := FALSE;
        WHILE (L <> NIL) AND NOT found DO
            IF L^.numeric THEN
                IF L^.range THEN
                    temp := Swap4 (IPAddress);
                    found := (temp >= Swap4(L^.numaddr))
                             AND (temp <= Swap4(L^.numaddr2))
                ELSE
                    found := L^.numaddr = IPAddress;
                END (*IF*);
            END (*IF*);
            IF NOT found THEN
                L := L^.next;
            END (*IF*);
        END (*WHILE*);
        RETURN found;
    END MatchNumeric;

(************************************************************************)

PROCEDURE MatchEither (L: HostList;  IPAddress: CARDINAL;
                                      VAR (*IN*) host: HostName): BOOLEAN;

    (* Returns TRUE if either the address IPAddress or the name host    *)
    (* is in list L.  IPAddress is in network byte order.               *)

    VAR found: BOOLEAN;  temp: CARDINAL;

    BEGIN
        found := FALSE;
        WHILE (L <> NIL) AND NOT found DO
            IF L^.numeric THEN
                IF L^.range THEN
                    temp := Swap4 (IPAddress);
                    found := (temp >= Swap4(L^.numaddr))
                             AND (temp <= Swap4(L^.numaddr2))
                ELSE
                    found := L^.numaddr = IPAddress;
                END (*IF*);
            END (*IF*);
            IF NOT found AND L^.text THEN
                found := NameMatch (host, L^.textaddr);
            END (*IF*);
            IF NOT found THEN
                L := L^.next;
            END (*IF*);
        END (*WHILE*);
        RETURN found;
    END MatchEither;

(************************************************************************)
(*                 THE EXTERNALLY CALLABLE PROCEDURES                   *)
(************************************************************************)

PROCEDURE CheckHost (IPAddress: CARDINAL;
                         VAR (*OUT*) IsBanned, MayRelay: BOOLEAN);

    (* Looks up our internal list of hosts, and returns results: *)
    (*     IsBanned:   this host is on our blacklist, we will    *)
    (*                 refuse any connection from it.            *)
    (*     MayRelay:   this host is one from whom we will        *)
    (*                 accept mail to be relayed.                *)

    VAR name: HostName;
        NameDetails: HostEntPtr;

    BEGIN
        NameDetails := gethostbyaddr (IPAddress, SIZE(CARDINAL), AF_INET);
        IF tcp_h_errno() = 0 THEN
            Strings.Assign (NameDetails^.h_name^, name);
        ELSE
            name := "???";
        END (*IF*);
        IsBanned := MatchEither (MasterList[banned], IPAddress, name);
        MayRelay := MatchEither (MasterList[mayrelay], IPAddress, name);
    END CheckHost;

(************************************************************************)

PROCEDURE MatchHostName (category: HostCategory;
                          VAR (*IN*) name: HostName): BOOLEAN;

    (* Returns TRUE if name matches a name in 'category'. *)

    BEGIN
        IF NameIsNumeric (name) THEN
            RETURN MatchNumeric (MasterList[category], inet_addr(name))
        ELSE RETURN MatchByName (MasterList[category], name);
        END (*IF*);
    END MatchHostName;

(************************************************************************)

PROCEDURE AcceptableRelayDestination (VAR (*IN*) name: HostName): BOOLEAN;

    (* Returns TRUE if name matches a name in the "acceptable relay     *)
    (* destinations" list.                                              *)

    BEGIN
        RETURN MatchHostName (relaydest, name);
    END AcceptableRelayDestination;

(************************************************************************)

PROCEDURE HostIsLocal (VAR (*IN*) name: HostName): BOOLEAN;

    (* Returns TRUE if name matches a name of the local host. *)
    (* An empty name is acceptable.                           *)

    VAR result: BOOLEAN;

    BEGIN
        IF name[0] = Nul THEN result := TRUE
        ELSE
            Obtain (LocalListLock);
            result := MatchHostName (local, name);
            Release (LocalListLock);
        END (*IF*);
        RETURN result;
    END HostIsLocal;

(************************************************************************)

PROCEDURE OurHostName (VAR (*OUT*) name: HostName);

    (* Returns the name of the host on which this program is running. *)
    (* Because of the way we construct the lists, the first entry     *)
    (* of MasterList[local] is guaranteed to have a 'textaddr'        *)
    (* component, and that's what we use as our official name.        *)

    BEGIN
        Obtain (LocalListLock);
        name := MasterList[local]^.textaddr;
        Release (LocalListLock);
    END OurHostName;

(************************************************************************)

PROCEDURE AppendOurHostName (VAR (*INOUT*) string: ARRAY OF CHAR);

    (* Like OurHostName, but appends the result to the original string. *)

    BEGIN
        Obtain (LocalListLock);
        Strings.Append (MasterList[local]^.textaddr, string);
        Release (LocalListLock);
    END AppendOurHostName;

(************************************************************************)
(*                            INITIALISATION                            *)
(************************************************************************)

PROCEDURE Duplicate (VAR (*INOUT*) p: HostList): HostList;

    (* Inserts a clone of p^ after it on the list, returns a pointer to *)
    (* the duplicate.                                                   *)

    VAR q: HostList;

    BEGIN
        NEW (q);
        q^ := p^;
        p^.next := q;
        RETURN q;
    END Duplicate;

(************************************************************************)

PROCEDURE Wild (name: HostName): BOOLEAN;

    (* Returns TRUE iff name contains wildcard characters. *)

    VAR j: CARDINAL;

    BEGIN
        IF name[0] = '.' THEN RETURN TRUE END(*IF*);
        j := 0;
        LOOP
            IF name[j] = Nul THEN RETURN FALSE
            ELSIF name[j] = '*' THEN RETURN TRUE
            ELSIF name[j] = '?' THEN RETURN TRUE
            ELSIF j = MAX(HostNameIndex) THEN RETURN FALSE
            ELSE INC (j);
            END (*IF*);
        END (*LOOP*);
    END Wild;

(************************************************************************)

PROCEDURE ExpandAliases (hostdata: HostList;  IsLocal: BOOLEAN;
                                      ConfirmToScreen: BOOLEAN): HostList;

    (* Starting from the existing data in hostdata^, fills in the       *)
    (* numaddr/textaddr fields if they are missing and if we can get    *)
    (* the information from the nameserver.  We might also expand the   *)
    (* single record to a list of records if the named host is found to *)
    (* have multiple IP addresses or multiple names.                    *)
    (* Returns a pointer to the last node on this sublist.              *)

    VAR HostInfo: HostEntPtr;  newname: HostName;  j, k: CARDINAL;
        found: BOOLEAN;
        current: HostList;

    BEGIN
        HostInfo := NIL;
        current := hostdata;
        IF hostdata^.text THEN
            IF NOT Wild(hostdata^.textaddr) THEN
                IF ConfirmToScreen THEN
                    WriteString ("Checking ");
                    WriteString (hostdata^.textaddr);
                    WriteLn;
                END (*IF*);
                HostInfo := gethostbyname (hostdata^.textaddr);
            END (*IF*);
        END (*IF*);

        IF (HostInfo = NIL) AND hostdata^.numeric AND NOT hostdata^.range THEN
            IF ConfirmToScreen THEN
                WriteString ("Checking ");
                WriteIPAddr (hostdata^.numaddr);
                WriteLn;
            END (*IF*);
            HostInfo := gethostbyaddr (hostdata^.numaddr, SIZE(CARDINAL), AF_INET);
        END (*IF*);

        (* We've got as much data as we're going to get from the        *)
        (* nameserver.  Now decode it.                                  *)

        IF HostInfo <> NIL THEN

            (* Official name *)

            IF HostInfo^.h_name <> NIL THEN
                Strings.Assign (HostInfo^.h_name^, newname);
                IF NOT current^.text THEN
                    current^.textaddr := newname;
                    current^.text := TRUE;
                ELSIF NOT NameMatch(newname, current^.textaddr) THEN
                    current := Duplicate (current);
                    current^.textaddr := newname;
                END (*IF*);

                IF IsLocal THEN

                    (* Allow abbreviated form for a local name. *)

                    Strings.FindNext ('.', current^.textaddr, 0, found, k);
                    IF found THEN
                        current := Duplicate (current);
                        current^.textaddr[k] := Nul;
                    END (*IF*);

                END (*IF*);
            END (*IF*);

            (* Aliases *)

            IF HostInfo^.h_aliases <> NIL THEN
                j := 0;
                WHILE HostInfo^.h_aliases^[j] <> NIL DO
                    Strings.Assign (HostInfo^.h_aliases^[j]^, newname);
                    IF NOT current^.text THEN
                        current^.textaddr := newname;
                        current^.text := TRUE;
                    ELSIF NOT NameMatch(newname, current^.textaddr) THEN
                        current := Duplicate (current);
                        current^.textaddr := newname;
                    END (*IF*);

                    IF IsLocal THEN

                        (* Allow abbreviated form for a local name. *)

                        Strings.FindNext ('.', current^.textaddr, 0, found, k);
                        IF found THEN
                            current := Duplicate (current);
                            current^.textaddr[k] := Nul;
                        END (*IF*);

                    END (*IF*);

                    INC (j);

                END (*IF*);
            END (*IF*);

            (* IP addresses *)

            IF HostInfo^.h_addr_list <> NIL THEN
                j := 0;
                WHILE HostInfo^.h_addr_list^[j] <> NIL DO
                    k := HostInfo^.h_addr_list^[j]^;
                    IF NOT current^.numeric THEN
                        current^.numaddr := k;
                        current^.numeric := TRUE;
                        current^.range := FALSE;
                    ELSIF current^.range OR (current^.numaddr <> k) THEN
                        current := Duplicate (current);
                        current^.numaddr := k;
                        current^.range := FALSE;
                    END (*IF*);
                    INC (j);
                END (*IF*);
            END (*IF*);
        END (*IF*);
        RETURN current;
    END ExpandAliases;

(************************************************************************)

PROCEDURE GetOurHostName (ConfirmToScreen: BOOLEAN): HostRecordPtr;

    (* Creates a host record with as official an answer as possible for *)
    (* the IP address and name of the host on which we're running.      *)
    (* Both the numeric and text parts of the record are filled out,    *)
    (* even if we have to fudge it.                                     *)

    VAR result: HostRecordPtr;  HostInfo: HostEntPtr;

    BEGIN
        NEW (result);
        WITH result^ DO
            next := NIL;
            numeric := FALSE;  range := FALSE;  text := FALSE;
        END (*WITH*);

        (* We use gethostname in preference to gethostid, because it    *)
        (* seems that gethostid can return a left-over number that's    *)
        (* not necessarily still valid.                                 *)

        HostInfo := NIL;
        IF NOT gethostname(result^.textaddr,MAX(HostNameIndex)+1) THEN
            result^.text := TRUE;
            IF ConfirmToScreen THEN
                WriteString ("Checking local name ");
                WriteString (result^.textaddr);
                WriteLn;
            END (*IF*);
            HostInfo := gethostbyname (result^.textaddr);
        END (*IF*);

        IF (HostInfo = NIL) THEN
            result^.numaddr := Swap4(gethostid());
            result^.numeric := TRUE;
            IF ConfirmToScreen THEN
                WriteString ("Checking local IP address ");
                WriteIPAddr (result^.numaddr);
                WriteLn;
            END (*IF*);
            HostInfo := gethostbyaddr (result^.numaddr, SIZE(CARDINAL), AF_INET);
        END (*IF*);

        (* We've got as much data as we're going to get from the        *)
        (* nameserver.  Take the "most official" part of it.            *)

        IF HostInfo <> NIL THEN

            (* Official name *)

            IF HostInfo^.h_name <> NIL THEN
                Strings.Assign (HostInfo^.h_name^, result^.textaddr);
                result^.text := TRUE;
            END (*IF*);

            (* First IP address *)

            IF (HostInfo^.h_addr_list <> NIL)
                        AND (HostInfo^.h_addr_list^[0] <> NIL) THEN
                result^.numaddr := HostInfo^.h_addr_list^[0]^;
                result^.numeric := TRUE;
            END (*IF*);
        END (*IF*);

        (* Make sure that both the textual part and the numeric         *)
        (* part of the result are filled in.                            *)

        IF NOT result^.numeric THEN
            result^.numaddr := 0;
            result^.numeric := TRUE;
        END (*IF*);
        IF NOT result^.text THEN
            IPToString (result^.numaddr, result^.textaddr);
            result^.text := TRUE;
        END (*IF*);

        RETURN result;

    END GetOurHostName;

(************************************************************************)

PROCEDURE DecodeRange (Name: HostName;
                        VAR (*OUT*) First, Second: ARRAY OF LOC): BOOLEAN;

    (* Returns TRUE iff Name is a numeric IP address or a range of      *)
    (* the form [a.b.c.d-e], in which case First and Second are set to  *)
    (* the lower and upper limits of the range, in host byte order.     *)
    (* (For a single address, First and Second get the same value.)     *)

    VAR j: CARDINAL;

    (********************************************************************)

    PROCEDURE ConvertByte (VAR (*OUT*) val: LOC): BOOLEAN;

        (* Converts decimal string starting at Name[j] to numeric in    *)
        (* val, updates j.  Returns FALSE if number not found.          *)

        TYPE CharSet = SET OF CHAR;
        CONST Digits = CharSet {'0'..'9'};
        VAR result: CARD8;

        BEGIN
            IF NOT (Name[j] IN Digits) THEN
                RETURN FALSE;
            END (*IF*);
            result := 0;
            REPEAT
                result := 10*result + (ORD(Name[j]) - ORD('0'));
                INC (j);
            UNTIL (j > MAX(HostNameIndex)) OR NOT (Name[j] IN Digits);
            val := CAST(LOC,result);
            RETURN TRUE;
        EXCEPT
            RETURN FALSE;
        END ConvertByte;

    (********************************************************************)

    VAR k: CARDINAL;  val: CARD8;  success: BOOLEAN;

    BEGIN
        j := 0;
        IF Name[0] = '[' THEN j := 1 END (*IF*);

        (* Convert a sequence of decimal numbers separated by '.' *)

        success := ConvertByte(First[0]);
        Second[0] := First[0];
        k := 1;
        WHILE success AND (k <= 3) AND (Name[j] = '.') DO
            INC (j);
            success := ConvertByte(First[k]);
            Second[k] := First[k];  INC (k);
        END (*WHILE*);

        (* If we have a '-' or '/', get one more number. *)

        IF success AND (k <= 4) AND ((Name[j] = '-') OR (Name[j] = '/')) THEN
            INC (j);
            success := ConvertByte(Second[k-1]);
        END (*IF*);

        (* Pad out remaining bytes of the result. *)

        IF success THEN
            WHILE k <= 3 DO
                val := 0;  First[k] := CAST(LOC,val);
                val := MAX(CARD8);  Second[k] := CAST(LOC,val);
                INC(k);
            END (*WHILE*);
        END (*IF*);

        (* Check that the string is correctly terminated. *)

        IF Name[0] = '[' THEN
            success := success AND (Name[j] = ']');
            INC (j);
        END (*IF*);
        success := success AND ((j >= MAX(HostNameIndex)) OR (Name[j] = CHR(0)));

        RETURN success;

    END DecodeRange;

(************************************************************************)

PROCEDURE LoadList (j: HostCategory;  LogToScreen: BOOLEAN);

    (* Loads the list from the INI file. *)

    VAR hini: OS2.HINI;
        BufferSize, BufferPosition: CARDINAL;
        bufptr: POINTER TO ARRAY [0..MAX(CARDINAL) DIV 4] OF CHAR;
        ListTail, newtail: HostRecordPtr;

    (********************************************************************)

    PROCEDURE ProcessOneName;

        VAR ThisName: HostName;  k: CARDINAL;
            hostdata: HostRecordPtr;

        BEGIN
            (* Get the next substring from bufptr^. *)

            k := 0;
            REPEAT
                ThisName[k] := bufptr^[BufferPosition];
                INC (k);  INC(BufferPosition);
            UNTIL ThisName[k-1] = Nul;

            (* If the new name is not empty, add it to the master list. *)

            IF ThisName[0] <> Nul THEN
                NEW (hostdata);
                WITH hostdata^ DO
                     next := NIL;
                     IF DecodeRange (ThisName, numaddr, numaddr2) THEN
                         numeric := TRUE;
                         range := numaddr2 <> numaddr;
                         text := FALSE;
                     ELSE
                         numeric := FALSE;
                         text := TRUE;
                         numaddr := 0;
                         textaddr := ThisName;
                     END (*IF*);
                END (*WITH*);
                newtail := ExpandAliases (hostdata, j=local, LogToScreen);
                IF ListTail = NIL THEN
                    MasterList[j] := hostdata;
                ELSE
                    ListTail^.next := hostdata;
                END (*IF*);
                ListTail := newtail;
            END (*IF*);

        END ProcessOneName;

    (********************************************************************)

    BEGIN          (* Body of procedure LoadList *)

        (* For the "local" list, the first entry should be the official *)
        (* name of this host.  It's important to ensure that this comes *)
        (* at the head of the list.                                     *)

        IF j = local THEN

            ListTail := GetOurHostName (LogToScreen);
            newtail := ExpandAliases (ListTail, TRUE, LogToScreen);

        ELSE

            (* Not the 'local' list, so start with an empty list.      *)

            ListTail := NIL;  newtail := NIL;

        END (*IF*);

        MasterList[j] := ListTail;  ListTail := newtail;

        (* The rest of the information comes from the INI file. *)

        bufptr := NIL;
        BufferSize := 0;
        BufferPosition := 0;

        (* Get the string of host names into a buffer. *)

        hini := OpenINIFile ("Weasel.INI");
        IF (hini <> OS2.NULLHANDLE) THEN
            IF OS2.PrfQueryProfileSize (hini, "$SYS", INILabel[j], BufferSize)
                    AND (BufferSize <> 0) THEN
                ALLOCATE (bufptr, BufferSize);
                OS2.PrfQueryProfileData (hini, "$SYS", INILabel[j], bufptr, BufferSize);
                OS2.PrfCloseProfile (hini);
            END (*IF*);
        END (*IF*);

        (* Process the buffer.  Each substring is terminated with a Nul. *)

        IF bufptr <> NIL THEN
            WHILE BufferPosition < BufferSize DO
                ProcessOneName;
            END (*WHILE*);
            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);

    END LoadList;

(************************************************************************)

PROCEDURE CompactifyList (j: HostCategory);

    (* Removes duplicates from the list, and packs it more neatly. *)

    VAR p, previous, current: HostRecordPtr;

    BEGIN
        p := MasterList[j];
        WHILE p <> NIL DO
            previous := p;  current := p^.next;
            WHILE current <> NIL DO

                (* Check the numeric part of the records. *)

                IF p^.numeric THEN

                    (* Duplicate number? *)

                    IF (NOT current^.range) AND current^.numeric
                                 AND (current^.numaddr = p^.numaddr) THEN
                        current^.numeric := FALSE;
                    END (*IF*);

                ELSIF current^.numeric THEN

                    (* Repacking possible. *)

                    p^.numeric := TRUE;  p^.numaddr := current^.numaddr;
                    current^.numeric := FALSE;

                END (*IF*);

                (* Check the textual part of the records. *)

                IF p^.text THEN

                    (* Redundant name in current^? *)

                    IF current^.text AND NameMatch(current^.textaddr, p^.textaddr) THEN
                        current^.text := FALSE;
                    END (*IF*);

                ELSIF current^.text THEN

                    (* Repacking possible. *)

                    p^.text := TRUE;  p^.textaddr := current^.textaddr;
                    current^.text := FALSE;

                END (*IF*);

                (* If we've now removed everything from current, get    *)
                (* rid of the record.                                   *)

                IF NOT (current^.numeric OR current^.text) THEN
                    previous^.next := current^.next;
                    DISPOSE (current);
                ELSE
                    previous := current;
                END (*IF*);
                current := previous^.next;
            END (*WHILE*);
            p := p^.next;
        END (*WHILE*)
    END CompactifyList;

(************************************************************************)

PROCEDURE DiscardList (j: HostCategory);

    (* Disposes of the list. *)

    VAR p, q: HostRecordPtr;

    BEGIN
        p := MasterList[j];  MasterList[j] := NIL;
        WHILE p <> NIL DO
            q := p;  p := p^.next;
            DISPOSE (q);
        END (*WHILE*);
    END DiscardList;

(************************************************************************)

PROCEDURE RecomputeOurHostName (VAR (*OUT*) IPAddress: CARDINAL;
                                VAR (*OUT*) name: HostName);

    (* This procedure is to be called when our host name or IP address  *)
    (* might have changed, e.g. because we've just come on-line.        *)
    (* Rebuilds the list of names that are considered to be names of    *)
    (* the local host, and returns the IP address and name at the head  *)
    (* of the rebuilt list.                                             *)

    BEGIN
        Obtain (LocalListLock);
        DiscardList (local);
        LoadList (local, FALSE);
        CompactifyList (local);
        Release (LocalListLock);
        IPAddress := MasterList[local]^.numaddr;
        name := MasterList[local]^.textaddr;
    END RecomputeOurHostName;

(************************************************************************)

VAR j: HostCategory;

BEGIN
    sock_init();
    CreateLock (LocalListLock);
    FOR j := MIN(HostCategory) TO MAX(HostCategory) DO
        LoadList (j, NotDetached());
    END (*FOR*);
    (*DumpList (local);*)
    FOR j := MIN(HostCategory) TO MAX(HostCategory) DO
        CompactifyList (j);
    END (*FOR*);
    (*DumpList (local);*)
END Hosts.

