MODULE LoadINI;

        (********************************************************)
        (*                                                      *)
        (*     Loads data into Weasel.INI from a text file      *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            16 July 1998                    *)
        (*  Last edited:        19 December 1998                *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*       This version is not loading the host lists     *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2, IOChan, ChanConsts, IOConsts, FileSys, SeqFile, Strings, STextIO;

FROM SYSTEM IMPORT LOC, ADR, MOVE;

FROM Storage IMPORT ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST
    Nul = CHR(0);  CtrlZ = CHR(26);
    ChunkSize = 4096;
    FilenameStringLength = 512;

TYPE
    CardArray = ARRAY [0..1] OF CARDINAL;
    NameString = ARRAY [0..31] OF CHAR;
    PassStringIndex = [0..31];
    PassString = ARRAY PassStringIndex OF CHAR;
    EmailAddress = ARRAY [0..511] OF CHAR;
    HostNameIndex = [0..255];
    HostName = ARRAY HostNameIndex OF CHAR;
    FilenameString = ARRAY [0..FilenameStringLength-1] OF CHAR;
    BufferIndex = [0..MAX(CARDINAL) DIV 4];
    BufferPointer = POINTER TO ARRAY BufferIndex OF CHAR;

VAR
    (* Anchor block handle for this application.  *)

    hab: OS2.HAB;

    (* INI file handle. *)

    hini: OS2.HINI;

    (* Output file handle. *)

    cid: IOChan.ChanId;

    (* Input line buffer. *)

    InputLine: ARRAY [0..1023] OF CHAR;
    LineNumber: CARDINAL;

(********************************************************************************)
(*                           UTILITY PROCEDURES                                 *)
(********************************************************************************)

PROCEDURE ToLower (VAR (*INOUT*) string: ARRAY OF CHAR);

    (* Converts all letters in string to lower case. *)

    TYPE CharSet = SET OF CHAR;

    CONST shift = ORD('a') - ORD('A');

    VAR j, length: CARDINAL;

    BEGIN
        length := Strings.Length(string);
        IF length > 0 THEN
            FOR j := 0 TO length-1 DO
                IF string[j] IN CharSet {'A'..'Z'} THEN
                    INC (string[j], shift);
                END (*IF*);
            END (*FOR*);
        END (*IF*);
    END ToLower;

(********************************************************************************)

PROCEDURE WriteCard (N: CARDINAL);

    (* Writes the value of N to standard output. *)

    BEGIN
        IF N > 9 THEN
            WriteCard (N DIV 10);  N := N MOD 10;
        END (*IF*);
        STextIO.WriteChar(CHR(ORD('0') + N));
    END WriteCard;

(********************************************************************************)
(*                           INPUT FROM TNI FILE                                *)
(********************************************************************************)

PROCEDURE ReadNextLine;

    (* Reads a line of input into InputLine.  At end of file we  *)
    (* set InputLine[0] = CtrlZ.                                 *)

    CONST CR = CHR(13);  LF = CHR(10);

    VAR j, amount: CARDINAL;  ch: CHAR;  status: IOConsts.ReadResults;

    BEGIN
        INC (LineNumber);  j := 0;
        LOOP
            IOChan.RawRead (cid, ADR(ch), 1, amount);
            status := IOChan.ReadResult(cid);
            IF (amount = 0) OR (status <> IOConsts.allRight)
                               OR (ch = CR) THEN
                EXIT (*LOOP*);
            END (*IF*);
            IF ch <> LF THEN
                InputLine[j] := ch;  INC(j);
            END (*IF*);
        END (*LOOP*);
        IF (j = 0) AND (status <> IOConsts.allRight) THEN
            InputLine[0] := CtrlZ;
        ELSE
            InputLine[j] := Nul;
        END (*IF*);
    END ReadNextLine;

(************************************************************************)

PROCEDURE OpenInputFile (VAR (*OUT*) cid: IOChan.ChanId;
                                  inname: ARRAY OF CHAR): BOOLEAN;

    (* Opens the file inname.TNI for input. *)

    VAR filename: FilenameString;
        result: ChanConsts.OpenResults;

    BEGIN
        LineNumber := 0;
        Strings.Assign (inname, filename);
        Strings.Append (".TNI", filename);
        SeqFile.OpenRead (cid, filename, ChanConsts.read+ChanConsts.raw, result);
        RETURN result = ChanConsts.opened;
    END OpenInputFile;

(********************************************************************************)
(*                          OUTPUT TO INI FILE                                  *)
(********************************************************************************)

PROCEDURE TranslateSYSparameters;

    (* Writes the miscellaneous INI data to the INI file. *)

    VAR bufpos: CARDINAL;

    (********************************************************************)

    PROCEDURE GetCard(): CARDINAL;

        TYPE CharSet = SET OF CHAR;

        VAR result: CARDINAL;

        BEGIN
            result := 0;
            WHILE InputLine[bufpos] = ' ' DO
                INC (bufpos);
            END (*WHILE*);
            WHILE InputLine[bufpos] IN CharSet{'0'..'9'} DO
                result := 10*result + ORD(InputLine[bufpos]) - ORD('0');
                INC (bufpos);
            END (*WHILE*);
            RETURN result;
        END GetCard;

    (****************************************************************)

    VAR port, MaxUsers, TimeoutLimit: CardArray;
        j: [0..1];
        TransactionLogLevel, ServerEnable: CARDINAL;

    BEGIN

        bufpos := 0;
        FOR j := 0 TO 1 DO
            port[j] := GetCard();
            MaxUsers[j] := GetCard();
            TimeoutLimit[j] := GetCard();
        END (*FOR*);
        TransactionLogLevel := GetCard();
        ServerEnable := GetCard();

        OS2.PrfWriteProfileData (hini, "$SYS", "ServerPort",
                                   ADR(port), SIZE(CardArray));
        OS2.PrfWriteProfileData (hini, "$SYS", "MaxUsers",
                                   ADR(MaxUsers), SIZE(CardArray));
        OS2.PrfWriteProfileData (hini, "$SYS", "TimeOut",
                                   ADR(TimeoutLimit), SIZE(CardArray));
        OS2.PrfWriteProfileData (hini, "$SYS", "TransLevel",
                                   ADR(TransactionLogLevel), SIZE(CARDINAL));
        OS2.PrfWriteProfileData (hini, "$SYS", "Enable",
                                   ADR(ServerEnable), SIZE(CARDINAL));

        ReadNextLine;
        OS2.PrfWriteProfileData (hini, "$SYS", "MailRoot", ADR(InputLine),
                                                    FilenameStringLength);
        ReadNextLine;
        IF (InputLine[0] <> '[') AND (InputLine[0] <> CtrlZ) THEN
            OS2.PrfWriteProfileData (hini, "$SYS", "ForwardRelay",
                                     ADR(InputLine), FilenameStringLength);
            ReadNextLine;
        END (*IF*);

    END TranslateSYSparameters;

(************************************************************************)

PROCEDURE ExtendBuffer (VAR (*INOUT*) bufptr: BufferPointer;
                                  VAR (*INOUT*) size: CARDINAL);

    (* Extends the size of bufptr^ by ChunkSize. *)

    VAR auxptr: BufferPointer;

    BEGIN
        ALLOCATE (auxptr, size+ChunkSize);
        MOVE (bufptr, auxptr, size);
        DEALLOCATE (bufptr, size);
        INC (size, ChunkSize);
        bufptr := auxptr;
    END ExtendBuffer;

(************************************************************************)

PROCEDURE TranslateUsers;

    (* Reads all user information from the TNI file, writes it to       *)
    (* the INI file.                                                    *)

    VAR password: PassString;
        j: CARDINAL;
        found: BOOLEAN;

    BEGIN
        WHILE (InputLine[0] <> '[') AND (InputLine[0] <> CtrlZ) DO
            Strings.Assign (InputLine, password);
            Strings.FindNext (' ', InputLine, 0, found, j);
            IF found THEN
                Strings.Delete (password, 0, j+1);
                InputLine[j] := Nul;
            ELSE
                password[0] := Nul;
            END (*IF*);
            ToLower (InputLine);
            OS2.PrfWriteProfileData (hini, InputLine, "Password",
                                ADR(password), MAX(PassStringIndex)+1);
            ReadNextLine;
        END (*WHILE*);
    END TranslateUsers;

(************************************************************************)

PROCEDURE TranslateOneAlias;

    (* Reads an alias list, writes it to the INI file. *)

    VAR name: NameString;
        Used, BufferSize: CARDINAL;
        bufptr: BufferPointer;
        j, length: CARDINAL;

    BEGIN
        Strings.Assign (InputLine, name);
        ALLOCATE (bufptr, ChunkSize);
        BufferSize := ChunkSize;
        IF name[0] = '+' THEN bufptr^[0] := CHR(1)
        ELSE bufptr^[0] := Nul
        END (*IF*);
        Strings.Delete (name, 0, 1);
        ToLower (name);
        Used := 1;
        ReadNextLine;
        WHILE InputLine[0] = ' ' DO
            Strings.Delete (InputLine, 0, 2);
            length := Strings.Length (InputLine);
            IF Used + length >= BufferSize THEN
                ExtendBuffer (bufptr, BufferSize);
            END (*IF*);
            IF length > 0 THEN
                FOR j := 0 TO length-1 DO
                    bufptr^[Used] := InputLine[j];  INC(Used);
                END (*FOR*);
            END (*IF*);
            bufptr^[Used] := Nul;  INC(Used);
            ReadNextLine;
        END (*WHILE*);
        IF Used >= BufferSize THEN
            ExtendBuffer (bufptr, BufferSize);
        END (*IF*);
        bufptr^[Used] := Nul;  INC(Used);
        OS2.PrfWriteProfileData (hini, "$ALIAS", name, bufptr, Used);
        DEALLOCATE (bufptr, BufferSize);
    END TranslateOneAlias;

(************************************************************************)

PROCEDURE TranslateAliases;

    (* Reads all alias definitions from the TNI file and writes them    *)
    (* to the INI file.                                                 *)

    BEGIN
        WHILE (InputLine[0] = '+') OR (InputLine[0] = '-') DO
            TranslateOneAlias;
        END (*WHILE*);
    END TranslateAliases;

(************************************************************************)
(*                              MAIN PROGRAM                            *)
(************************************************************************)

TYPE DataGroup = (nogroup, general, usernames, aliases);

(************************************************************************)

PROCEDURE Section(): DataGroup;

    (* Assumption: InputLine holds a section header. *)

    VAR result: DataGroup;

    BEGIN
        (* Skip past blank lines. *)

        LOOP
            WHILE InputLine[0] = ' ' DO
                Strings.Delete (InputLine, 0, 1);
            END (*WHILE*);
            IF InputLine[0] <> Nul THEN
                EXIT (*LOOP*);
            END (*IF*);
            ReadNextLine;
        END (*LOOP*);

        (* Now we should have a line starting with '['. *)

        IF InputLine[0] = '[' THEN
            CASE CAP(InputLine[1]) OF
              | 'G':  result := general;
              | 'U':  result := usernames;
              | 'A':  result := aliases;
            ELSE
                STextIO.WriteString ("Unknown section name at line ");
                WriteCard (LineNumber);  STextIO.WriteLn;
                result := nogroup;
            END (*IF*);
        ELSE
            IF InputLine[0] <> CtrlZ THEN
                STextIO.WriteString ("Missing '[' at line ");
                WriteCard (LineNumber);  STextIO.WriteLn;
            END (*IF*);
            result := nogroup;
        END (*IF*);

        (* Consume the line we've just checked. *)

        ReadNextLine;
        RETURN result;

    END Section;

(************************************************************************)

PROCEDURE RunTheProgram;

    BEGIN
        IF OpenInputFile(cid, "Weasel") THEN
            hini := OS2.PrfOpenProfile (hab, "weasel.ini");
            IF hini = OS2.NULLHANDLE THEN
                STextIO.WriteString ("Could not open WEASEL.INI");
                STextIO.WriteLn;
            ELSE
                ReadNextLine;
                LOOP
                    CASE Section() OF
                      | nogroup:     EXIT (*LOOP*);
                      | general:     TranslateSYSparameters;
                      | usernames:   TranslateUsers;
                      | aliases:     TranslateAliases;
                    END (*CASE*);
                END (*LOOP*);
                OS2.PrfCloseProfile (hini);
            END (*IF*);
            SeqFile.Close (cid);
        ELSE
            STextIO.WriteString ("Could not open input file");
            STextIO.WriteLn;
        END (*IF*);
    END RunTheProgram;

(************************************************************************)

BEGIN
    hab := OS2.WinInitialize (0);
    RunTheProgram;
END LoadINI.

