IMPLEMENTATION MODULE EditUsers;

        (********************************************************)
        (*                                                      *)
        (*               Weasel user editor                     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            24 April 1998                   *)
        (*  Last edited:        16 March 1999                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2, Strings, FileSys;

FROM MultiScreen IMPORT
    (* type *)  ScreenGroup, VirtualScreen,
    (* proc *)  CreateScreenGroup, CreateVirtualScreen, MapToVirtualScreen,
                VirtualScreenOf, RemoveVirtualScreen, RemoveScreenGroup;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  CreateListBox, CursorMovements, WindowOf,
                LBAppend, LBCurrent, HighlightOn, HighlightOff, LBDeleteCurrent,
                LBInsertAfter, ReplaceCurrent, LBSort, ItemNumberOf;

FROM InetUtilities IMPORT
    (* proc *)  ToLower;

FROM Users IMPORT
    (* proc *)  EditUserData, RemoveUser;

FROM AliasLists IMPORT
    (* proc *)  EditAliasData, RenameAlias, RemoveAlias;

FROM SetupINI IMPORT
    (* proc *)  OurINIHandle;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, SetCursor, PutOnTop,
                EditString, EditAborted, WriteString, WriteLn, GetKey,
                GetScreenSize, SetActivePage;

FROM Keyboard IMPORT
    (* proc *)  StuffKeyboardBuffer, PutBack;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST Nul = CHR(0);

TYPE
    NameStringIndex = [0..31];
    NameString = ARRAY NameStringIndex OF CHAR;
    CharArrayPointer = POINTER TO ARRAY [0..MAX(CARDINAL) DIV 4] OF CHAR;

VAR
    (* Number of display rows on screen.  *)

    ScreenRows: CARDINAL;

(************************************************************************)
(*             ENSURING THAT EVERY USER HAS A MAIL DIRECTORY            *)
(************************************************************************)

PROCEDURE CheckUserDirectories (VAR (*IN*) MailRoot: ARRAY OF CHAR);

    (* Makes sure that MailRoot has a subdirectory for every user in    *)
    (* the user database.  Also creates a couple of compulsory          *)
    (* directories, if they don't already exist.                        *)

    VAR hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferSize: CARDINAL;
        name: NameString;
        j, k: CARDINAL;
        DirName: ARRAY [0..511] OF CHAR;

    BEGIN
        (* Create the mailroot directory if it doesn't already exist.   *)
        (* Note that we temporarily have to remove the trailing '\'     *)
        (* from the directory name.                                     *)

        j := Strings.Length (MailRoot);
        IF j > 0 THEN
            MailRoot[j-1] := Nul;
            EVAL (FileSys.CreateDirectory(MailRoot));
            Strings.Append ('\', MailRoot);
        END (*IF*);

        (* Get the list of users from the INI file. *)

        hini := OurINIHandle();
        IF (hini <> OS2.NULLHANDLE)
                  AND OS2.PrfQueryProfileSize (hini, NIL, NIL, BufferSize)
                  AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, NIL, NIL, bufptr, BufferSize);
            j := 0;

            (* Each time around this loop we extract one user name. *)

            WHILE (j < BufferSize) AND (bufptr^[j] <> Nul) DO
                Strings.Assign (MailRoot, DirName);
                k := 0;
                REPEAT
                    name[k] := bufptr^[j];
                    INC (k);  INC (j);
                UNTIL (j >= BufferSize) OR (bufptr^[j-1] = Nul)
                                        OR (k > MAX(NameStringIndex));
                IF name[0] <> "$" THEN

                    (* User name found, try to create a directory.      *)
                    (* (If the directory already exists the operation   *)
                    (* will fail anyway.)                               *)

                    Strings.Append (name, DirName);
                    EVAL (FileSys.CreateDirectory(DirName));

                END (*IF*);
            END (*WHILE*);
            DEALLOCATE (bufptr, BufferSize);

        END (*IF*);

        (* All users checked.  Now make sure the "postmaster" and       *)
        (* "forward" directories exist.                                 *)

        Strings.Assign (MailRoot, DirName);
        Strings.Append ("postmaster", DirName);
        EVAL (FileSys.CreateDirectory(DirName));
        Strings.Assign (MailRoot, DirName);
        Strings.Append ("forward", DirName);
        EVAL (FileSys.CreateDirectory (DirName));

    END CheckUserDirectories;

(************************************************************************)
(*                  DISPLAYING THE LIST OF USERS                        *)
(************************************************************************)

PROCEDURE LoadNames (bufptr: CharArrayPointer;  BufferSize: CARDINAL;
                     LB: ListBox);

    (* Loads a sequence of strings from bufptr^ to LB. *)

    VAR j, k: CARDINAL;
        name: NameString;

    BEGIN
        j := 0;

        (* Each time around this loop we extract one name. *)

        WHILE (j < BufferSize) AND (bufptr^[j] <> Nul) DO
            k := 0;
            REPEAT
                name[k] := bufptr^[j];
                INC (k);  INC (j);
            UNTIL (j >= BufferSize) OR (bufptr^[j-1] = Nul)
                                    OR (k > MAX(NameStringIndex));
            IF name[0] <> "$" THEN
                LBAppend (LB, name);
            END (*IF*);
        END (*WHILE*);
    END LoadNames;

(************************************************************************)

PROCEDURE FillUserList (LB: ListBox);

    (* Adds all known users to the listbox. *)

    VAR hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferSize: CARDINAL;

    BEGIN
        (* Get the list of users from the INI file. *)

        hini := OurINIHandle();
        IF (hini <> OS2.NULLHANDLE)
                  AND OS2.PrfQueryProfileSize (hini, NIL, NIL, BufferSize)
                  AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, NIL, NIL, bufptr, BufferSize);
            LoadNames (bufptr, BufferSize, LB);
            DEALLOCATE (bufptr, BufferSize);
            LBSort (LB);

        END (*IF*);
    END FillUserList;

(************************************************************************)

PROCEDURE FillAliasList (LB: ListBox);

    (* Adds all known aliases to the listbox. *)

    VAR hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferSize: CARDINAL;

    BEGIN
        (* Get the list of aliases from the INI file. *)

        hini := OurINIHandle();
        IF (hini <> OS2.NULLHANDLE)
                  AND OS2.PrfQueryProfileSize (hini, "$ALIAS", NIL, BufferSize)
                  AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, "$ALIAS", NIL, bufptr, BufferSize);
            LoadNames (bufptr, BufferSize, LB);
            DEALLOCATE (bufptr, BufferSize);
            LBSort (LB);
        END (*IF*);

    END FillAliasList;

(************************************************************************)
(*                           THE USER EDITOR                            *)
(************************************************************************)

PROCEDURE UserEditor;

    (* Runs as an autonomous task. *)

    CONST Esc = CHR(1BH);
        BoxTop = 2;  BoxLeft = 2;  BoxWidth = 35;

    VAR ListWindow, TopBar, BottomBar: Window;  UserList: ListBox;
        ch: CHAR;
        UserName: ARRAY [0..BoxWidth-1] OF CHAR;
        BoxHeight: CARDINAL;
        OurGroup: ScreenGroup;
        OurPage: VirtualScreen;

    BEGIN
        BoxHeight := ScreenRows - BoxTop - 4;

        OurGroup := CreateScreenGroup (1);
        OurPage := CreateVirtualScreen (OurGroup);

        OpenWindowHidden (BottomBar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (BottomBar, OurPage);
        WriteString (BottomBar, " A add  E edit  Del delete  X exit");
        SetCursor (BottomBar, 0, 55);
        WriteString (BottomBar, "F4,F5 previous/next page");

        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, OurPage);
        WriteString (TopBar, "    WEASEL USER EDITOR");

        OpenWindowHidden (ListWindow, black, white, BoxTop, BoxTop+BoxHeight+1,
                      BoxLeft, BoxLeft+BoxWidth+2, noframe, nodivider);
        MapToVirtualScreen (ListWindow, OurPage);
        UserList := CreateListBox (ListWindow, 1, 1, BoxHeight, BoxWidth);
        FillUserList (UserList);
        HighlightOn (UserList);

        LOOP
            EVAL (CursorMovements (UserList));
            ch := GetKey(ListWindow);
            IF ch = Nul THEN
                ch := GetKey(ListWindow);
                IF ch = 'S' THEN                      (* Del = delete *)
                    LBCurrent (UserList, UserName);
                    RemoveUser (UserName);
                    LBDeleteCurrent (UserList);
                END (*IF*);
            ELSIF CAP(ch) = 'A' THEN                       (* A = add *)
                UserName := "";
                EditUserData (OurPage, UserName);
                LBInsertAfter (UserList, UserName);
            ELSIF (CAP(ch) = 'E') OR (ch = CHR(13)) THEN   (* E = edit *)
                LBCurrent (UserList, UserName);
                EditUserData (OurPage, UserName);
                ReplaceCurrent (UserList, UserName);
            ELSIF CAP(ch) = 'X' THEN                       (* X = exit *)
                SetActivePage (0);
                StuffKeyboardBuffer (CHR(27));
            END (*IF*);
        END (*LOOP*);

    END UserEditor;

(************************************************************************)
(*                          THE ALIAS EDITOR                            *)
(************************************************************************)

PROCEDURE EditAliasList (LB: ListBox;  BoxWidth: CARDINAL);

    (* Allows delete/add/edit operations on LB. *)

    TYPE NameIndex = [0..255];

    VAR w: Window;  OurPage: VirtualScreen;  ch: CHAR;
        OldName, AliasName: ARRAY NameIndex OF CHAR;

    (********************************************************************)

    PROCEDURE EditNewName;

        (* Edits the variable 'AliasName', checking for duplicate names.*)
        (* An empty string is an acceptable result.                     *)

        VAR err: Window;

        BEGIN
            LOOP
                HighlightOff(LB);
                EditString (WindowOf(LB), AliasName, MAX(NameIndex)+1, BoxWidth);
                HighlightOn(LB);
                ToLower (AliasName);
                IF EditAborted() THEN
                    AliasName := OldName;
                END (*IF*);
                IF Strings.Equal(AliasName, OldName)
                         OR (ItemNumberOf(LB, AliasName) = 0) THEN
                    EXIT (*LOOP*);
                END (*IF*);
                OpenWindowHidden (err, yellow, red, 5, 9, 40, 69, noframe, nodivider);
                MapToVirtualScreen (err, OurPage);
                WriteLn (err);
                WriteString (err, "      Duplicate name");
                WriteLn (err);
                WriteString (err, "   Type <Enter> to re-edit");
                WriteLn (err);
                WriteString (err, "      or Esc to abort");
                REPEAT
                    ch := GetKey (err);
                UNTIL (ch = CHR(27)) OR (ch = CHR(13));
                CloseWindow (err);
                IF ch = CHR(27) THEN
                    AliasName := OldName;
                    PutBack(ch);
                END (*IF*);
            END (*LOOP*);
        END EditNewName;

    (********************************************************************)

    BEGIN
        w := WindowOf(LB);
        OurPage := VirtualScreenOf(w);
        HighlightOn (LB);
        LOOP
            EVAL (CursorMovements (LB));
            ch := GetKey(w);
            IF ch = Nul THEN
                ch := GetKey(w);
                IF ch = 'S' THEN                      (* Del = delete *)
                    LBCurrent (LB, AliasName);
                    RemoveAlias (AliasName);
                    LBDeleteCurrent (LB);
                END (*IF*);
            ELSIF CAP(ch) = 'A' THEN                       (* A = add *)
                OldName := "";
                LBInsertAfter (LB, OldName);
                AliasName := OldName;
                EditNewName;
                IF AliasName[0] = Nul THEN
                    LBDeleteCurrent (LB);
                ELSE
                    ReplaceCurrent (LB, AliasName);
                    EditAliasData (OurPage, AliasName);
                END (*IF*);
            ELSIF (CAP(ch) = 'E') OR (ch = CHR(13)) THEN   (* E = edit *)
                LBCurrent (LB, AliasName);
                IF AliasName[0] <> Nul THEN
                    EditAliasData (OurPage, AliasName);
                    ReplaceCurrent (LB, AliasName);
                END (*IF*);
            ELSIF CAP(ch) = 'R' THEN                       (* R = rename *)
                LBCurrent (LB, OldName);
                IF OldName[0] <> Nul THEN
                    AliasName := OldName;
                    EditNewName;
                    IF NOT Strings.Equal(AliasName, OldName) THEN
                        ReplaceCurrent (LB, AliasName);
                        RenameAlias (OldName, AliasName);
                    END (*IF*);
                END (*IF*);
            ELSIF CAP(ch) = 'X' THEN                       (* X = exit *)
                SetActivePage (0);
                StuffKeyboardBuffer (CHR(27));
            END (*IF*);
        END (*LOOP*);
    END EditAliasList;

(************************************************************************)

PROCEDURE AliasEditor;

    (* Runs as an autonomous task. *)

    CONST Esc = CHR(1BH);
        BoxTop = 2;  BoxLeft = 10;  BoxWidth = 35;

    VAR ListWindow, TopBar, BottomBar: Window;  AliasList: ListBox;
        BoxHeight: CARDINAL;
        OurGroup: ScreenGroup;
        OurPage: VirtualScreen;

    BEGIN
        CreateTask (UserEditor, 2, "User editor");
        BoxHeight := ScreenRows - BoxTop - 4;

        OurGroup := CreateScreenGroup (1);
        OurPage := CreateVirtualScreen (OurGroup);

        OpenWindowHidden (BottomBar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (BottomBar, OurPage);
        WriteString (BottomBar, " A add  E edit  R rename  Del delete  X exit");
        SetCursor (BottomBar, 0, 55);
        WriteString (BottomBar, "F4,F5 previous/next page");

        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, OurPage);
        WriteString (TopBar, "    ALIASES");

        OpenWindowHidden (ListWindow, black, white, BoxTop, BoxTop+BoxHeight+1,
                      BoxLeft, BoxLeft+BoxWidth+1, noframe, nodivider);
        MapToVirtualScreen (ListWindow, OurPage);
        AliasList := CreateListBox (ListWindow, 1, 1, BoxHeight, BoxWidth);
        FillAliasList (AliasList);
        EditAliasList (AliasList, BoxWidth);

    END AliasEditor;

(************************************************************************)
(*                         INITIALISATION                               *)
(************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    GetScreenSize (ScreenRows, dummy);
    CreateTask (AliasEditor, 2, "Alias editor");
END EditUsers.

